import Gio from 'gi://Gio';
import GLib from 'gi://GLib';

export const KBD_IFACE_XML = `<node>
  <interface name="org.gnome.SettingsDaemon.Power.Keyboard">
    <property name="Brightness" type="i" access="readwrite"/>
    <property name="Steps" type="i" access="read"/>
    <method name="StepUp">
      <arg type="i" name="new_percentage" direction="out"/>
    </method>
    <method name="StepDown">
      <arg type="i" name="new_percentage" direction="out"/>
    </method>
    <method name="Toggle">
      <arg type="i" name="new_percentage" direction="out"/>
    </method>
    <signal name="BrightnessChanged">
      <arg name="brightness" type="i"/>
      <arg name="source" type="s"/>
    </signal>
  </interface>
</node>`;

export const IDLE_IFACE_XML = `<node>
  <interface name="org.gnome.Mutter.IdleMonitor">
    <method name="AddIdleWatch">
      <arg type="t" name="interval" direction="in"/>
      <arg type="u" name="id" direction="out"/>
    </method>
    <method name="AddUserActiveWatch">
      <arg type="u" name="id" direction="out"/>
    </method>
    <method name="RemoveWatch">
      <arg type="u" name="id" direction="in"/>
    </method>
    <signal name="WatchFired">
      <arg name="id" type="u"/>
    </signal>
  </interface>
</node>`;

export const PROPS_IFACE_XML = `<node>
  <interface name="org.freedesktop.DBus.Properties">
    <method name="Get">
      <arg type="s" name="interface_name" direction="in"/>
      <arg type="s" name="property_name" direction="in"/>
      <arg type="v" name="value" direction="out"/>
    </method>
    <method name="Set">
      <arg type="s" name="interface_name" direction="in"/>
      <arg type="s" name="property_name" direction="in"/>
      <arg type="v" name="value" direction="in"/>
    </method>
    <method name="GetAll">
      <arg type="s" name="interface_name" direction="in"/>
      <arg type="a{sv}" name="properties" direction="out"/>
    </method>
    <signal name="PropertiesChanged">
      <arg type="s" name="interface_name"/>
      <arg type="a{sv}" name="changed_properties"/>
      <arg type="as" name="invalidated_properties"/>
    </signal>
  </interface>
</node>`;

export const KbdProxy = Gio.DBusProxy.makeProxyWrapper(KBD_IFACE_XML);
export const IdleMonitorProxy = Gio.DBusProxy.makeProxyWrapper(IDLE_IFACE_XML);
export const PropsProxy = Gio.DBusProxy.makeProxyWrapper(PROPS_IFACE_XML);

const METHOD_CACHE = new WeakMap();

function coerceArg(signature, value) {
  if (signature === 't') {
    if (typeof value === 'number')
      return BigInt(Math.max(0, Math.floor(value)));
    return value;
  }

  if (signature === 'u') {
    const n = Number(value);
    if (Number.isNaN(n))
      return value;
    return Math.max(0, Math.min(0xFFFFFFFF, Math.floor(n)));
  }

  if (signature === 'q') {
    const n = Number(value);
    if (Number.isNaN(n))
      return value;
    return Math.max(0, Math.min(0xFFFF, Math.floor(n)));
  }

  return value;
}

function getMethodMeta(proxy, method) {
  let cache = METHOD_CACHE.get(proxy);
  if (!cache) {
    cache = new Map();
    METHOD_CACHE.set(proxy, cache);
  }

  if (cache.has(method))
    return cache.get(method);

  const methodInfo = proxy.g_interface_info?.lookup_method(method);
  if (!methodInfo)
    throw new Error(`Missing method in introspection: ${method}`);

  const argSignatures = methodInfo.in_args?.map(arg => arg.signature) ?? [];
  const sig = argSignatures.join('');
  const meta = {sig, argSignatures};
  cache.set(method, meta);
  return meta;
}

export async function makeProxy(ProxyClass, busName, objectPath) {
  return ProxyClass.newAsync(
    Gio.DBus.session,
    busName,
    objectPath,
    null,
    Gio.DBusProxyFlags.DO_NOT_AUTO_START
  );
}

export async function callProxyMethod(proxy, method, args) {
  args ??= [];
  const meta = getMethodMeta(proxy, method);
  if (args.length !== meta.argSignatures.length) {
    throw new Error(
      `Argument count mismatch for ${method}: expected ${meta.argSignatures.length}, got ${args.length}`
    );
  }

  const coercedArgs = meta.argSignatures.map((sig, index) => coerceArg(sig, args[index]));
  const params = meta.sig ? new GLib.Variant(`(${meta.sig})`, coercedArgs) : null;
  return new Promise((resolve, reject) => {
    proxy.call(method, params, Gio.DBusCallFlags.NONE, -1, null, (src, res) => {
      try {
        resolve(proxy.call_finish(res));
      } catch (e) {
        reject(e);
      }
    });
  });
}

export async function getPropertyVariant(proxy, propsProxy, ifaceName, name) {
  const cached = proxy.get_cached_property(name);
  if (cached)
    return cached;

  const res = await callProxyMethod(propsProxy, 'Get', [ifaceName, name]);
  return res.deepUnpack()[0];
}

export async function getPropertyValue(proxy, propsProxy, ifaceName, name) {
  const variant = await getPropertyVariant(proxy, propsProxy, ifaceName, name);
  return variant.deepUnpack();
}
