import GLib from 'gi://GLib';

import {Extension} from 'resource:///org/gnome/shell/extensions/extension.js';

import {
  KbdProxy,
  IdleMonitorProxy,
  PropsProxy,
  callProxyMethod,
  getPropertyValue,
  makeProxy,
} from './dbus.js';

const KBD_BUS = 'org.gnome.SettingsDaemon.Power';
const KBD_IFACE = 'org.gnome.SettingsDaemon.Power.Keyboard';
const KBD_PATH = '/org/gnome/SettingsDaemon/Power';
const KBD_MAX_PERCENT = 100;
const DEBUG_SETTING = 'debug-logging';

const IDLE_PATH = '/org/gnome/Mutter/IdleMonitor/Core';
const IDLE_BUS = 'org.gnome.Mutter.IdleMonitor';

function coercePropertyValue(signature, value) {
  if (signature === 't') {
    if (typeof value === 'number')
      return BigInt(Math.max(0, Math.floor(value)));
    return value;
  }

  if (signature === 'u') {
    const n = Number(value);
    if (Number.isNaN(n))
      return value;
    return Math.max(0, Math.min(0xFFFFFFFF, Math.floor(n)));
  }

  if (signature === 'q') {
    const n = Number(value);
    if (Number.isNaN(n))
      return value;
    return Math.max(0, Math.min(0xFFFF, Math.floor(n)));
  }

  if (signature === 'i') {
    const n = Number(value);
    if (Number.isNaN(n))
      return value;
    return Math.max(-0x80000000, Math.min(0x7FFFFFFF, Math.floor(n)));
  }

  return value;
}

class GsdKbdBacklight {
  constructor(debugLog, errorLog) {
    this._proxy = null;
    this._props = null;
    this._debugLog = debugLog;
    this._errorLog = errorLog;
  }

  async init() {
    try {
      this._proxy = await makeProxy(KbdProxy, KBD_BUS, KBD_PATH);
      this._props = await makeProxy(PropsProxy, KBD_BUS, KBD_PATH);
      this._debugLog(`Connected to ${KBD_BUS} ${KBD_IFACE} at ${KBD_PATH}`);
      return true;
    } catch (e) {
      this._errorLog('Failed to init keyboard proxy', e);
      return false;
    }
  }

  get available() {
    return this._proxy !== null;
  }

  async getBrightness() {
    const value = await this._getPropertyValue('Brightness');
    this._debugLog(`Brightness read: ${value}`);
    return value;
  }

  async setBrightness(value) {
    const clamped = Math.max(0, Math.min(KBD_MAX_PERCENT, value));
    this._debugLog(`Brightness set: ${clamped}`);
    await this._setProperty('Brightness', clamped);
  }

  async _setProperty(name, value) {
    const signature = this._proxy.g_interface_info?.lookup_property(name)?.signature;
    if (!signature)
      throw new Error(`Missing property signature for ${name}`);

    const valueVariant = new GLib.Variant(signature, coercePropertyValue(signature, value));
    const res = await callProxyMethod(this._props, 'Set', [KBD_IFACE, name, valueVariant]);
    if (res)
      res.deepUnpack();
  }

  async _getPropertyValue(name) {
    return getPropertyValue(this._proxy, this._props, KBD_IFACE, name);
  }

}

class MutterIdleMonitorDbus {
  constructor(debugLog, errorLog) {
    this._proxy = null;
    this._signalId = 0;
    this._debugLog = debugLog;
    this._errorLog = errorLog;
  }

  async init() {
    try {
      this._proxy = await makeProxy(IdleMonitorProxy, IDLE_BUS, IDLE_PATH);
      this._debugLog(`Using D-Bus idle monitor: ${IDLE_BUS}${IDLE_PATH}`);
      return true;
    } catch (e) {
      this._errorLog('Failed to init idle monitor', e);
      return false;
    }
  }

  connectWatchFired(callback) {
    if (!this._proxy)
      return;

    this._signalId = this._proxy.connect('g-signal', (_p, _sender, signalName, params) => {
      if (signalName !== 'WatchFired')
        return;
      const id = params.deepUnpack()[0];
      callback(id);
    });
  }

  disconnectSignals() {
    if (this._proxy && this._signalId) {
      this._proxy.disconnect(this._signalId);
      this._signalId = 0;
    }
  }

  async addIdleWatch(intervalMs) {
    const res = await callProxyMethod(this._proxy, 'AddIdleWatch', [intervalMs]);
    return res.deepUnpack()[0];
  }

  async addUserActiveWatch() {
    const res = await callProxyMethod(this._proxy, 'AddUserActiveWatch', []);
    return res.deepUnpack()[0];
  }

  async removeWatch(id) {
    const res = await callProxyMethod(this._proxy, 'RemoveWatch', [id]);
    if (res)
      res.deepUnpack();
  }
}

export default class KbdIdleBacklightExtension extends Extension {
  enable() {
    this._disabled = false;
    this._resetPromise = null;
    this._settings = this.getSettings();
    this._timeoutSeconds = Math.max(5, this._settings.get_int('timeout-seconds'));
    this._debug = this._settings.get_boolean(DEBUG_SETTING);
    this._debugLog = (message) => {
      if (this._debug)
        log(`[kbd-idle-backlight] ${message}`);
    };
    this._errorLog = (message, error) => {
      if (!error) {
        log(`[kbd-idle-backlight] ${message}`);
        return;
      }
      if (error instanceof Error) {
        logError(error, `[kbd-idle-backlight] ${message}`);
        return;
      }
      log(`[kbd-idle-backlight] ${message}: ${String(error)}`);
    };

    this._kbd = new GsdKbdBacklight(this._debugLog, this._errorLog);
    this._idle = new MutterIdleMonitorDbus(this._debugLog, this._errorLog);

    this._idleWatchId = 0;
    this._activeWatchId = 0;

    this._forcedOff = false;
    this._savedLevel = 0;

    this._waitingForActivityWhileOff = false;

    this._settingsChangedId = this._settings.connect('changed', () => {
      this._timeoutSeconds = Math.max(5, this._settings.get_int('timeout-seconds'));
      this._debug = this._settings.get_boolean(DEBUG_SETTING);
      this._resetWatches().catch(e => {
        this._errorLog('Reset watches failed', e);
      });
    });

    this._initAsync();
  }

  async _initAsync() {
    const kbdOk = await this._kbd.init();
    const idleOk = await this._idle.init();

    this._debugLog(`Init status: kbd=${kbdOk} idle=${idleOk}`);
    if (!kbdOk || !idleOk)
      return;

    this._idle.connectWatchFired((id) => {
      this._onWatchFired(id).catch(e => {
        this._errorLog('Watch handler failed', e);
      });
    });

    try {
      await this._resetWatches();
    } catch (e) {
      this._errorLog('Reset watches failed', e);
    }
  }

  disable() {
    this._disabled = true;
    this._idle?.disconnectSignals();

    if (this._settingsChangedId) {
      this._settings.disconnect(this._settingsChangedId);
      this._settingsChangedId = 0;
    }

    const idle = this._idle;
    const kbd = this._kbd;
    const forcedOff = this._forcedOff;
    const savedLevel = this._savedLevel;

    this._cleanupWatches(idle).then(async () => {
      if (forcedOff && kbd && typeof savedLevel === 'number' && savedLevel > 0) {
        await kbd.setBrightness(savedLevel).catch(e => {
          this._errorLog('Restore brightness on disable failed', e);
        });
      }
    }).catch(e => {
      this._errorLog('Cleanup watches failed', e);
    });

    this._settings = null;
    this._kbd = null;
    this._idle = null;
    this._resetPromise = null;
    this._debugLog = () => {};
    this._errorLog = () => {};
    this._forcedOff = false;
  }

  async _cleanupWatches(idle = this._idle) {
    if (!idle)
      return;

    const idleId = this._idleWatchId;
    const activeId = this._activeWatchId;

    this._idleWatchId = 0;
    this._activeWatchId = 0;

    if (idleId)
      await idle.removeWatch(idleId).catch(e => {
        this._errorLog('Remove idle watch failed', e);
      });
    if (activeId)
      await idle.removeWatch(activeId).catch(e => {
        this._errorLog('Remove active watch failed', e);
      });
  }

  async _resetWatches() {
    if (this._resetPromise)
      return this._resetPromise;

    this._resetPromise = (async () => {
      if (this._disabled)
        return;
      if (!this._kbd?.available || !this._idle)
        return;

      await this._cleanupWatches();
      if (this._disabled || !this._kbd?.available || !this._idle)
        return;

      if (this._forcedOff || this._waitingForActivityWhileOff) {
        try {
          this._activeWatchId = await this._idle.addUserActiveWatch();
        } catch (e) {
          this._errorLog('Add user-active watch failed', e);
          await this._resetWatches();
        }
        return;
      }

      const intervalMs = this._timeoutSeconds * 1000;
      this._idleWatchId = await this._idle.addIdleWatch(intervalMs);
    })().finally(() => {
      this._resetPromise = null;
    });

    return this._resetPromise;
  }

  async _onWatchFired(id) {
    if (this._disabled)
      return;
    if (id === this._idleWatchId) {
      this._idleWatchId = 0;
      this._debugLog('Idle watch fired');

      const cur = await this._kbd.getBrightness().catch(e => {
        this._errorLog('Read brightness failed', e);
        return null;
      });
      if (typeof cur !== 'number') {
        await this._resetWatches();
        return;
      }

      this._savedLevel = cur;

      if (cur === 0) {
        this._forcedOff = false;
        this._waitingForActivityWhileOff = true;

        this._activeWatchId = await this._idle.addUserActiveWatch();
        return;
      }

      await this._kbd.setBrightness(0).catch(e => {
        this._errorLog('Set brightness failed', e);
      });
      this._forcedOff = true;
      this._waitingForActivityWhileOff = false;

      try {
        this._activeWatchId = await this._idle.addUserActiveWatch();
      } catch (e) {
        this._errorLog('Add user-active watch failed', e);
        await this._resetWatches();
      }
      return;
    }

    if (id === this._activeWatchId) {
      this._activeWatchId = 0;
      this._debugLog('User active watch fired');

      const level = this._savedLevel;

      if (this._waitingForActivityWhileOff) {
        this._waitingForActivityWhileOff = false;
        await this._resetWatches();
        return;
      }

      if (this._forcedOff) {
        const current = await this._kbd.getBrightness().catch(e => {
          this._errorLog('Read brightness before restore failed', e);
          return null;
        });
        if (typeof current === 'number' && current > 0) {
          this._savedLevel = current;
          this._forcedOff = false;
          await this._resetWatches();
          return;
        }

        if (typeof level !== 'number' || level <= 0) {
          this._forcedOff = false;
          await this._resetWatches();
          return;
        }

        const clamped = Math.max(0, Math.min(level, KBD_MAX_PERCENT));
        await this._kbd.setBrightness(clamped).catch(e => {
          this._errorLog('Restore brightness failed', e);
        });
        this._forcedOff = false;
      }

      await this._resetWatches();
    }
  }
}
