import Adw from 'gi://Adw';
import Gio from 'gi://Gio';
import Gtk from 'gi://Gtk?version=4.0';

import {ExtensionPreferences, gettext as _} from 'resource:///org/gnome/Shell/Extensions/js/extensions/prefs.js';

import {KbdProxy, PropsProxy, getPropertyValue, makeProxy} from './dbus.js';

const KBD_BUS = 'org.gnome.SettingsDaemon.Power';
const KBD_IFACE = 'org.gnome.SettingsDaemon.Power.Keyboard';
const KBD_PATH = '/org/gnome/SettingsDaemon/Power';

async function detectKbdProxies() {
  try {
    const kbd = await makeProxy(KbdProxy, KBD_BUS, KBD_PATH);
    const props = await makeProxy(PropsProxy, KBD_BUS, KBD_PATH);
    return {kbd, props};
  } catch (_) {
    return null;
  }
}

export default class KbdIdleBacklightPrefs extends ExtensionPreferences {
  fillPreferencesWindow(window) {
    this.initTranslations();
    const settings = this.getSettings();

    const page = new Adw.PreferencesPage({
      title: _('Keyboard Backlight'),
      icon_name: 'input-keyboard-symbolic',
    });

    const group = new Adw.PreferencesGroup({
      title: _('Idle Timeout'),
      description: _('Turn off keyboard backlight after user inactivity and restore previous level on activity.'),
    });

    const timeoutRow = new Adw.SpinRow({
      title: _('Timeout (seconds)'),
      subtitle: _('How long to wait before turning the backlight off.'),
      adjustment: new Gtk.Adjustment({
        lower: 5,
        upper: 3600,
        step_increment: 5,
        page_increment: 60,
        value: settings.get_int('timeout-seconds'),
      }),
    });
    settings.bind(
      'timeout-seconds',
      timeoutRow,
      'value',
      Gio.SettingsBindFlags.DEFAULT
    );

    const detectedRow = new Adw.ActionRow({
      title: _('Brightness range'),
      subtitle: _('Detecting…'),
    });

    const debugRow = new Adw.SwitchRow({
      title: _('Debug logging'),
      subtitle: _('Write verbose logs to the journal.'),
    });
    settings.bind(
      'debug-logging',
      debugRow,
      'active',
      Gio.SettingsBindFlags.DEFAULT
    );

    group.add(timeoutRow);
    group.add(debugRow);
    group.add(detectedRow);

    page.add(group);
    window.add(page);

    let closed = false;
    window.connect('close-request', () => {
      closed = true;
      return false;
    });

    (async () => {
      const proxies = await detectKbdProxies();
      if (!proxies) {
        if (!closed && detectedRow.get_root())
          detectedRow.subtitle = _('Keyboard backlight control is not available on this system.');
        return;
      }

      const cur = await getPropertyValue(proxies.kbd, proxies.props, KBD_IFACE, 'Brightness');
      if (!closed && detectedRow.get_root()) {
        if (typeof cur !== 'number')
          detectedRow.subtitle = _('0..100 (current: unavailable)');
        else
          detectedRow.subtitle = _('0..100 (current: %d)').format(cur);
      }
    })().catch(() => {
      if (!closed && detectedRow.get_root())
        detectedRow.subtitle = _('Detection failed');
    });
  }
}
