'use strict';

import St from 'gi://St';
import GLib from 'gi://GLib';
import GObject from 'gi://GObject';
import Gio from 'gi://Gio';
import Clutter from 'gi://Clutter';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import * as PopupMenu from 'resource:///org/gnome/shell/ui/popupMenu.js';
import {Extension} from 'resource:///org/gnome/shell/extensions/extension.js';

// Security helper function for command validation
function validateCommand(command) {
    // List of allowed base commands for security
    const allowedCommands = ['docker', 'podman', 'pgrep'];
    
    // Extract base command (first word before any space)
    const commandParts = command.trim().split(/\s+/);
    const baseCommand = commandParts[0];
    
    // Check if base command is in allowed list
    if (!allowedCommands.includes(baseCommand)) {
        throw new Error(`Unauthorized command: ${baseCommand}`);
    }
    
    // Additional security checks
    if (command.includes('&&') || command.includes('||') || command.includes('|')) {
        throw new Error('Command chaining is not allowed');
    }
    
    if (command.includes('rm ') || command.includes('delete') || command.includes('remove')) {
        throw new Error('Destructive commands are not allowed');
    }
    
    return true;
}

// Debug helper function
function debugLog(message, settings = null) {
    if (settings && settings.get_boolean('debug-mode')) {
        console.debug(`WINBOAT: ${message}`);
    }
}

export default class BoatmanExtension extends Extension {
    enable() {
        this._settings = this.getSettings();
        this._indicator = new BoatmanIndicator(this, this._settings);
        Main.panel.addToStatusArea(this.uuid, this._indicator);
    }

    disable() {
        this._indicator?.destroy();
        this._indicator = null;
        this._settings = null;
    }
}

class BoatmanIndicator extends PanelMenu.Button {
    static {
        GObject.registerClass(this);
    }

    constructor(extension, settings) {
        super(0.0, 'WinBoat');
        this._extension = extension;
        this._settings = settings;
        this._containerRuntime = null;
        
        // Performance optimization: cache for settings and frequently accessed data
        this._settingsCache = new Map();
        this._statsCache = {
            data: null,
            timestamp: 0,
            ttl: 30000 // 30 seconds cache TTL
        };
        this._runtimeCache = {
            runtime: null,
            timestamp: 0,
            ttl: 300000 // 5 minutes cache TTL
        };
        
        // UI/UX improvements: progress indicators and state management
        this._progressIndicator = null;
        this._isOperating = false; // Track if container operation is in progress
        this._lastOperation = null; // Track last operation for better feedback
        
        // Store last stats to persist across menu opens/closes
        this._lastStats = {
            cpu: 'Click to check',
            memory: 'Click to check',
            ports: 'Click to check',
            timestamp: null
        };
        
        // Auto-shutdown tracking
        this._autoShutdownTimer = null;
        this._countdownTimer = null;
        this._lastActivityTime = null;
        this._autoShutdownEnabled = false;
        this._warningShown = false;
        this._warningNotification = null;
        
        // Set up the panel icon with proper GNOME Shell styling
        this._icon = new St.Icon({
            icon_name: 'system-run-symbolic',
            style_class: 'system-status-icon',
            y_expand: false,
            y_align: Clutter.ActorAlign.CENTER
        });
        this.add_child(this._icon);
        
        // Set up the menu with proper GNOME Shell structure
        this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
        
        // Add status item
        this._statusItem = new PopupMenu.PopupMenuItem('Checking status...');
        this._statusItem.reactive = false;
        this.menu.addMenuItem(this._statusItem);
        
        // Add progress indicator placeholder (will be shown/hidden as needed)
        this._progressIndicator = new PopupMenu.PopupMenuItem('Ready');
        this._progressIndicator.reactive = false;
        this._progressIndicator.visible = false;
        this._progressIndicator.add_style_class_name('popup-subtitle');
        this.menu.addMenuItem(this._progressIndicator);
        
        // Add separator
        this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
        
        // Add Open WinBoat App item (before action buttons)
        this._webUiItem = new PopupMenu.PopupMenuItem('Open Interface');
        this._webUiItem.connect('activate', () => this._openWinBoatApplication());
        this.menu.addMenuItem(this._webUiItem);
        
        // Add action buttons
        this._startButton = new PopupMenu.PopupMenuItem('Start WinBoat');
        this._stopButton = new PopupMenu.PopupMenuItem('Stop WinBoat');
        this._restartButton = new PopupMenu.PopupMenuItem('Restart WinBoat');
        
        this._startButton.connect('activate', () => this._startWinBoat());
        this._stopButton.connect('activate', () => this._stopWinBoat());
        this._restartButton.connect('activate', () => this._restartWinBoat());
        
        this.menu.addMenuItem(this._startButton);
        this.menu.addMenuItem(this._stopButton);
        this.menu.addMenuItem(this._restartButton);
        
        // Add separator
        this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
        
        // Add uptime info item
        this._uptimeItem = new PopupMenu.PopupMenuItem('Checking uptime...');
        this._uptimeItem.reactive = false;
        this.menu.addMenuItem(this._uptimeItem);
        
        // Add auto-shutdown countdown item (hidden by default)
        this._countdownItem = new PopupMenu.PopupMenuItem('Auto-shutdown: Disabled');
        this._countdownItem.reactive = false;
        this._countdownItem.visible = false;
        this.menu.addMenuItem(this._countdownItem);
        
        // Add separator
        this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
        
        // Add stats items (hidden by default)
        this._cpuItem = new PopupMenu.PopupMenuItem('CPU: Click to check');
        this._cpuItem.reactive = true;
        this._cpuItem.connect('activate', () => {
            this._recordActivity();
            this._updateStats();
        });
        this.menu.addMenuItem(this._cpuItem);
        
        this._memoryItem = new PopupMenu.PopupMenuItem('Memory: Click to check');
        this._memoryItem.reactive = true;
        this._memoryItem.connect('activate', () => {
            this._recordActivity();
            this._updateStats();
        });
        this.menu.addMenuItem(this._memoryItem);
        
        this._portsItem = new PopupMenu.PopupMenuItem('Ports: Click to check');
        this._portsItem.reactive = true;
        this._portsItem.connect('activate', () => {
            this._recordActivity();
            this._updateStats();
        });
        this.menu.addMenuItem(this._portsItem);
        
        // Add separator
        this.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
     
        // Add refresh item
        const refreshItem = new PopupMenu.PopupMenuItem('Refresh Status');
        refreshItem.connect('activate', () => {
            this._recordActivity();
            this._updateStatus();
            // Also refresh stats if they are enabled AND container is running
            if (this._getCachedSetting('show-stats') && this._lastStatus === 'running') {
                this._updateStats();
            }
        });
        this.menu.addMenuItem(refreshItem);
        
        // Connect to settings changes with cache invalidation
        this._showUptimeSignalId = this._settings.connect('changed::show-uptime', () => {
            this._invalidateSetting('show-uptime');
            this._updateStatus();
        });
        this._showStatsSignalId = this._settings.connect('changed::show-stats', () => {
            this._invalidateSetting('show-stats');
            this._updateMenuVisibility();
        });
        this._compactModeSignalId = this._settings.connect('changed::compact-mode', () => {
            this._invalidateSetting('compact-mode');
            this._updateStatus();
        });
        
        // Auto-shutdown settings connections with cache invalidation
        this._autoShutdownEnabledSignalId = this._settings.connect('changed::auto-shutdown-enabled', () => {
            this._invalidateSetting('auto-shutdown-enabled');
            this._updateAutoShutdownSettings();
        });
        this._autoShutdownHoursSignalId = this._settings.connect('changed::auto-shutdown-hours', () => {
            this._invalidateSetting('auto-shutdown-hours');
            this._updateAutoShutdownSettings();
        });
        this._autoShutdownWarningSignalId = this._settings.connect('changed::auto-shutdown-warning-minutes', () => {
            this._invalidateSetting('auto-shutdown-warning-minutes');
            this._updateAutoShutdownSettings();
        });
        
        // Set up periodic updates using configured refresh interval
        this._scheduleStatusUpdates();

        // Initial status update
        this._updateStatus();
        
        // Initialize auto-shutdown system
        this._updateAutoShutdownSettings();
    }

    // Performance optimization: cached settings access
    _getCachedSetting(key) {
        if (this._settingsCache.has(key)) {
            return this._settingsCache.get(key);
        }
        
        let value;
        switch (key) {
            case 'refresh-interval':
                value = this._settings.get_int(key);
                break;
            case 'enable-notifications':
            case 'show-uptime':
            case 'show-stats':
            case 'compact-mode':
            case 'debug-mode':
            case 'auto-shutdown-enabled':
                value = this._settings.get_boolean(key);
                break;
            case 'runtime-mode':
                value = this._settings.get_string(key);
                break;
            case 'auto-shutdown-hours':
            case 'auto-shutdown-warning-minutes':
                value = this._settings.get_int(key);
                break;
            case 'last-activity-timestamp':
                value = this._settings.get_int64(key);
                break;
            default:
                value = null;
        }
        
        this._settingsCache.set(key, value);
        return value;
    }
    
    // Performance optimization: invalidate settings cache
    _invalidateSettingsCache() {
        this._settingsCache.clear();
    }
    
    // Performance optimization: cache invalidation for specific key
    _invalidateSetting(key) {
        this._settingsCache.delete(key);
    }
    
    // UI/UX improvements: show progress indicator using built-in GNOME styles
    _showProgressIndicator(message) {
        // Use the existing progress indicator instead of creating a new one
        this._progressIndicator.label.text = message;
        this._progressIndicator.visible = true;
        this._progressIndicator.add_style_class_name('popup-subtitle');
    }
    
    // UI/UX improvements: hide progress indicator
    _hideProgressIndicator() {
        if (this._progressIndicator) {
            this._progressIndicator.visible = false;
            this._progressIndicator.remove_style_class_name('popup-subtitle');
        }
    }
    
    // UI/UX improvements: set button states using built-in GNOME styles
    _setButtonStates(operating, operation = null) {
        this._isOperating = operating;
        this._lastOperation = operation;
        
        // Disable all action buttons during operation
        this._startButton.reactive = !operating;
        this._stopButton.reactive = !operating;
        this._restartButton.reactive = !operating;
        
        // Use built-in GNOME Shell styles for visual feedback
        if (operating) {
            this._startButton.add_style_class_name('popup-inactive-menu-item');
            this._stopButton.add_style_class_name('popup-inactive-menu-item');
            this._restartButton.add_style_class_name('popup-inactive-menu-item');
        } else {
            this._startButton.remove_style_class_name('popup-inactive-menu-item');
            this._stopButton.remove_style_class_name('popup-inactive-menu-item');
            this._restartButton.remove_style_class_name('popup-inactive-menu-item');
        }
    }
    
    // UI/UX improvements: show operation feedback
    _showOperationFeedback(operation, success, details = null) {
        const compactMode = this._getCachedSetting('compact-mode');
        let message = '';
        
        if (success) {
            switch (operation) {
                case 'start':
                    message = compactMode ? 'Starting...' : 'Starting WinBoat...';
                    break;
                case 'stop':
                    message = compactMode ? 'Stopping...' : 'Stopping WinBoat...';
                    break;
                case 'restart':
                    message = compactMode ? 'Restarting...' : 'Restarting WinBoat...';
                    break;
                default:
                    message = 'Processing...';
            }
        } else {
            switch (operation) {
                case 'start':
                    message = compactMode ? 'Start failed' : 'Failed to start WinBoat';
                    break;
                case 'stop':
                    message = compactMode ? 'Stop failed' : 'Failed to stop WinBoat';
                    break;
                case 'restart':
                    message = compactMode ? 'Restart failed' : 'Failed to restart WinBoat';
                    break;
                default:
                    message = 'Operation failed';
            }
        }
        
        if (details && !compactMode) {
            message += `\n${details}`;
        }
        
        this._statusItem.label.text = message;
    }
    
    _setStatusIcon(status) {
        let icon_name = null;

        switch (status) {
            case 'running':
                icon_name = 'emblem-ok-symbolic';
                break;
            case 'stopped':
                icon_name = 'media-playback-stop-symbolic';
                break;
            case 'booting':
                icon_name = 'system-run-symbolic';
                break;
            case 'docker_down':
            case 'error':
            default:
                icon_name = 'action-unavailable-symbolic';
                break;
        }

        // Try custom icons first, then fallback to symbolic
        try {
            const customIcons = {
                'running': 'boat.svg',
                'stopped': 'anchor.svg',
                'booting': 'life-guard.svg',
                'error': 'life-guard.svg',
                'docker_down': 'life-guard.svg'
            };
            
            const fileName = customIcons[status] || 'life-guard.svg';
            const file = this._extension.dir.get_child('icons').get_child(fileName);
            
            if (file && file.query_exists(null)) {
                const gicon = Gio.icon_new_for_string(file.get_path());
                this._icon.gicon = gicon;
                console.info(`WINBOAT: Loaded custom icon: ${fileName}`);
            } else {
                // Fallback to symbolic
                this._icon.icon_name = icon_name;
                console.info(`WINBOAT: Using symbolic icon: ${icon_name}`);
            }
        } catch (e) {
            console.error('WINBOAT: Failed to load custom icon, using symbolic', e);
            // Fallback to symbolic
            this._icon.icon_name = icon_name;
        }
    }

    // Performance optimization: optimized schedule status updates with cache invalidation
    _scheduleStatusUpdates() {
        const interval = Math.max(1, this._getCachedSetting('refresh-interval'));

        if (this._timeout) {
            GLib.Source.remove(this._timeout);
            this._timeout = null;
        }

        this._timeout = GLib.timeout_add_seconds(
            GLib.PRIORITY_DEFAULT,
            interval,
            () => {
                this._updateStatus();
                return GLib.SOURCE_CONTINUE;
            }
        );

        // React to changes in refresh-interval with cache invalidation
        if (!this._refreshSignalId) {
            this._refreshSignalId = this._settings.connect('changed::refresh-interval', () => {
                this._invalidateSetting('refresh-interval'); // Invalidate cache
                this._scheduleStatusUpdates();
            });
        }
    }

    async _runCommandAsync(command) {
        return new Promise((resolve, reject) => {
            try {
                // Validate command for security
                validateCommand(command);
                
                const proc = Gio.Subprocess.new(
                    ['/bin/bash', '-c', command],
                    Gio.SubprocessFlags.STDOUT_PIPE | Gio.SubprocessFlags.STDERR_SILENCE
                );
                
                proc.wait_check_async(null, (source, result) => {
                    try {
                        const success = source.wait_check_finish(result);
                        resolve(success);
                    } catch (e) {
                        reject(e);
                    }
                });
            } catch (e) {
                reject(e);
            }
        });
    }

    async _runCommandAsyncWithOutput(command, timeoutSeconds = null) {
        return new Promise((resolve, reject) => {
            try {
                // Validate command for security
                validateCommand(command);
                
                const proc = Gio.Subprocess.new(
                    ['/bin/bash', '-c', command],
                    Gio.SubprocessFlags.STDOUT_PIPE | Gio.SubprocessFlags.STDERR_PIPE
                );

                let timeoutId = 0;
                let settled = false;

                const settle = (result) => {
                    if (settled)
                        return;

                    settled = true;

                    if (timeoutId) {
                        GLib.Source.remove(timeoutId);
                        timeoutId = 0;
                    }

                    resolve(result);
                };

                if (timeoutSeconds && timeoutSeconds > 0) {
                    timeoutId = GLib.timeout_add_seconds(
                        GLib.PRIORITY_DEFAULT,
                        timeoutSeconds,
                        () => {
                            proc.force_exit();
                            settle([false, '', 'Command timed out']);
                            return GLib.SOURCE_REMOVE;
                        }
                    );
                }

                proc.communicate_utf8_async(null, null, (source, result) => {
                    try {
                        const [, stdout, stderr] = source.communicate_utf8_finish(result);
                        const success = source.get_successful();
                        settle([success, stdout, stderr]);
                    } catch (e) {
                        if (!settled) {
                            settled = true;
                            if (timeoutId)
                                GLib.Source.remove(timeoutId);
                            reject(e);
                        }
                    }
                });
            } catch (e) {
                reject(e);
            }
        });
    }

    // Performance optimization: cached runtime detection
    async _getContainerRuntime() {
        // Check cache first
        const now = Date.now();
        if (this._runtimeCache.runtime && 
            (now - this._runtimeCache.timestamp) < this._runtimeCache.ttl) {
            return this._runtimeCache.runtime;
        }

        if (this._containerRuntime) {
            // Update cache with existing runtime
            this._runtimeCache.runtime = this._containerRuntime;
            this._runtimeCache.timestamp = now;
            return this._containerRuntime;
        }

        // Respect user override first
        const mode = this._getCachedSetting('runtime-mode');
        if (mode === 'docker' || mode === 'podman') {
            try {
                // Add timeout to prevent hanging
                const success = await this._runCommandAsync(`${mode} ps`);
                if (success) {
                    this._containerRuntime = mode;
                    this._runtimeCache.runtime = mode;
                    this._runtimeCache.timestamp = now;
                    return mode;
                }
            } catch (e) {
                // Fall through to auto-detect
                debugLog(`Runtime override failed for ${mode}, falling back to auto-detect`, this._settings);
            }
        }

        const runtimes = ['docker', 'podman'];

        for (const runtime of runtimes) {
            try {
                // Add timeout to prevent hanging
                const success = await this._runCommandAsync(`${runtime} ps`);
                if (success) {
                    this._containerRuntime = runtime;
                    this._runtimeCache.runtime = runtime;
                    this._runtimeCache.timestamp = now;
                    debugLog(`Auto-detected runtime: ${runtime}`, this._settings);
                    return runtime;
                }
            } catch (e) {
                // Ignore and try next runtime
                debugLog(`Runtime check failed for ${runtime}`, this._settings);
            }
        }

        return null;
    }

    _clearBootCheckTimeout() {
        if (this._bootCheckTimeout) {
            GLib.Source.remove(this._bootCheckTimeout);
            this._bootCheckTimeout = null;
        }
    }

    _clearStatusUpdateTimeout() {
        if (this._statusUpdateTimeout) {
            GLib.Source.remove(this._statusUpdateTimeout);
            this._statusUpdateTimeout = null;
        }
    }

    _clearStatsUpdateTimeout() {
        if (this._statsUpdateTimeout) {
            GLib.Source.remove(this._statsUpdateTimeout);
            this._statsUpdateTimeout = null;
        }
    }

    // Performance optimization: optimized status update with cached settings
    async _updateStatus() {
        const previousStatus = this._lastStatus;
        let status = await this._isBoatmanRunning();

        // Ensure we always have a well-formed status object
        if (!status || typeof status !== 'object' || !status.status) {
            status = { status: 'error' };
        }

        this._lastStatus = status.status;

        // Performance optimization: cache compact mode to avoid repeated calls
        const compactMode = this._getCachedSetting('compact-mode');

        switch (status.status) {
            case 'running':
                this._setStatusIcon('running');
                this._statusItem.label.text = compactMode ? 'Running' : 'WinBoat is running';
                this._startButton.visible = false;
                this._stopButton.visible = true;
                this._restartButton.visible = true;
                
                // Re-enable action buttons
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                
                // Record activity and start auto-shutdown timer when container starts running
                if (previousStatus !== 'running') {
                    this._recordActivity();
                    this._startAutoShutdownTimer();
                }
                
                // Performance optimization: only update uptime if enabled and cache is stale
                if (this._getCachedSetting('show-uptime')) {
                    const uptime = await this._getContainerUptime();
                    this._uptimeItem.label.text = compactMode ? uptime : `Uptime: ${uptime}`;
                }
                break;
                
            case 'booting':
                this._setStatusIcon('booting');
                this._statusItem.label.text = compactMode ? 'Starting...' : 'WinBoat is starting...';
                this._startButton.visible = false;
                this._stopButton.visible = true;
                this._restartButton.visible = false;
                
                // Re-enable action buttons
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                
                // Update uptime if enabled
                if (this._getCachedSetting('show-uptime')) {
                    this._uptimeItem.label.text = compactMode ? 'Starting...' : 'Starting...';
                }
                
                // Check status again in 5 seconds
                this._clearBootCheckTimeout();
                this._bootCheckTimeout = GLib.timeout_add_seconds(
                    GLib.PRIORITY_DEFAULT,
                    5,
                    () => {
                        this._bootCheckTimeout = null;
                        this._updateStatus();
                        return GLib.SOURCE_REMOVE;
                    }
                );
                break;
                
            case 'docker_down':
                this._setStatusIcon('docker_down');
                this._statusItem.label.text = compactMode ? 'Docker down' : 'Docker not running';
                this._startButton.visible = false;
                this._stopButton.visible = false;
                this._restartButton.visible = false;
                
                // Re-enable action buttons
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                
                // Update uptime if enabled
                if (this._getCachedSetting('show-uptime')) {
                    this._uptimeItem.label.text = compactMode ? 'Docker down' : 'Docker unavailable';
                }
                break;
                
            case 'error':
                this._setStatusIcon('error');
                this._statusItem.label.text = compactMode ? 'Error' : 'Error checking WinBoat status';
                this._startButton.visible = true;
                this._stopButton.visible = false;
                this._restartButton.visible = false;
                
                // Re-enable action buttons
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                
                // Update uptime if enabled
                if (this._getCachedSetting('show-uptime')) {
                    this._uptimeItem.label.text = compactMode ? 'Unknown' : 'Status unknown';
                }
                break;

            case 'stopped':
            default:
                this._setStatusIcon('stopped');
                this._statusItem.label.text = compactMode ? 'Stopped' : 'WinBoat is stopped';
                this._startButton.visible = true;
                this._stopButton.visible = false;
                this._restartButton.visible = false;
                
                // Re-enable action buttons
                this._startButton.reactive = true;
                this._stopButton.reactive = true;
                this._restartButton.reactive = true;
                
                // Clear auto-shutdown timers when container stops
                if (previousStatus === 'running') {
                    this._clearAutoShutdownTimers();
                    this._settings.set_int64('last-activity-timestamp', 0);
                    this._lastActivityTime = null;
                    this._warningShown = false;
                }
                
                // Update uptime if enabled
                if (this._getCachedSetting('show-uptime')) {
                    this._uptimeItem.label.text = compactMode ? 'Stopped' : 'Container stopped';
                }
        }
        
        // Update menu visibility based on settings
        this._updateMenuVisibility();
        
        // Update countdown display
        this._updateCountdownDisplay();
        
        this._maybeNotify(previousStatus, status.status);
        return status.status === 'running';
    }

    // Performance optimization: optimized menu visibility updates
    _updateMenuVisibility() {
        // Use cached settings for better performance
        const showUptime = this._getCachedSetting('show-uptime');
        const showStats = this._getCachedSetting('show-stats');
        const compactMode = this._getCachedSetting('compact-mode');
        
        // Only show stats if container is running and stats setting is enabled
        const containerRunning = this._lastStatus === 'running';
        const shouldShowStats = showStats && containerRunning;

        // Show/hide uptime item
        if (this._uptimeItem) {
            this._uptimeItem.visible = showUptime;
        }

        // Show/hide stats items and separator
        if (this._cpuItem && this._memoryItem && this._portsItem) {
            this._cpuItem.visible = shouldShowStats;
            this._memoryItem.visible = shouldShowStats;
            this._portsItem.visible = shouldShowStats;
            
            // Only restore/update stats when container is running and stats are enabled
            if (shouldShowStats) {
                const timeAgo = this._formatTimestamp(this._lastStats.timestamp);
                
                // Performance optimization: avoid repeated string operations
                const cpuText = `CPU: ${this._lastStats.cpu}${timeAgo}`;
                const memoryText = `Memory: ${this._lastStats.memory}${timeAgo}`;
                const portsText = `Ports: ${this._lastStats.ports}${timeAgo}`;
                
                this._cpuItem.label.text = cpuText;
                this._memoryItem.label.text = memoryText;
                this._portsItem.label.text = portsText;
            }
        }

        // Performance optimization: batch update button labels
        const buttonLabels = compactMode ? {
            start: 'Start',
            stop: 'Stop',
            restart: 'Restart',
            webUi: 'Interface'
        } : {
            start: 'Start WinBoat',
            stop: 'Stop WinBoat',
            restart: 'Restart WinBoat',
            webUi: 'Open Interface'
        };
        
        this._startButton.label.text = buttonLabels.start;
        this._stopButton.label.text = buttonLabels.stop;
        this._restartButton.label.text = buttonLabels.restart;
        this._webUiItem.label.text = buttonLabels.webUi;
    }

    _maybeNotify(previousStatus, currentStatus) {
        if (!previousStatus || previousStatus === currentStatus)
            return;

        let message = null;

        if (currentStatus === 'running') {
            message = 'WinBoat is now running.';
        } else if (currentStatus === 'stopped') {
            message = 'WinBoat has stopped.';
        } else if (currentStatus === 'docker_down' || currentStatus === 'error') {
            message = 'WinBoat extension: container runtime or WinBoat status error.';
        }

        if (message)
            this._notifyUser('WinBoat', message);
    }

    _notifyUser(summary, body = null, persistent = false) {
        if (!this._settings || !this._settings.get_boolean('enable-notifications'))
            return null;

        const title = summary || 'WinBoat';
        const message = body ? Main.notify(title, body) : Main.notify(title);

        if (persistent && message && typeof message.setTransient === 'function') {
            message.setTransient(false);
        }

        return message;
    }

    async _isBoatmanRunning() {
        try {
            const runtime = await this._getContainerRuntime();
            if (!runtime) {
                debugLog('Container runtime not accessible', this._settings);
                return { status: 'docker_down' };
            }

            // Use docker inspect for reliable status check
            const cmd = `${runtime} inspect WinBoat --format "{{.State.Status}}"`;
            debugLog(`Running command: ${cmd}`, this._settings);
            const [success, stdout, stderr] = await this._runCommandAsyncWithOutput(cmd);

            debugLog(`Command result: success=${success}, stdout=${stdout ? stdout.toString().trim() : 'null'}, stderr=${stderr ? stderr.toString().trim() : 'null'}`, this._settings);

            if (!success || !stdout) {
                debugLog('Container not found or inspect failed', this._settings);
                return { status: 'stopped' };
            }

            const status = stdout.toString().trim();
            debugLog(`Container status: ${status}`, this._settings);

            if (status === 'running') {
                return { status: 'running', containerRunning: true };
            } else {
                return { status: 'stopped' };
            }
                
        } catch (e) {
            // Enhanced error logging with context
            console.error('WINBOAT: Error checking status', {
                error: e.message,
                stack: e.stack,
                runtime: this._containerRuntime,
                timestamp: new Date().toISOString()
            });
            return { status: 'error' };
        }
    }

    _checkAutoShutdown() {
        if (!this._autoShutdownEnabled || this._lastStatus !== 'running') {
            return;
        }
        
        const now = Math.floor(Date.now() / 1000);
        const hours = this._settings.get_int('auto-shutdown-hours');
        const timeoutSeconds = hours * 3600;
        const warningThreshold = this._getWarningLeadSeconds();
        
        // Always check for active processes to keep activity fresh
        this._hasActiveProcesses().then(hasActiveProcesses => {
            if (hasActiveProcesses) {
                // Active processes detected - reset activity timer
                debugLog('Active processes detected, resetting activity timer', this._settings);
                this._recordActivity();
                // Clear any pending warning
                if (this._warningShown) {
                    this._warningShown = false;
                    this._dismissWarningNotification();
                }
                return;
            }
            
            // No active processes - check if timeout reached
            if (this._lastActivityTime) {
                const timeUntilShutdown = timeoutSeconds - (now - this._lastActivityTime);
                
                // Show warning 5 minutes before shutdown
                if (timeUntilShutdown <= warningThreshold && timeUntilShutdown > 0 && !this._warningShown) {
                    this._warningShown = true;
                    this._showShutdownWarning(timeUntilShutdown);
                    return;
                }
                
                // Shutdown time reached
                if (timeUntilShutdown <= 0) {
                    debugLog(`Auto-shutdown triggered after ${hours} hours of inactivity`, this._settings);
                    this._performAutoShutdown().catch(e => {
                        console.error('WINBOAT: Error during auto-shutdown', e);
                    });
                }
            }
        }).catch(e => {
            // Enhanced error logging with context
            console.error('WINBOAT: Error during auto-shutdown', {
                error: e.message,
                stack: e.stack,
                autoShutdownEnabled: this._autoShutdownEnabled,
                lastActivityTime: this._lastActivityTime,
                timestamp: new Date().toISOString()
            });
        });
    }

    // Performance optimization: optimized activity recording with cache awareness
    _recordActivity() {
        if (!this._autoShutdownEnabled || this._lastStatus !== 'running') {
            return;
        }
        
        const now = Math.floor(Date.now() / 1000);
        
        // Only write to GSettings if activity changed significantly (at least 30 seconds)
        // This reduces disk I/O and GSettings synchronization overhead
        if (this._lastActivityTime && (now - this._lastActivityTime) < 30) {
            return; // Too soon, don't write
        }
        
        this._lastActivityTime = now;
        this._settings.set_int64('last-activity-timestamp', now);
        this._warningShown = false; // Reset warning flag when activity recorded
        this._dismissWarningNotification();
        debugLog(`Activity recorded: ${new Date(now * 1000).toLocaleString()}`, this._settings);
    }

    _snoozeAutoShutdown() {
        debugLog('Snoozing auto-shutdown for 30 minutes', this._settings);
        this._recordActivity(); // Reset the activity timer
        this._warningShown = false;
        
        // Show notification
        this._notifyUser('WinBoat', 'Auto-shutdown snoozed for 30 minutes');
    }

    _turnOffAutoShutdown() {
        debugLog('Turning off auto-shutdown', this._settings);
        this._settings.set_boolean('auto-shutdown-enabled', false);
        this._warningShown = false;
        
        // Show notification
        this._notifyUser('WinBoat', 'Auto-shutdown turned off');
    }

    async _hasActiveProcesses() {
        try {
            const runtime = await this._getContainerRuntime();
            if (!runtime) {
                return false;
            }

            // Check for common Windows processes with shorter timeout to prevent hanging
            const watchedProcesses = 'xfreerdp3|winboat|explorer|winword|excel|powerpnt|chrome|firefox|msedge';
            const cmd = `${runtime} exec WinBoat pgrep -x "${watchedProcesses}"`;
            
            // Use shorter timeout (3 seconds instead of 5) for better responsiveness
            const [success] = await this._runCommandAsyncWithOutput(cmd, 3);
            return success;
                
        } catch (e) {
            // Enhanced error logging with context
            console.error('WINBOAT: Error checking processes', {
                error: e.message,
                stack: e.stack,
                runtime: this._containerRuntime,
                timestamp: new Date().toISOString()
            });
            // On error, assume activity to be safe (fail-safe approach)
            debugLog('Process check failed, assuming activity for safety', this._settings);
            this._recordActivity();
        }
    }

    _getWarningLeadSeconds() {
        const allowedMinutes = [5, 10, 15];
        let minutes = this._settings.get_int('auto-shutdown-warning-minutes');

        if (!allowedMinutes.includes(minutes))
            minutes = 5;

        return minutes * 60;
    }

    // UI/UX improvements: enhanced start WinBoat with progress feedback and detailed debugging
    async _startWinBoat() {
        try {
            // Show progress and set button states
            this._showProgressIndicator('Starting WinBoat...');
            this._setButtonStates(true, 'start');
            this._showOperationFeedback('start', true);
            
            debugLog('Starting WinBoat container...', this._settings);
            
            const runtime = await this._getContainerRuntime();
            debugLog(`Detected runtime: ${runtime}`, this._settings);
            
            if (!runtime) {
                debugLog('No container runtime detected', this._settings);
                this._showOperationFeedback('start', false, 'No container runtime detected');
                this._notifyUser('WinBoat', 'No container runtime (Docker/Podman) detected.');
                this._setButtonStates(false);
                return;
            }
            
            const command = `${runtime} start WinBoat`;
            debugLog(`Executing command: ${command}`, this._settings);
            
            const success = await this._runCommandAsync(command);
            debugLog(`Command result: success=${success}`, this._settings);

            if (!success) {
                debugLog('Container start command failed', this._settings);
                this._showOperationFeedback('start', false, 'Container failed to start');
                this._notifyUser('WinBoat', 'Failed to start WinBoat container');
                this._setButtonStates(false);
                return;
            }

            debugLog('WinBoat container started successfully', this._settings);
            this._notifyUser('WinBoat', 'WinBoat container started');
            
            // Record activity when container starts
            this._recordActivity();
            
            // Force immediate status update using GLib timeout
            this._clearStatusUpdateTimeout();
            this._statusUpdateTimeout = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 1000, () => {
                this._statusUpdateTimeout = null;
                this._updateStatus();
                return GLib.SOURCE_REMOVE;
            });
            
        } catch (e) {
            // Enhanced error logging with context
            console.error('WINBOAT: Error starting container', {
                error: e.message,
                stack: e.stack,
                runtime: this._containerRuntime,
                timestamp: new Date().toISOString()
            });
            debugLog(`Exception in startWinBoat: ${e.message}`, this._settings);
            this._showOperationFeedback('start', false, e.message);
        } finally {
            // Always hide progress and reset button states
            this._hideProgressIndicator();
            this._setButtonStates(false);
        }
    }

    // UI/UX improvements: enhanced stop WinBoat with progress feedback
    async _stopWinBoat() {
        try {
            // Show progress and set button states
            this._showProgressIndicator('Stopping WinBoat...');
            this._setButtonStates(true, 'stop');
            this._showOperationFeedback('stop', true);

            const runtime = await this._getContainerRuntime();
            if (!runtime) {
                this._showOperationFeedback('stop', false, 'No container runtime detected');
                this._notifyUser('WinBoat', 'No container runtime (Docker/Podman) detected.');
                this._setButtonStates(false);
                return;
            }
            
            const success = await this._runCommandAsync(`${runtime} stop WinBoat`);

            if (!success) {
                this._showOperationFeedback('stop', false, 'Container failed to stop');
                this._notifyUser('WinBoat', 'Failed to stop WinBoat container');
                this._setButtonStates(false);
                return;
            }

            // Force immediate status update using GLib timeout
            this._clearStatusUpdateTimeout();
            this._statusUpdateTimeout = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 1000, () => {
                this._statusUpdateTimeout = null;
                this._updateStatus();
                return GLib.SOURCE_REMOVE;
            });
            
        } catch (e) {
            // Enhanced error logging with context
            console.error('WINBOAT: Error stopping container', {
                error: e.message,
                stack: e.stack,
                runtime: this._containerRuntime,
                timestamp: new Date().toISOString()
            });
            this._showOperationFeedback('stop', false, e.message);
        } finally {
            // Always hide progress and reset button states
            this._hideProgressIndicator();
            this._setButtonStates(false);
        }
    }

    // UI/UX improvements: enhanced restart WinBoat with progress feedback
    async _restartWinBoat() {
        try {
            // Show progress and set button states
            this._showProgressIndicator('Restarting WinBoat...');
            this._setButtonStates(true, 'restart');
            this._showOperationFeedback('restart', true);
            
            // Record activity on restart
            this._recordActivity();

            const runtime = await this._getContainerRuntime();
            if (!runtime) {
                this._showOperationFeedback('restart', false, 'No container runtime detected');
                this._notifyUser('WinBoat', 'No container runtime (Docker/Podman) detected.');
                this._setButtonStates(false);
                return;
            }
            
            const success = await this._runCommandAsync(`${runtime} restart WinBoat`);

            if (!success) {
                this._showOperationFeedback('restart', false, 'Container failed to restart');
                this._notifyUser('WinBoat', 'Failed to restart WinBoat container');
                this._setButtonStates(false);
                return;
            }

            this._notifyUser('WinBoat', 'WinBoat container restarted');
            
            // Force immediate status update using GLib timeout
            this._clearStatusUpdateTimeout();
            this._statusUpdateTimeout = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 1000, () => {
                this._statusUpdateTimeout = null;
                this._updateStatus();
                return GLib.SOURCE_REMOVE;
            });
            
        } catch (e) {
            // Enhanced error logging with context
            console.error('WINBOAT: Error restarting container', {
                error: e.message,
                stack: e.stack,
                runtime: this._containerRuntime,
                timestamp: new Date().toISOString()
            });
            this._showOperationFeedback('restart', false, e.message);
        } finally {
            // Always hide progress and reset button states
            this._hideProgressIndicator();
            this._setButtonStates(false);
        }
    }

    async _getContainerUptime() {
        try {
            const runtime = await this._getContainerRuntime();
            if (!runtime)
                return 'Unknown';

            // Get container start time with timeout to prevent hanging
            const [success, stdout] = await this._runCommandAsyncWithOutput(
                `${runtime} inspect WinBoat --format "{{.State.StartedAt}}"`,
                8 // 8 second timeout
            );

            if (!success || !stdout) {
                return 'Unknown';
            }

            const startedAt = stdout.toString().trim();
            const startTime = new Date(startedAt).getTime();
            const now = Date.now();
            const uptimeMs = now - startTime;

            if (uptimeMs < 0 || isNaN(uptimeMs))
                return 'Unknown';

            // Format uptime nicely
            const seconds = Math.floor(uptimeMs / 1000);
            const minutes = Math.floor(seconds / 60);
            const hours = Math.floor(minutes / 60);
            const days = Math.floor(hours / 24);

            if (days > 0) {
                return `${days}d ${hours % 24}h ${minutes % 60}m`;
            } else if (hours > 0) {
                return `${hours}h ${minutes % 60}m`;
            } else if (minutes > 0) {
                return `${minutes}m ${seconds % 60}s`;
            } else {
                return `${seconds}s`;
            }

        } catch (e) {
            // Enhanced error logging with context
            console.error('WINBOAT: Error getting uptime', {
                error: e.message,
                stack: e.stack,
                runtime: this._containerRuntime,
                timestamp: new Date().toISOString()
            });
            return 'Error';
        }
    }

    // Performance optimization: cached stats fetching
    async _getContainerStats() {
        // Check cache first
        const now = Date.now();
        if (this._statsCache.data && 
            (now - this._statsCache.timestamp) < this._statsCache.ttl &&
            this._lastStatus === 'running') {
            debugLog('Using cached stats', this._settings);
            return this._statsCache.data;
        }

        try {
            const runtime = await this._getContainerRuntime();
            if (!runtime) {
                return { cpu: 'Runtime unavailable', memory: 'Runtime unavailable', ports: 'Runtime unavailable' };
            }

            // Get container stats using docker stats --no-stream
            const statsCmd = `${runtime} stats WinBoat --no-stream --format "{{.CPUPerc}}\t{{.MemUsage}}"`;
            const [statsSuccess, statsStdout, statsStderr] = await this._runCommandAsyncWithOutput(statsCmd, 10); // 10 second timeout

            if (!statsSuccess || !statsStdout) {
                console.warn(`WINBOAT: Stats command failed: ${statsStderr ? statsStderr.toString().trim() : 'No output'}`);
                return { cpu: 'Stats unavailable', memory: 'Stats unavailable', ports: 'Stats unavailable' };
            }

            const statsOutput = statsStdout.toString().trim();
            debugLog(`Raw stats output: "${statsOutput}"`, this._settings);
            
            if (!statsOutput) {
                return { cpu: 'No data', memory: 'No data', ports: 'No data' };
            }

            // Parse stats
            const parts = statsOutput.split(/\s+/).filter(part => part.length > 0);
            
            if (parts.length < 2) {
                console.warn(`WINBOAT: Stats parse error - not enough parts: ${parts.length}`);
                return { cpu: 'Parse error', memory: 'Parse error', ports: 'Parse error' };
            }

            // First part should be CPU percentage, second should be memory usage
            let cpuPercent = parts[0];
            const memUsage = parts[1];

            // Validate CPU format (should end with %)
            if (!cpuPercent.includes('%')) {
                console.warn(`WINBOAT: Invalid CPU format: ${cpuPercent}`);
                return { cpu: 'Invalid CPU', memory: memUsage, ports: 'Ports unavailable' };
            }

            // Extract numeric value and format properly
            const cpuValue = parseFloat(cpuPercent.replace('%', ''));
            
            // Handle Docker's multi-core CPU reporting
            if (isNaN(cpuValue)) {
                cpuPercent = '0%';
            } else {
                cpuPercent = `${cpuValue.toFixed(1)}%`;
            }

            // Extract only the used memory from "used / total" format
            let usedMemory = memUsage;
            if (memUsage.includes('/')) {
                usedMemory = memUsage.split('/')[0].trim();
            }

            // Get ports separately
            const ports = await this._getContainerPorts();

            debugLog(`Parsed stats - CPU: ${cpuPercent}, Memory: ${usedMemory}, Ports: ${ports}`, this._settings);

            const result = { 
                cpu: cpuPercent, 
                memory: usedMemory,
                ports: ports
            };

            // Cache the result
            this._statsCache.data = result;
            this._statsCache.timestamp = now;

            return result;

        } catch (e) {
            // Enhanced error logging with context
            console.error('WINBOAT: Error getting stats', {
                error: e.message,
                stack: e.stack,
                runtime: this._containerRuntime,
                timestamp: new Date().toISOString()
            });
            return { cpu: 'Error', memory: 'Error', ports: 'Error' };
        }
    }

    async _getContainerPorts() {
        try {
            const runtime = await this._getContainerRuntime();
            if (!runtime) {
                return 'Runtime unavailable';
            }

            // Get container ports (filter by container name) with timeout
            const cmd = `${runtime} ps --filter "name=WinBoat" --format "{{.Ports}}"`;
            const [success, stdout, stderr] = await this._runCommandAsyncWithOutput(cmd, 8); // 8 second timeout

            if (!success || !stdout) {
                console.warn(`WINBOAT: Ports command failed: ${stderr ? stderr.toString().trim() : 'No output'}`);
                return 'Ports unavailable';
            }

            const portsOutput = stdout.toString().trim();
            debugLog(`Raw ports output: "${portsOutput}"`, this._settings);
            
            if (!portsOutput || portsOutput === '<nil>' || portsOutput === '') {
                return 'No ports exposed';
            }

            // Parse ports and format nicely
            const portMappings = portsOutput.split(',').map(port => port.trim()).filter(port => port.length > 0);
            
            if (portMappings.length === 0) {
                return 'No ports exposed';
            }

            // Extract just the external ports for cleaner display
            const externalPorts = portMappings.map(mapping => {
                const match = mapping.match(/:(\d+)->/);
                return match ? match[1] : mapping.split('->')[0];
            }).filter(port => port && port !== '');

            // Remove duplicates
            const uniquePorts = [...new Set(externalPorts)];

            if (uniquePorts.length === 0) {
                return 'Internal only';
            }

            // Format ports - clean and simple
            const portString = uniquePorts.join(', ');
            
            // If port string is too long, split into two lines
            if (portString.length > 30) {
                const midPoint = Math.ceil(uniquePorts.length / 2);
                const firstLine = uniquePorts.slice(0, midPoint).join(', ');
                const secondLine = uniquePorts.slice(midPoint).join(', ');
                return `${firstLine}\n${secondLine}`;
            }

            return portString;

        } catch (e) {
            // Enhanced error logging with context
            console.error('WINBOAT: Error getting ports', {
                error: e.message,
                stack: e.stack,
                runtime: this._containerRuntime,
                timestamp: new Date().toISOString()
            });
            return 'Error';
        }
    }

    _formatTimestamp(timestamp) {
        if (!timestamp) return '';
        
        const now = Date.now();
        const diff = now - timestamp;
        
        if (diff < 60000) { // Less than 1 minute
            return ' (just now)';
        } else if (diff < 3600000) { // Less than 1 hour
            const minutes = Math.floor(diff / 60000);
            return ` (${minutes}m ago)`;
        } else {
            const hours = Math.floor(diff / 3600000);
            return ` (${hours}h ago)`;
        }
    }

    // UI/UX improvements: enhanced stats update with built-in GNOME styles
    async _updateStats() {
        try {
            // Only update stats if container is running
            if (this._lastStatus !== 'running') {
                debugLog('Skipping stats update - container is not running', this._settings);
                return;
            }
            
            debugLog('Updating stats on user request...', this._settings);
            
            // Show loading state using built-in GNOME styles
            this._cpuItem.label.text = 'CPU: Loading...';
            this._memoryItem.label.text = 'Memory: Loading...';
            this._portsItem.label.text = 'Ports: Loading...';
            
            // Use built-in GNOME Shell styles for loading state
            this._cpuItem.add_style_class_name('popup-subtitle');
            this._memoryItem.add_style_class_name('popup-subtitle');
            this._portsItem.add_style_class_name('popup-subtitle');
            
            // Update stats with a small delay to prevent menu from closing
            this._clearStatsUpdateTimeout();
            this._statsUpdateTimeout = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 50, () => {
                this._statsUpdateTimeout = null;
                this._performStatsUpdate();
                return GLib.SOURCE_REMOVE;
            });
            
        } catch (e) {
            // Enhanced error logging with context
            console.error('WINBOAT: Error updating stats', {
                error: e.message,
                stack: e.stack,
                timestamp: new Date().toISOString()
            });
            this._cpuItem.label.text = 'CPU: Error';
            this._memoryItem.label.text = 'Memory: Error';
            this._portsItem.label.text = 'Ports: Error';
            
            // Remove loading style classes
            this._cpuItem.remove_style_class_name('popup-subtitle');
            this._memoryItem.remove_style_class_name('popup-subtitle');
            this._portsItem.remove_style_class_name('popup-subtitle');
        }
    }

    // UI/UX improvements: enhanced perform stats update with built-in GNOME styles
    async _performStatsUpdate() {
        try {
            // Get stats
            const stats = await this._getContainerStats();
            
            // Store last stats for persistence
            this._lastStats = {
                cpu: stats.cpu,
                memory: stats.memory,
                ports: stats.ports,
                timestamp: Date.now()
            };
            
            // Update display using standard GNOME styling
            this._cpuItem.label.text = `CPU: ${stats.cpu}`;
            this._memoryItem.label.text = `Memory: ${stats.memory}`;
            this._portsItem.label.text = `Ports: ${stats.ports}`;
            
            // Remove loading style classes - no custom animations needed
            this._cpuItem.remove_style_class_name('popup-subtitle');
            this._memoryItem.remove_style_class_name('popup-subtitle');
            this._portsItem.remove_style_class_name('popup-subtitle');
            
            debugLog(`Stats updated - CPU: ${stats.cpu}, Memory: ${stats.memory}, Ports: ${stats.ports}`, this._settings);
            
        } catch (e) {
            // Enhanced error logging with context
            console.error('WINBOAT: Error updating stats', {
                error: e.message,
                stack: e.stack,
                timestamp: new Date().toISOString()
            });
            this._cpuItem.label.text = 'CPU: Error';
            this._memoryItem.label.text = 'Memory: Error';
            this._portsItem.label.text = 'Ports: Error';
            
            // Remove loading style classes
            this._cpuItem.remove_style_class_name('popup-subtitle');
            this._memoryItem.remove_style_class_name('popup-subtitle');
            this._portsItem.remove_style_class_name('popup-subtitle');
        }
    }

    _openWinBoatApplication() {
        try {
            // Record activity when user opens WinBoat interface
            this._recordActivity();
            
            const windowActors = global.get_window_actors();
            let winBoatWindow = null;
            
            for (const actor of windowActors) {
                const window = actor.get_meta_window();
                const appId = window.get_gtk_application_id();
                const title = window.get_title();
                const wmClass = window.get_wm_class();
                
                if (appId && (appId === 'winboat' || appId === 'WinBoat')) {
                    winBoatWindow = window;
                    break;
                }
                
                if (wmClass && (wmClass === 'winboat' || wmClass === 'WinBoat')) {
                    winBoatWindow = window;
                    break;
                }
                
                if (title && (title === 'winboat' || title === 'WinBoat')) {
                    winBoatWindow = window;
                    break;
                }
            }
            
            if (winBoatWindow) {
                winBoatWindow.activate(global.get_current_time());
                console.info('WINBOAT: Focused WinBoat window');
            } else {
                this._launchWinBoatApp();
            }
            
        } catch (e) {
            // Enhanced error logging with context
            console.error('WINBOAT: Error managing WinBoat application', {
                error: e.message,
                stack: e.stack,
                timestamp: new Date().toISOString()
            });
            this._notifyUser('WinBoat', 'Failed to open WinBoat application');
        }
    }

    _launchWinBoatApp() {
        try {
            const appInfo = Gio.AppInfo.get_default_for_type('x-scheme-handler/winboat', false) ||
                          Gio.AppInfo.get_default_for_type('application/x-winboat', false);
            
            if (appInfo) {
                appInfo.launch([], null);
                console.info('WINBOAT: Launched WinBoat application');
            } else {
                // Fallback: try to launch by common executable names with validation
                const commonNames = ['winboat', 'WinBoat', 'winboat-desktop'];
                let launched = false;
                
                for (const appName of commonNames) {
                    try {
                        // Validate app name for security
                        if (!/^[a-zA-Z0-9_-]+$/.test(appName)) {
                            console.warn(`WINBOAT: Invalid app name format: ${appName}`);
                            continue;
                        }
                        
                        const proc = Gio.Subprocess.new(
                            [appName],
                            Gio.SubprocessFlags.NONE
                        );
                        launched = true;
                        console.info(`WINBOAT: Launched ${appName}`);
                        break;
                    } catch (e) {
                        // Try next name
                        debugLog(`Failed to launch ${appName}: ${e.message}`, this._settings);
                    }
                }
                
                if (!launched) {
                    this._notifyUser('WinBoat', 'WinBoat application not found. Please install WinBoat first.');
                }
            }
        } catch (e) {
            // Enhanced error logging with context
            console.error('WINBOAT: Error launching WinBoat application', {
                error: e.message,
                stack: e.stack,
                timestamp: new Date().toISOString()
            });
            this._notifyUser('WinBoat', 'Failed to launch WinBoat application');
        }
    }

    // Performance optimization: optimized auto-shutdown settings with cache management
    _updateAutoShutdownSettings() {
        const enabled = this._getCachedSetting('auto-shutdown-enabled');
        const hours = this._getCachedSetting('auto-shutdown-hours');
        
        debugLog(`Auto-shutdown settings updated: enabled=${enabled}, hours=${hours}`, this._settings);
        
        this._autoShutdownEnabled = enabled;
        
        if (enabled) {
            this._startAutoShutdownTimer();
        } else {
            this._clearAutoShutdownTimers();
            this._updateCountdownDisplay();
        }
    }

    _startAutoShutdownTimer() {
        this._clearAutoShutdownTimers();
        
        if (!this._autoShutdownEnabled || this._lastStatus !== 'running') {
            this._updateCountdownDisplay();
            return;
        }
        
        // Get or initialize last activity time
        let lastActivity = this._settings.get_int64('last-activity-timestamp');
        if (lastActivity === 0) {
            lastActivity = Math.floor(Date.now() / 1000);
            this._settings.set_int64('last-activity-timestamp', lastActivity);
        }
        
        this._lastActivityTime = lastActivity;
        debugLog(`Starting auto-shutdown timer with last activity: ${new Date(lastActivity * 1000).toLocaleString()}`, this._settings);
        
        this._scheduleAutoShutdownCheck();
        this._scheduleCountdownUpdate();
    }

    _scheduleAutoShutdownCheck() {
        if (this._autoShutdownTimer) {
            GLib.Source.remove(this._autoShutdownTimer);
        }
        
        // Check every minute
        this._autoShutdownTimer = GLib.timeout_add_seconds(
            GLib.PRIORITY_DEFAULT,
            60, // Check every minute
            () => {
                this._checkAutoShutdown();
                return GLib.SOURCE_CONTINUE;
            }
        );
    }

    _scheduleCountdownUpdate() {
        if (this._countdownTimer) {
            GLib.Source.remove(this._countdownTimer);
        }
        
        // Update countdown every 30 seconds (good balance between UX and performance)
        this._countdownTimer = GLib.timeout_add_seconds(
            GLib.PRIORITY_DEFAULT,
            30,
            () => {
                this._updateCountdownDisplay();
                return GLib.SOURCE_CONTINUE;
            }
        );
    }

    async _performAutoShutdown() {
        // Double-check that auto-shutdown is still enabled before proceeding
        if (!this._autoShutdownEnabled) {
            debugLog('Auto-shutdown is disabled, aborting shutdown', this._settings);
            return;
        }
        
        debugLog('Performing auto-shutdown', this._settings);
        
        // Show notification
        const hours = this._settings.get_int('auto-shutdown-hours');
        this._notifyUser('WinBoat', `WinBoat container automatically stopped after ${hours} hours of inactivity.`);
        this._dismissWarningNotification();
        
        // Stop the container and wait for it to complete
        try {
            await this._stopWinBoat();
            debugLog('Auto-shutdown completed successfully', this._settings);
        } catch (e) {
            // Enhanced error logging with context
            console.error('WINBOAT: Auto-shutdown stop command failed', {
                error: e.message,
                stack: e.stack,
                autoShutdownEnabled: this._autoShutdownEnabled,
                hours: hours,
                timestamp: new Date().toISOString()
            });
            // Still reset the timer to prevent repeated attempts
        }
        
        // Reset activity timestamp
        this._settings.set_int64('last-activity-timestamp', 0);
        this._lastActivityTime = null;
        this._warningShown = false;
        this._dismissWarningNotification();
    }

    _showShutdownWarning(secondsRemaining) {
        debugLog(`Showing shutdown warning - ${secondsRemaining} seconds remaining`, this._settings);
        
        const minutes = Math.ceil(secondsRemaining / 60);
        const title = 'WinBoat Auto-Shutdown Warning';
        const message = `WinBoat container will shut down in ${minutes} minute${minutes > 1 ? 's' : ''} due to inactivity.`;
        
        // Show notification (persistent)
        this._dismissWarningNotification();
        this._warningNotification = this._notifyUser(title, message, true);
        
        // Also log to console for debug
        console.info(`WINBOAT: ${message}`);
    }

    // Performance optimization: optimized countdown display with cached settings
    _updateCountdownDisplay() {
        if (!this._countdownItem) {
            return;
        }
        
        const compactMode = this._getCachedSetting('compact-mode');
        
        // Show status if auto-shutdown is enabled
        if (!this._autoShutdownEnabled) {
            this._countdownItem.visible = false;
            this._dismissWarningNotification();
            return;
        }
        
        this._countdownItem.visible = true;
        
        // If container is not running, show disabled status
        if (this._lastStatus !== 'running') {
            this._countdownItem.label.text = compactMode ? 'Auto-shutdown: Waiting' : 'Auto-shutdown: Waiting for container';
            return;
        }
        
        const now = Math.floor(Date.now() / 1000);
        const hours = this._getCachedSetting('auto-shutdown-hours');
        const timeoutSeconds = hours * 3600;
        
        if (!this._lastActivityTime) {
            this._countdownItem.label.text = compactMode ? 'Auto-shutdown: --:--:--' : 'Auto-shutdown in: --:--:--';
            return;
        }
        
        const elapsed = now - this._lastActivityTime;
        const remaining = Math.max(0, timeoutSeconds - elapsed);
        
        if (remaining <= 0) {
            this._countdownItem.label.text = compactMode ? 'Auto-shutdown: Soon' : 'Auto-shutdown: Imminent';
        } else {
            const hoursRemaining = Math.floor(remaining / 3600);
            const minutesRemaining = Math.floor((remaining % 3600) / 60);
            const secondsRemaining = remaining % 60;
            
            // Performance optimization: pre-format strings
            const timeString = `${hoursRemaining.toString().padStart(2, '0')}:${minutesRemaining.toString().padStart(2, '0')}:${secondsRemaining.toString().padStart(2, '0')}`;
            this._countdownItem.label.text = compactMode ? `Auto-shutdown: ${timeString}` : `Auto-shutdown in: ${timeString}`;
        }
    }

    _clearAutoShutdownTimers() {
        if (this._autoShutdownTimer) {
            GLib.Source.remove(this._autoShutdownTimer);
            this._autoShutdownTimer = null;
        }
        if (this._countdownTimer) {
            GLib.Source.remove(this._countdownTimer);
            this._countdownTimer = null;
        }
        this._dismissWarningNotification();
    }

    _dismissWarningNotification() {
        if (this._warningNotification && typeof this._warningNotification.destroy === 'function') {
            this._warningNotification.destroy();
        }
        this._warningNotification = null;
    }

    // Performance optimization: enhanced destroy with cache cleanup and UI cleanup using built-in styles
    destroy() {
        // Clean up all timeouts
        this._clearBootCheckTimeout();
        this._clearStatusUpdateTimeout();
        this._clearStatsUpdateTimeout();
        this._clearAutoShutdownTimers();
        
        if (this._timeout) {
            GLib.Source.remove(this._timeout);
            this._timeout = null;
        }
        
        // Reset warning flag
        this._warningShown = false;
        
        // UI/UX improvements: clean up progress indicators and states using built-in styles
        this._hideProgressIndicator();
        this._setButtonStates(false);
        
        // Reset progress indicator state
        if (this._progressIndicator) {
            this._progressIndicator.visible = false;
            this._progressIndicator.remove_style_class_name('popup-subtitle');
        }
        
        // Remove built-in GNOME style classes from menu items
        if (this._cpuItem) {
            this._cpuItem.remove_style_class_name('popup-subtitle');
            this._cpuItem.remove_style_class_name('popup-inactive-menu-item');
        }
        if (this._memoryItem) {
            this._memoryItem.remove_style_class_name('popup-subtitle');
            this._memoryItem.remove_style_class_name('popup-inactive-menu-item');
        }
        if (this._portsItem) {
            this._portsItem.remove_style_class_name('popup-subtitle');
            this._portsItem.remove_style_class_name('popup-inactive-menu-item');
        }
        
        // Performance optimization: clear caches
        this._invalidateSettingsCache();
        this._statsCache.data = null;
        this._statsCache.timestamp = 0;
        this._runtimeCache.runtime = null;
        this._runtimeCache.timestamp = 0;
        
        // Disconnect all signals
        if (this._refreshSignalId) {
            this._settings.disconnect(this._refreshSignalId);
            this._refreshSignalId = null;
        }
        if (this._showUptimeSignalId) {
            this._settings.disconnect(this._showUptimeSignalId);
            this._showUptimeSignalId = null;
        }
        if (this._showStatsSignalId) {
            this._settings.disconnect(this._showStatsSignalId);
            this._showStatsSignalId = null;
        }
        if (this._compactModeSignalId) {
            this._settings.disconnect(this._compactModeSignalId);
            this._compactModeSignalId = null;
        }
        if (this._autoShutdownEnabledSignalId) {
            this._settings.disconnect(this._autoShutdownEnabledSignalId);
            this._autoShutdownEnabledSignalId = null;
        }
        if (this._autoShutdownHoursSignalId) {
            this._settings.disconnect(this._autoShutdownHoursSignalId);
            this._autoShutdownHoursSignalId = null;
        }
        if (this._autoShutdownWarningSignalId) {
            this._settings.disconnect(this._autoShutdownWarningSignalId);
            this._autoShutdownWarningSignalId = null;
        }
        
        // Call parent destroy
        super.destroy();
    }
}