/*!
 * Copyright (C) 2023 Lju
 *
 * This file is part of Astra Monitor extension for GNOME Shell.
 * [https://github.com/AstraExt/astra-monitor]
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */
import GLib from 'gi://GLib';
import St from 'gi://St';
import Clutter from 'gi://Clutter';
import Shell from 'gi://Shell';
import { gettext as _, pgettext } from 'resource:///org/gnome/shell/extensions/extension.js';
import Grid from '../grid.js';
import Utils from '../utils/utils.js';
import Config from '../config.js';
import MenuBase from '../menu.js';
import StorageBars from './storageBars.js';
import StorageGraph from './storageGraph.js';
import StorageMonitor from './storageMonitor.js';
export default class StorageMenu extends MenuBase {
    constructor(sourceActor, arrowAlignment, arrowSide) {
        super(sourceActor, arrowAlignment, { name: 'Storage Menu', arrowSide });
        this.privilegedTopProcesses = false;
        this.updateTimer = 0;
        Utils.verbose('Initializing storage menu');
        this.addMenuSection(_('Storage'));
        this.createActivitySection();
        this.addTopProcesses();
        this.createDeviceList();
        this.addUtilityButtons();
        this.setStyle();
        Config.connect(this, 'changed::theme-style', this.setStyle.bind(this));
    }
    setStyle() {
        const lightTheme = Utils.themeStyle === 'light';
        const styleClass = lightTheme ? 'astra-monitor-menu-key-light' : 'astra-monitor-menu-key';
        this.totalReadSpeedValueLabel.styleClass = styleClass;
        this.totalWriteSpeedValueLabel.styleClass = styleClass;
    }
    createActivitySection() {
        const defaultStyle = '';
        const hoverButton = new St.Button({
            reactive: true,
            trackHover: true,
            style: defaultStyle,
        });
        const grid = new Grid({ styleClass: 'astra-monitor-menu-subgrid' });
        hoverButton.set_child(grid);
        this.graph = new StorageGraph({
            width: 200 - 2 - 15,
            mini: false,
        });
        grid.addToGrid(this.graph, 2);
        const totalReadSpeedLabel = new St.Label({
            text: _('Global Read:'),
            xExpand: true,
            styleClass: 'astra-monitor-menu-label',
            style: 'margin-top:0.25em;',
        });
        grid.addToGrid(totalReadSpeedLabel);
        this.totalReadSpeedValueLabel = new St.Label({
            text: '-',
            xExpand: true,
        });
        grid.addToGrid(this.totalReadSpeedValueLabel);
        const totalWriteSpeedLabel = new St.Label({
            text: _('Global Write:'),
            xExpand: true,
            styleClass: 'astra-monitor-menu-label',
        });
        grid.addToGrid(totalWriteSpeedLabel);
        this.totalWriteSpeedValueLabel = new St.Label({
            text: '-',
            xExpand: true,
        });
        grid.addToGrid(this.totalWriteSpeedValueLabel);
        this.createActivityPopup(hoverButton);
        hoverButton.connect('enter-event', () => {
            hoverButton.style = defaultStyle + this.selectionStyle;
            if (this.storageActivityPopup)
                this.storageActivityPopup.open(true);
        });
        hoverButton.connect('leave-event', () => {
            hoverButton.style = defaultStyle;
            if (this.storageActivityPopup)
                this.storageActivityPopup.close(true);
        });
        this.addToMenu(hoverButton, 2);
    }
    createActivityPopup(sourceActor) {
        this.storageActivityPopup = new MenuBase(sourceActor, 0.05, { numCols: 2 });
        this.storageActivityPopup.addMenuSection(_('Total Activity'));
        this.storageActivityPopup.addToMenu(new St.Label({
            text: _('Read'),
            styleClass: 'astra-monitor-menu-sub-key',
        }));
        const totalReadValueLabel = new St.Label({ text: '', style: 'text-align:left;' });
        this.storageActivityPopup.addToMenu(totalReadValueLabel);
        this.storageActivityPopup.totalReadValueLabel = totalReadValueLabel;
        this.storageActivityPopup.addToMenu(new St.Label({
            text: _('Write'),
            styleClass: 'astra-monitor-menu-sub-key',
        }));
        const totalWriteValueLabel = new St.Label({ text: '', style: 'text-align:left;' });
        this.storageActivityPopup.addToMenu(totalWriteValueLabel);
        this.storageActivityPopup.totalWriteValueLabel = totalWriteValueLabel;
    }
    addTopProcesses() {
        const separator = this.addMenuSection(_('Top processes') + ` (${GLib.get_user_name()})`, false, false);
        separator.style = 'margin-bottom:0;padding-bottom:0;';
        separator.xExpand = true;
        separator.hide();
        const subSeparator = new St.Label({
            text: _('(click to show all system processes)'),
            styleClass: 'astra-monitor-menu-key-mid-center',
            xExpand: true,
        });
        subSeparator.hide();
        const separatorGrid = new Grid({ numCols: 1, styleClass: 'astra-monitor-menu-subgrid' });
        separatorGrid.addToGrid(separator);
        separatorGrid.addToGrid(subSeparator);
        const separatorButton = new St.Button({
            reactive: true,
            trackHover: true,
        });
        separatorButton.set_child(separatorGrid);
        separatorButton.connect('clicked', () => {
            if (Utils.hasIotop()) {
                Utils.storageMonitor.startIOTop();
            }
        });
        this.addToMenu(separatorButton, 2);
        const defaultStyle = '';
        const hoverButton = new St.Button({
            reactive: true,
            trackHover: true,
            style: defaultStyle,
        });
        hoverButton.hide();
        const grid = new Grid({ numCols: 3, styleClass: 'astra-monitor-menu-subgrid' });
        const labels = [];
        const numProcesses = 3;
        for (let i = 0; i < numProcesses; i++) {
            const label = new St.Label({
                text: '',
                styleClass: 'astra-monitor-menu-cmd-name',
                style: 'max-width:85px;',
                xExpand: true,
            });
            grid.addToGrid(label);
            const readContainer = new St.Widget({
                layoutManager: new Clutter.GridLayout({
                    orientation: Clutter.Orientation.HORIZONTAL,
                }),
                style: 'margin-left:0;margin-right:0;width:5.5em;',
            });
            const readActivityIcon = new St.Icon({
                gicon: Utils.getLocalIcon('am-up-symbolic'),
                fallbackIconName: 'go-up-symbolic',
                styleClass: 'astra-monitor-menu-icon-mini',
                style: 'color:rgba(255,255,255,0.5);',
            });
            readContainer.add_child(readActivityIcon);
            const readValue = new St.Label({
                text: '',
                styleClass: 'astra-monitor-menu-cmd-usage',
                xExpand: true,
            });
            readContainer.add_child(readValue);
            grid.addToGrid(readContainer);
            const writeContainer = new St.Widget({
                layoutManager: new Clutter.GridLayout({
                    orientation: Clutter.Orientation.HORIZONTAL,
                }),
                style: 'margin-left:0;margin-right:0;width:5.5em;',
            });
            const writeActivityIcon = new St.Icon({
                gicon: Utils.getLocalIcon('am-down-symbolic'),
                fallbackIconName: 'go-down-symbolic',
                styleClass: 'astra-monitor-menu-icon-mini',
                style: 'color:rgba(255,255,255,0.5);',
            });
            writeContainer.add_child(writeActivityIcon);
            const writeValue = new St.Label({
                text: '',
                styleClass: 'astra-monitor-menu-cmd-usage',
                xExpand: true,
            });
            writeContainer.add_child(writeValue);
            grid.addToGrid(writeContainer);
            labels.push({
                label,
                read: {
                    container: readContainer,
                    value: readValue,
                    icon: readActivityIcon,
                },
                write: {
                    container: writeContainer,
                    value: writeValue,
                    icon: writeActivityIcon,
                },
            });
        }
        hoverButton.set_child(grid);
        this.createTopProcessesPopup(hoverButton);
        hoverButton.connect('enter-event', () => {
            hoverButton.style = defaultStyle + this.selectionStyle;
            if (this.topProcessesPopup)
                this.topProcessesPopup.open(true);
        });
        hoverButton.connect('leave-event', () => {
            hoverButton.style = defaultStyle;
            if (this.topProcessesPopup)
                this.topProcessesPopup.close(true);
        });
        this.addToMenu(hoverButton, 2);
        this.topProcesses = {
            separator,
            subSeparator,
            labels,
            hoverButton,
        };
    }
    createTopProcessesPopup(sourceActor) {
        this.topProcessesPopup = new MenuBase(sourceActor, 0.05);
        this.topProcessesPopup.section = this.topProcessesPopup.addMenuSection(_('Top processes') + ` (${GLib.get_user_name()})`);
        this.topProcessesPopup.section.style = 'min-width:500px;';
        this.topProcessesPopup.processes = new Map();
        const grid = new Grid({
            xExpand: true,
            xAlign: Clutter.ActorAlign.START,
            numCols: 2,
            styleClass: 'astra-monitor-menu-subgrid',
        });
        for (let i = 0; i < StorageMonitor.TOP_PROCESSES_LIMIT; i++) {
            const readContainer = new St.Widget({
                layoutManager: new Clutter.GridLayout({
                    orientation: Clutter.Orientation.HORIZONTAL,
                }),
                style: 'margin-left:0;margin-right:0;width:6em;',
            });
            const readActivityIcon = new St.Icon({
                gicon: Utils.getLocalIcon('am-up-symbolic'),
                fallbackIconName: 'go-up-symbolic',
                styleClass: 'astra-monitor-menu-icon-mini',
                style: 'color:rgba(255,255,255,0.5);',
            });
            readContainer.add_child(readActivityIcon);
            const readValue = new St.Label({
                text: '-',
                styleClass: 'astra-monitor-menu-cmd-usage',
                xExpand: true,
            });
            readContainer.add_child(readValue);
            grid.addGrid(readContainer, 0, i * 2, 1, 1);
            const writeContainer = new St.Widget({
                layoutManager: new Clutter.GridLayout({
                    orientation: Clutter.Orientation.HORIZONTAL,
                }),
                style: 'margin-left:0;margin-right:0;width:6em;',
            });
            const writeActivityIcon = new St.Icon({
                gicon: Utils.getLocalIcon('am-down-symbolic'),
                fallbackIconName: 'go-down-symbolic',
                styleClass: 'astra-monitor-menu-icon-mini',
                style: 'color:rgba(255,255,255,0.5);',
            });
            writeContainer.add_child(writeActivityIcon);
            const writeValue = new St.Label({
                text: '-',
                styleClass: 'astra-monitor-menu-cmd-usage',
                xExpand: true,
            });
            writeContainer.add_child(writeValue);
            grid.addGrid(writeContainer, 0, i * 2 + 1, 1, 1);
            const label = new St.Label({
                text: '-',
                styleClass: 'astra-monitor-menu-cmd-name-full',
            });
            grid.addGrid(label, 1, i * 2, 1, 1);
            const description = new St.Label({
                text: '-',
                styleClass: 'astra-monitor-menu-cmd-description',
            });
            grid.addGrid(description, 1, i * 2 + 1, 1, 1);
            this.topProcessesPopup.processes.set(i, {
                label,
                description,
                read: {
                    container: readContainer,
                    value: readValue,
                    icon: readActivityIcon,
                },
                write: {
                    container: writeContainer,
                    value: writeValue,
                    icon: writeActivityIcon,
                },
            });
        }
        this.topProcessesPopup.addToMenu(grid, 2);
    }
    createDeviceList() {
        if (this.deviceSection === undefined) {
            this.addMenuSection(_('Devices'));
            this.deviceSection = new Grid({ styleClass: 'astra-monitor-menu-subgrid' });
            this.noDevicesLabel = new St.Label({
                text: _('No storage device found'),
                styleClass: 'astra-monitor-menu-label-warning',
                style: 'font-style:italic;',
            });
            this.deviceSection.addToGrid(this.noDevicesLabel, 2);
            this.devices = new Map();
            this.devicesInfoPopup = new Map();
            this.devicesTotalsPopup = new Map();
            this.addToMenu(this.deviceSection, 2);
            Config.connect(this, 'changed::storage-ignored', this.updateDeviceList.bind(this));
            Config.connect(this, 'changed::storage-ignored-regex', this.updateDeviceList.bind(this));
        }
    }
    async updateDeviceList() {
        const devices = await Utils.getBlockDevicesAsync();
        if (devices.size > 0)
            this.noDevicesLabel.hide();
        else
            this.noDevicesLabel.show();
        const ignoredDevices = Config.get_json('storage-ignored');
        if (ignoredDevices && Array.isArray(ignoredDevices) && ignoredDevices.length > 0) {
            for (const kname of ignoredDevices) {
                for (const [id, device] of devices.entries()) {
                    if (device.kname === kname) {
                        devices.delete(id);
                        break;
                    }
                }
            }
        }
        const ignoredRegex = Config.get_string('storage-ignored-regex');
        if (ignoredRegex) {
            try {
                const regex = new RegExp(`^${ignoredRegex}$`, 'i');
                for (const [id, device] of devices.entries()) {
                    if (regex.test(device.kname))
                        devices.delete(id);
                }
            }
            catch (e) {
            }
        }
        for (const [id, device] of this.devices.entries()) {
            if (!devices.has(id)) {
                this.deviceSection.remove_child(device.container);
                this.devices.delete(id);
                device.bar?.destroy();
                this.devicesInfoPopup.get(id)?.close(true);
                this.devicesInfoPopup.get(id)?.destroy();
                this.devicesInfoPopup.delete(id);
                this.devicesTotalsPopup.get(id)?.close(true);
                this.devicesTotalsPopup.get(id)?.destroy();
                this.devicesTotalsPopup.delete(id);
            }
        }
        const idList = Array.from(devices.keys());
        const mainDisk = Config.get_string('storage-main');
        if (mainDisk) {
            const mainDiskIndex = idList.indexOf(mainDisk);
            if (mainDiskIndex > 0) {
                idList.splice(mainDiskIndex, 1);
                idList.unshift(mainDisk);
            }
        }
        for (const id of idList) {
            const deviceData = devices.get(id);
            let device;
            let infoPopup;
            let totalsPopup;
            if (!this.devices.has(id)) {
                device = this.createBlockDevice(id);
                this.deviceSection.addToGrid(device.container, 2);
                this.devices.set(id, device);
            }
            else {
                device = this.devices.get(id);
            }
            if (!device)
                continue;
            if (!this.devicesInfoPopup.has(id)) {
                infoPopup = this.createDeviceInfoPopup(device.container);
                this.devicesInfoPopup.set(id, infoPopup);
            }
            else {
                infoPopup = this.devicesInfoPopup.get(id);
            }
            if (!infoPopup)
                continue;
            if (!this.devicesTotalsPopup.has(id)) {
                totalsPopup = this.createDeviceTotalsPopup(device.container);
                this.devicesTotalsPopup.set(id, totalsPopup);
            }
            else {
                totalsPopup = this.devicesTotalsPopup.get(id);
            }
            if (!totalsPopup)
                continue;
            if (!deviceData)
                continue;
            try {
                this.updateBlockDevice(device, deviceData);
            }
            catch (e) {
                Utils.error('Error updating storage device info', e);
            }
        }
    }
    createBlockDevice(id) {
        const container = new Grid({
            xExpand: true,
            styleClass: 'astra-monitor-menu-subgrid',
            style: 'padding-top:0.3em;margin-bottom:0.3em;',
        });
        const topInfoGrid = new Grid({ styleClass: 'astra-monitor-menu-subgrid' });
        const topInfoButton = new St.Button({
            reactive: true,
            trackHover: true,
            style: '',
        });
        topInfoButton.set_child(topInfoGrid);
        const headerGrid = new Grid({
            numCols: 3,
            styleClass: 'astra-monitor-menu-subgrid',
        });
        const icon = new St.Icon({
            styleClass: 'astra-monitor-menu-icon',
            style: 'padding-left:0.25em;',
        });
        headerGrid.addToGrid(icon);
        const label = new St.Label({
            text: '',
            styleClass: 'astra-monitor-menu-label',
        });
        headerGrid.addToGrid(label);
        const name = new St.Label({
            text: '',
            xExpand: true,
            styleClass: 'astra-monitor-menu-key-mid',
        });
        headerGrid.addToGrid(name);
        topInfoGrid.addToGrid(headerGrid, 2);
        const barGrid = new St.Widget({
            layoutManager: new Clutter.GridLayout({ orientation: Clutter.Orientation.VERTICAL }),
            style: 'margin-left:0;',
        });
        const sizeLabel = new St.Label({
            text: '',
            yAlign: Clutter.ActorAlign.CENTER,
            styleClass: 'astra-monitor-menu-key-mid',
        });
        barGrid.layoutManager.attach(sizeLabel, 0, 0, 1, 1);
        const bar = new StorageBars({
            numBars: 1,
            width: 160 - 2 - 4,
            height: 0.5,
            mini: false,
            layout: 'horizontal',
            xAlign: Clutter.ActorAlign.START,
            style: 'margin-left:0;margin-bottom:0;margin-right:0;border:solid 1px #555;',
        });
        barGrid.layoutManager.attach(bar, 1, 0, 1, 1);
        const barLabel = new St.Label({
            text: '0%',
            yAlign: Clutter.ActorAlign.CENTER,
            style: 'width:2.7em;font-size:0.8em;text-align:right;margin-right:0.25em;margin-top:0.2em;',
        });
        barGrid.layoutManager.attach(barLabel, 2, 0, 1, 1);
        topInfoGrid.addToGrid(barGrid, 2);
        container.addToGrid(topInfoButton, 2);
        topInfoButton.connect('enter-event', () => {
            topInfoButton.style = this.selectionStyle;
            const popup = this.devicesInfoPopup.get(id);
            if (popup?.empty === false)
                popup?.open(true);
        });
        topInfoButton.connect('leave-event', () => {
            topInfoButton.style = '';
            const popup = this.devicesInfoPopup.get(id);
            popup?.close(true);
        });
        const rwButton = new St.Button({
            reactive: true,
            trackHover: true,
            xExpand: true,
            style: '',
        });
        const rwContainer = new St.Widget({
            layoutManager: new Clutter.GridLayout({ orientation: Clutter.Orientation.HORIZONTAL }),
            xExpand: true,
            style: 'margin-left:0;margin-right:0;',
        });
        rwButton.set_child(rwContainer);
        const readContainer = new St.Widget({
            layoutManager: new Clutter.GridLayout({ orientation: Clutter.Orientation.HORIZONTAL }),
            xExpand: true,
            style: 'margin-left:0;margin-right:0;',
        });
        const readLabel = new St.Label({
            text: pgettext('short for read', 'R'),
            styleClass: 'astra-monitor-menu-label',
            style: 'padding-right:0.15em;',
        });
        readContainer.add_child(readLabel);
        const readActivityIcon = new St.Icon({
            gicon: Utils.getLocalIcon('am-up-symbolic'),
            fallbackIconName: 'go-up-symbolic',
            styleClass: 'astra-monitor-menu-icon-mini',
            style: 'color:rgba(255,255,255,0.5);',
        });
        readContainer.add_child(readActivityIcon);
        const readValueLabel = new St.Label({
            text: '-',
            xExpand: true,
            styleClass: 'astra-monitor-menu-key-mid',
        });
        readContainer.add_child(readValueLabel);
        readContainer.set_width(100);
        rwContainer.add_child(readContainer);
        const writeContainer = new St.Widget({
            layoutManager: new Clutter.GridLayout({ orientation: Clutter.Orientation.HORIZONTAL }),
            xExpand: true,
            style: 'margin-left:0;margin-right:0;',
        });
        const writeLabel = new St.Label({
            text: pgettext('short for write', 'W'),
            styleClass: 'astra-monitor-menu-label',
            style: 'padding-right:0.15em;',
        });
        writeContainer.add_child(writeLabel);
        const writeActivityIcon = new St.Icon({
            gicon: Utils.getLocalIcon('am-down-symbolic'),
            fallbackIconName: 'go-down-symbolic',
            styleClass: 'astra-monitor-menu-icon-mini',
            style: 'color:rgba(255,255,255,0.5);',
        });
        writeContainer.add_child(writeActivityIcon);
        const writeValueLabel = new St.Label({
            text: '-',
            xExpand: true,
            styleClass: 'astra-monitor-menu-key-mid',
        });
        writeContainer.add_child(writeValueLabel);
        writeContainer.set_width(100);
        rwContainer.add_child(writeContainer);
        rwButton.connect('enter-event', () => {
            rwButton.style = this.selectionStyle;
            const popup = this.devicesTotalsPopup.get(id);
            popup?.open(true);
        });
        rwButton.connect('leave-event', () => {
            rwButton.style = '';
            const popup = this.devicesTotalsPopup.get(id);
            popup?.close(true);
        });
        container.addToGrid(rwButton, 2);
        return {
            data: null,
            container,
            icon,
            label,
            name,
            barGrid,
            bar,
            barLabel,
            sizeLabel,
            readValueLabel,
            readActivityIcon,
            writeValueLabel,
            writeActivityIcon,
        };
    }
    static get deviceInfoPopupConfiguration() {
        return [
            {
                title: _('Basic Info'),
                labels: 'labelsS1',
                values: 'valuesS1',
                sectionNr: 'section1',
                fields: [
                    { key: 'name', label: _('Name') },
                    { key: 'type', label: _('Type') },
                    { key: 'model', label: _('Model'), parent: true },
                    { key: 'vendor', label: _('Vendor'), parent: true },
                    { key: 'serial', label: _('Serial'), parent: true },
                    { key: 'size', label: _('Size'), formatAsBytes: true },
                    { key: 'state', label: _('State'), parent: true },
                    { key: 'subsystems', label: _('Subsystems') },
                ],
            },
            {
                title: _('File System and Mounting Info'),
                labels: 'labelsS2',
                values: 'valuesS2',
                sectionNr: 'section2',
                fields: [
                    { key: 'fstype', label: _('File System Type') },
                    { key: 'label', label: _('Label') },
                    { key: 'uuid', label: _('UUID') },
                    { key: 'mountpoints', label: _('Mount Points') },
                    { key: 'fsavail', label: _('Available Space'), formatAsBytes: true },
                    { key: 'fssize', label: _('File System Size'), formatAsBytes: true },
                    { key: 'fsused', label: _('Used Space'), formatAsBytes: true },
                    { key: 'fsuse%', label: _('Used Space (%)') },
                    { key: 'fsver', label: _('File System Version') },
                    { key: 'fsroots', label: _('File System Roots') },
                ],
            },
            {
                title: _('Physical and Disk Details'),
                labels: 'labelsS3',
                values: 'valuesS3',
                sectionNr: 'section3',
                fields: [
                    { key: 'phy-sec', label: _('Physical Sector Size'), formatAsBytes: true },
                    { key: 'log-sec', label: _('Logical Sector Size'), formatAsBytes: true },
                    { key: 'min-io', label: _('Minimum IO Size'), formatAsBytes: true },
                    { key: 'opt-io', label: _('Optimal IO Size'), formatAsBytes: true },
                    { key: 'rota', label: _('Rotational'), checkNull: true },
                    { key: 'rq-size', label: _('Request Size'), formatAsBytes: true },
                    { key: 'alignment', label: _('Alignment Offset'), formatAsBytes: true },
                    { key: 'disc-aln', label: _('Discard Alignment'), formatAsBytes: true },
                    { key: 'disc-gran', label: _('Discard Granularity'), formatAsBytes: true },
                    { key: 'disc-max', label: _('Discard Max Size'), formatAsBytes: true },
                    { key: 'disc-zero', label: _('Discard Zeroes Data'), checkNull: true },
                ],
            },
            {
                title: _('Partition Info'),
                labels: 'labelsS4',
                values: 'valuesS4',
                sectionNr: 'section4',
                fields: [
                    { key: 'parttype', label: _('Partition Type') },
                    { key: 'partlabel', label: _('Partition Label') },
                    { key: 'partuuid', label: _('Partition UUID') },
                    { key: 'partn', label: _('Partition Number') },
                    { key: 'pttype', label: _('Partition Table Type') },
                    { key: 'ptuuid', label: _('Partition Table UUID') },
                ],
            },
            {
                title: _('Performance and Settings'),
                labels: 'labelsS5',
                values: 'valuesS5',
                sectionNr: 'section5',
                fields: [
                    { key: 'ra', label: _('Read Ahead'), formatAsBytes: true },
                    { key: 'sched', label: _('Scheduler') },
                    { key: 'dax', label: _('Direct Access'), checkNull: true },
                    { key: 'mq', label: _('Multiqueue'), checkNull: true },
                ],
            },
            {
                title: _('Advanced Identifiers and States'),
                labels: 'labelsS6',
                values: 'valuesS6',
                sectionNr: 'section6',
                fields: [
                    { key: 'id-link', label: _('ID Link') },
                    { key: 'id', label: _('ID') },
                    { key: 'maj:min', label: _('Major:Minor') },
                    { key: 'hctl', label: _('HCTL') },
                    { key: 'kname', label: _('Kernel Name') },
                    { key: 'path', label: _('Path') },
                    { key: 'rev', label: _('Revision') },
                    { key: 'wwn', label: _('World Wide Name') },
                    { key: 'tran', label: _('Transport') },
                    { key: 'hotplug', label: _('Hotplug'), checkNull: true },
                    { key: 'rand', label: _('Random'), checkNull: true },
                    { key: 'group', label: _('Group') },
                    { key: 'owner', label: _('Owner') },
                    { key: 'mode', label: _('Mode') },
                    { key: 'ro', label: _('Read Only'), checkNull: true },
                    { key: 'rm', label: _('Removable'), checkNull: true },
                    { key: 'wsame', label: _('Write Same'), checkNull: true },
                    { key: 'zoned', label: _('Zoned'), checkNull: true },
                    { key: 'zone-sz', label: _('Zone Size'), formatAsBytes: true },
                    { key: 'zone-wgran', label: _('Zone Write Granularity'), formatAsBytes: true },
                    { key: 'zone-app', label: _('Zone Append'), checkNull: true },
                    { key: 'zone-nr', label: _('Zone Number') },
                    { key: 'zone-omax', label: _('Zone Open Max') },
                    { key: 'zone-amax', label: _('Zone Active Max') },
                ],
            },
        ];
    }
    createDeviceInfoPopup(sourceActor) {
        const popup = new MenuBase(sourceActor, 0.05, {
            numCols: 4,
        });
        popup.empty = true;
        const configuration = StorageMenu.deviceInfoPopupConfiguration;
        for (const section of configuration) {
            popup[section.sectionNr] = popup.addMenuSection(section.title, true, true);
            popup[section.labels] = [];
            popup[section.values] = [];
            for (let i = 0; i < section.fields.length; i++) {
                const label = new St.Label({ text: '', styleClass: 'astra-monitor-menu-sub-key' });
                popup.addToMenu(label);
                popup[section.labels].push(label);
                const value = new St.Label({
                    text: '',
                    styleClass: 'astra-monitor-menu-sub-value',
                });
                popup.addToMenu(value);
                popup[section.values].push(value);
            }
        }
        return popup;
    }
    createDeviceTotalsPopup(sourceActor) {
        const popup = new MenuBase(sourceActor, 0.05, {
            numCols: 2,
        });
        popup.addMenuSection(_('Total Device Activity'));
        popup.addToMenu(new St.Label({
            text: _('Read'),
            styleClass: 'astra-monitor-menu-sub-key',
        }));
        const totalReadValueLabel = new St.Label({ text: '', style: 'text-align:left;' });
        popup.addToMenu(totalReadValueLabel);
        popup.totalReadValueLabel = totalReadValueLabel;
        popup.addToMenu(new St.Label({
            text: _('Write'),
            styleClass: 'astra-monitor-menu-sub-key',
        }));
        const totalWriteValueLabel = new St.Label({ text: '', style: 'text-align:left;' });
        popup.addToMenu(totalWriteValueLabel);
        popup.totalWriteValueLabel = totalWriteValueLabel;
        return popup;
    }
    updateBlockDevice(device, deviceData) {
        device.data = deviceData;
        const icon = {
            gicon: Utils.getLocalIcon('am-harddisk-symbolic'),
            fallbackIconName: 'drive-harddisk-symbolic',
        };
        if (deviceData.removable) {
            icon.gicon = Utils.getLocalIcon('am-media-removable-symbolic');
            icon.fallbackIconName = 'media-removable-symbolic';
        }
        else if ((deviceData.filesystem && deviceData.filesystem.startsWith('swap')) ||
            deviceData.mountpoints.includes('/boot') ||
            deviceData.mountpoints.includes('[SWAP]')) {
            icon.gicon = Utils.getLocalIcon('am-linux-symbolic');
            icon.fallbackIconName = 'drive-harddisk-system-symbolic';
        }
        else if (deviceData.type.startsWith('raid') ||
            deviceData.type.startsWith('lvm') ||
            deviceData.type.startsWith('md')) {
            icon.gicon = Utils.getLocalIcon('am-raid-symbolic');
            icon.fallbackIconName = 'drive-harddisk-raid-symbolic';
        }
        else if (deviceData.type.startsWith('cdrom') ||
            deviceData.type.startsWith('rom') ||
            deviceData.type.endsWith('rom')) {
            icon.fallbackIconName = 'drive-optical-symbolic';
        }
        else if (deviceData.type.startsWith('floppy')) {
            icon.fallbackIconName = 'media-floppy-symbolic';
        }
        if (icon.gicon)
            device.icon.gicon = icon.gicon;
        device.icon.fallbackIconName = icon.fallbackIconName;
        let label = deviceData.label || '';
        if (!label) {
            if (deviceData.model)
                label = deviceData.model;
            else if (deviceData.vendor)
                label = deviceData.vendor;
            else
                label = _('Disk');
        }
        device.label.text = label;
        let name = deviceData.name;
        if (name && name.length > 17)
            name = name.substring(0, 15) + '…';
        device.name.text = name ? `[${name}]` : '';
        if (!Number.isNaN(deviceData.usage) && deviceData.usage !== undefined) {
            device.barGrid.visible = true;
            device.barLabel.text = `${deviceData.usage}%`;
            device.bar.setUsage({ size: deviceData.size, usePercentage: deviceData.usage });
        }
        else {
            device.barGrid.visible = false;
        }
        const size = deviceData.size;
        if (size)
            device.sizeLabel.text = Utils.formatBytes(size, 'kB-KB', 3);
        else
            device.sizeLabel.text = '-';
    }
    addUtilityButtons() {
        super.addUtilityButtons('storage', box => {
            const appSys = Shell.AppSystem.get_default();
            const baobabApp = appSys.lookup_app('org.gnome.baobab.desktop');
            if (baobabApp) {
                const button = new St.Button({ styleClass: 'button' });
                button.child = new St.Icon({
                    gicon: Utils.getLocalIcon('am-pie-symbolic'),
                    fallbackIconName: 'baobab-symbolic',
                });
                button.connect('clicked', () => {
                    this.close(true);
                    baobabApp.activate();
                });
                box.add_child(button);
            }
            const diskApp = appSys.lookup_app('org.gnome.DiskUtility.desktop');
            if (diskApp) {
                const button = new St.Button({ styleClass: 'button' });
                button.child = new St.Icon({
                    gicon: Utils.getLocalIcon('am-disk-utility-symbolic'),
                    fallbackIconName: 'utilities-disk-utility-symbolic',
                });
                button.connect('clicked', () => {
                    this.close(true);
                    diskApp.activate();
                });
                box.add_child(button);
            }
        });
    }
    async onOpen() {
        this.update('storageIO', true);
        Utils.storageMonitor.listen(this, 'storageIO', this.update.bind(this, 'storageIO', false));
        this.update('detailedStorageIO', true);
        Utils.storageMonitor.listen(this, 'detailedStorageIO', this.update.bind(this, 'detailedStorageIO', false));
        Utils.storageMonitor.requestUpdate('detailedStorageIO');
        if (Utils.GTop) {
            this.topProcesses.separator.show();
            this.topProcesses.hoverButton.show();
            if (Utils.hasIotop()) {
                this.topProcesses.subSeparator.show();
            }
            this.clear('topProcesses');
            this.update('topProcesses', true);
            Utils.storageMonitor.listen(this, 'topProcesses', this.update.bind(this, 'topProcesses', false));
            Utils.storageMonitor.requestUpdate('topProcesses');
            Utils.storageMonitor.listen(this, 'topProcessesIOTop', this.update.bind(this, 'topProcessesIOTop', false));
            Utils.storageMonitor.listen(this, 'topProcessesIOTopStop', this.update.bind(this, 'topProcessesIOTopStop', false));
        }
        this.update('deviceList', true);
        if (!this.updateTimer) {
            this.updateTimer = GLib.timeout_add(GLib.PRIORITY_DEFAULT, Utils.storageMonitor.updateFrequency * 1000 * 2, () => {
                this.update('deviceList', true);
                Utils.storageMonitor.requestUpdate('storageInfo');
                return true;
            });
        }
        this.update('storageInfo', true);
        Utils.storageMonitor.listen(this, 'storageInfo', this.update.bind(this, 'storageInfo', false));
        Utils.storageMonitor.requestUpdate('storageInfo');
    }
    onClose() {
        Utils.storageMonitor.unlisten(this, 'storageIO');
        Utils.storageMonitor.unlisten(this, 'detailedStorageIO');
        Utils.storageMonitor.unlisten(this, 'topProcesses');
        Utils.storageMonitor.unlisten(this, 'topProcessesIOTop');
        Utils.storageMonitor.unlisten(this, 'storageInfo');
        if (this.updateTimer) {
            GLib.source_remove(this.updateTimer);
            this.updateTimer = 0;
        }
    }
    needsUpdate(code, forced = false) {
        if (forced) {
            const valueTime = Utils.storageMonitor.getCurrentValueTime(code);
            return !(valueTime && Date.now() - valueTime > Utils.storageMonitor.updateFrequency);
        }
        return super.needsUpdate(code, forced);
    }
    update(code, forced = false) {
        if (!this.needsUpdate(code, forced)) {
            return;
        }
        if (code === 'deviceList') {
            Utils.lowPriorityTask(() => {
                this.updateDeviceList();
            }, GLib.PRIORITY_DEFAULT);
            return;
        }
        if (code === 'storageIO') {
            const usage = Utils.storageMonitor.getUsageHistory('storageIO');
            this.graph.setUsageHistory(usage);
            const current = Utils.storageMonitor.getCurrentValue('storageIO');
            if (current) {
                const unit = Config.get_string('storage-io-unit');
                if (current.bytesReadPerSec)
                    this.totalReadSpeedValueLabel.text = Utils.formatBytesPerSec(current.bytesReadPerSec, unit, 3);
                else
                    this.totalReadSpeedValueLabel.text = '-';
                if (current.bytesWrittenPerSec)
                    this.totalWriteSpeedValueLabel.text = Utils.formatBytesPerSec(current.bytesWrittenPerSec, unit, 3);
                else
                    this.totalWriteSpeedValueLabel.text = '-';
                if (this.storageActivityPopup) {
                    if (this.storageActivityPopup.totalReadValueLabel) {
                        if (current.totalBytesRead)
                            this.storageActivityPopup.totalReadValueLabel.text = Utils.formatBytes(current.totalBytesRead, 'kB-KB', 3);
                        else
                            this.storageActivityPopup.totalReadValueLabel.text = '-';
                    }
                    if (this.storageActivityPopup.totalWriteValueLabel) {
                        if (current.totalBytesWritten)
                            this.storageActivityPopup.totalWriteValueLabel.text = Utils.formatBytes(current.totalBytesWritten, 'kB-KB', 3);
                        else
                            this.storageActivityPopup.totalWriteValueLabel.text = '-';
                    }
                }
            }
            else {
                this.totalReadSpeedValueLabel.text = '-';
                this.totalWriteSpeedValueLabel.text = '-';
                if (this.storageActivityPopup) {
                    if (this.storageActivityPopup.totalReadValueLabel)
                        this.storageActivityPopup.totalReadValueLabel.text = '-';
                    if (this.storageActivityPopup.totalWriteValueLabel)
                        this.storageActivityPopup.totalWriteValueLabel.text = '-';
                }
            }
            return;
        }
        if (code === 'detailedStorageIO') {
            const current = Utils.storageMonitor.getCurrentValue('detailedStorageIO');
            if (current) {
                for (const [id, device] of this.devices.entries()) {
                    if (device.data === null)
                        continue;
                    const kname = device.data.kname;
                    if (kname) {
                        const data = current.get(kname);
                        if (data) {
                            const unit = Config.get_string('storage-io-unit');
                            if (data.bytesReadPerSec) {
                                device.readValueLabel.text = Utils.formatBytesPerSec(data.bytesReadPerSec, unit, 3);
                                const readColor = Config.get_string('storage-menu-arrow-color1') ??
                                    'rgba(29,172,214,1.0)';
                                device.readActivityIcon.style = `color:${readColor};`;
                            }
                            else {
                                device.readValueLabel.text = '-';
                                device.readActivityIcon.style = 'color:rgba(255,255,255,0.5);';
                            }
                            if (data.bytesWrittenPerSec) {
                                device.writeValueLabel.text = Utils.formatBytesPerSec(data.bytesWrittenPerSec, unit, 3);
                                const writeColor = Config.get_string('storage-menu-arrow-color2') ??
                                    'rgba(214,29,29,1.0)';
                                device.writeActivityIcon.style = `color:${writeColor};`;
                            }
                            else {
                                device.writeValueLabel.text = '-';
                                device.writeActivityIcon.style = 'color:rgba(255,255,255,0.5);';
                            }
                            const totalsPopup = this.devicesTotalsPopup.get(id);
                            if (totalsPopup) {
                                if (totalsPopup.totalReadValueLabel) {
                                    if (data.totalBytesRead)
                                        totalsPopup.totalReadValueLabel.text = Utils.formatBytes(data.totalBytesRead, 'kB-KB', 3);
                                    else
                                        totalsPopup.totalReadValueLabel.text = '-';
                                }
                                if (totalsPopup.totalWriteValueLabel) {
                                    if (data.totalBytesWritten)
                                        totalsPopup.totalWriteValueLabel.text = Utils.formatBytes(data.totalBytesWritten, 'kB-KB', 3);
                                    else
                                        totalsPopup.totalWriteValueLabel.text = '-';
                                }
                            }
                        }
                    }
                    else {
                        device.readValueLabel.text = '-';
                        device.readActivityIcon.style = 'color:rgba(255,255,255,0.5);';
                        device.writeValueLabel.text = '-';
                        device.writeActivityIcon.style = 'color:rgba(255,255,255,0.5);';
                        const totalsPopup = this.devicesTotalsPopup.get(id);
                        if (totalsPopup) {
                            if (totalsPopup.totalReadValueLabel)
                                totalsPopup.totalReadValueLabel.text = '-';
                            if (totalsPopup.totalWriteValueLabel)
                                totalsPopup.totalWriteValueLabel.text = '-';
                        }
                    }
                }
            }
            return;
        }
        if (code === 'topProcessesIOTopStop') {
            this.stopPrivilegedTopProcesses();
            return;
        }
        if (code === 'topProcesses' || code === 'topProcessesIOTop') {
            let topProcesses;
            if (code === 'topProcessesIOTop') {
                this.startPrivilegedTopProcesses();
                topProcesses = Utils.storageMonitor.getCurrentValue('topProcessesIOTop');
            }
            else {
                if (this.privilegedTopProcesses)
                    return;
                topProcesses = Utils.storageMonitor.getCurrentValue('topProcesses');
            }
            for (let i = 0; i < StorageMonitor.TOP_PROCESSES_LIMIT; i++) {
                if (!topProcesses ||
                    !Array.isArray(topProcesses) ||
                    !topProcesses[i] ||
                    !topProcesses[i].process) {
                    if (i < 3) {
                        this.topProcesses.labels[i].label.text = '-';
                        this.topProcesses.labels[i].read.value.text = '-';
                        this.topProcesses.labels[i].read.icon.style =
                            'color:rgba(255,255,255,0.5);';
                        this.topProcesses.labels[i].write.value.text = '-';
                        this.topProcesses.labels[i].write.icon.style =
                            'color:rgba(255,255,255,0.5);';
                    }
                    if (this.topProcessesPopup && this.topProcessesPopup.processes) {
                        const popupElement = this.topProcessesPopup.processes.get(i);
                        if (popupElement) {
                            popupElement.label.hide();
                            popupElement.description?.hide();
                            popupElement.read.container.hide();
                            popupElement.write.container.hide();
                        }
                    }
                }
                else {
                    const unit = Config.get_string('storage-io-unit');
                    const topProcess = topProcesses[i];
                    const process = topProcess.process;
                    const read = topProcess.read;
                    const write = topProcess.write;
                    if (i < 3) {
                        this.topProcesses.labels[i].label.text = process.exec;
                        if (read > 0) {
                            const readColor = Config.get_string('storage-menu-arrow-color1') ??
                                'rgba(29,172,214,1.0)';
                            this.topProcesses.labels[i].read.icon.style = `color:${readColor};`;
                            this.topProcesses.labels[i].read.value.text = Utils.formatBytesPerSec(read, unit, 3);
                        }
                        else {
                            this.topProcesses.labels[i].read.icon.style =
                                'color:rgba(255,255,255,0.5);';
                            this.topProcesses.labels[i].read.value.text = '-';
                        }
                        if (write > 0) {
                            const writeColor = Config.get_string('storage-menu-arrow-color2') ??
                                'rgba(214,29,29,1.0)';
                            this.topProcesses.labels[i].write.icon.style = `color:${writeColor};`;
                            this.topProcesses.labels[i].write.value.text = Utils.formatBytesPerSec(write, unit, 3);
                        }
                        else {
                            this.topProcesses.labels[i].write.icon.style =
                                'color:rgba(255,255,255,0.5);';
                            this.topProcesses.labels[i].write.value.text = '-';
                        }
                    }
                    if (this.topProcessesPopup && this.topProcessesPopup.processes) {
                        const popupElement = this.topProcessesPopup.processes.get(i);
                        if (popupElement) {
                            popupElement.label.show();
                            popupElement.label.text = process.exec;
                            if (popupElement.description) {
                                popupElement.description.show();
                                popupElement.description.text = process.cmd;
                            }
                            popupElement.read.container.show();
                            if (read > 0) {
                                const readColor = Config.get_string('storage-menu-arrow-color1') ??
                                    'rgba(29,172,214,1.0)';
                                popupElement.read.icon.style = `color:${readColor};`;
                                popupElement.read.value.text = Utils.formatBytesPerSec(read, unit, 3);
                            }
                            else {
                                popupElement.read.icon.style = 'color:rgba(255,255,255,0.5);';
                                popupElement.read.value.text = '-';
                            }
                            popupElement.write.container.show();
                            if (write > 0) {
                                const writeColor = Config.get_string('storage-menu-arrow-color2') ??
                                    'rgba(214,29,29,1.0)';
                                popupElement.write.icon.style = `color:${writeColor};`;
                                popupElement.write.value.text = Utils.formatBytesPerSec(write, unit, 3);
                            }
                            else {
                                popupElement.write.icon.style = 'color:rgba(255,255,255,0.5);';
                                popupElement.write.value.text = '-';
                            }
                        }
                    }
                }
            }
            return;
        }
        if (code === 'storageInfo') {
            const storageInfo = Utils.storageMonitor.getCurrentValue('storageInfo');
            const formatValue = (value, isBytes = false) => {
                if (Array.isArray(value))
                    return value.join('\n');
                if (isBytes && typeof value === 'number')
                    return Utils.formatBytes(value, 'kB-KB', 4);
                if (typeof value === 'boolean')
                    return value ? _('Yes') : _('No');
                let str = value?.toString().trim() ?? '';
                if (str.length > 100)
                    str = str.substring(0, 97) + '…';
                if (str.length > 50)
                    str = str.substring(0, str.length / 2) + ' ⏎\n' + str.substring(str.length / 2);
                return str;
            };
            const configuration = StorageMenu.deviceInfoPopupConfiguration;
            for (const [id, popup] of this.devicesInfoPopup.entries()) {
                const info = storageInfo.get(id);
                if (!info) {
                    popup.empty = true;
                    popup.close(true);
                    continue;
                }
                popup.empty = false;
                for (const section of configuration) {
                    let i = 0;
                    const labels = popup[section.labels];
                    const values = popup[section.values];
                    for (const field of section.fields) {
                        let value;
                        if (info[field.key])
                            value = info[field.key];
                        else if (field.parent && info.parent && info.parent[field.key])
                            value = info.parent[field.key];
                        if (field.checkNull ? value != null : value) {
                            const formattedValue = formatValue(value, field.formatAsBytes);
                            labels[i].text = field.label;
                            labels[i].show();
                            values[i].text = formattedValue;
                            values[i].show();
                            i++;
                        }
                    }
                    if (i === 0)
                        popup[section.sectionNr].hide();
                    else
                        popup[section.sectionNr].show();
                    for (; i < labels.length; i++) {
                        labels[i].hide();
                        values[i].hide();
                    }
                }
            }
            return;
        }
    }
    startPrivilegedTopProcesses() {
        if (this.privilegedTopProcesses) {
            return;
        }
        this.privilegedTopProcesses = true;
        this.topProcesses.subSeparator.text = _('(showing all system processes)');
        this.topProcesses.separator.text = _('Top processes') + ` (root)`;
        this.topProcessesPopup.section.text = _('Top processes') + ` (root)`;
    }
    stopPrivilegedTopProcesses() {
        if (!this.privilegedTopProcesses) {
            return;
        }
        this.privilegedTopProcesses = false;
        Utils.storageMonitor.stopIOTop();
        this.topProcesses.subSeparator.text = _('(click to show all system processes)');
        this.topProcesses.separator.text = _('Top processes') + ` (${GLib.get_user_name()})`;
        this.topProcessesPopup.section.text = _('Top processes') + ` (${GLib.get_user_name()})`;
    }
    clear(code = 'all') {
        if (code === 'all') {
            this.totalReadSpeedValueLabel.text = '-';
            this.totalWriteSpeedValueLabel.text = '-';
        }
        if (code === 'all' || code === 'devices') {
            for (const [_id, device] of this.devices.entries()) {
                device.readValueLabel.text = '-';
                device.readActivityIcon.style = 'color:rgba(255,255,255,0.5);';
                device.writeValueLabel.text = '-';
                device.writeActivityIcon.style = 'color:rgba(255,255,255,0.5);';
            }
        }
        if (code === 'all' || code === 'topProcesses') {
            for (const process of this.topProcesses.labels) {
                process.label.text = '-';
                process.read.value.text = '-';
                process.read.icon.style = 'color:rgba(255,255,255,0.5);';
                process.write.value.text = '-';
                process.write.icon.style = 'color:rgba(255,255,255,0.5);';
            }
        }
    }
    destroy() {
        Config.clear(this);
        this.storageActivityPopup?.destroy();
        this.storageActivityPopup = undefined;
        this.graph?.destroy();
        this.graph = undefined;
        this.topProcessesPopup?.destroy();
        this.topProcessesPopup = undefined;
        for (const [id, device] of this.devices.entries()) {
            device.bar?.destroy();
            device.bar = undefined;
            this.devicesInfoPopup.get(id)?.close(true);
            this.devicesInfoPopup.get(id)?.destroy();
            this.devicesInfoPopup.delete(id);
            this.devicesTotalsPopup.get(id)?.close(true);
            this.devicesTotalsPopup.get(id)?.destroy();
            this.devicesTotalsPopup.delete(id);
        }
        this.deviceSection.remove_all_children();
        this.deviceSection?.destroy();
        this.deviceSection = undefined;
        super.destroy();
    }
}
