// =============================================================================
// DRAG & DROP MANAGER
// =============================================================================
// Handles drag-and-drop window rearrangement for SLAB tiling.
// Detects when a tiled window is grabbed, shows visual preview of drop zones,
// and performs window swap on drop.
import Meta from "gi://Meta";
import { DropZoneOverlay, getDropZoneAtPosition } from "./overlay.js";
// =============================================================================
// MODULE STATE
// =============================================================================
let overlay = null;
let grabBeginSignalId = null;
let grabEndSignalId = null;
let positionChangedSignalId = null;
/** Current layout zones (updated when tiling is recalculated) */
let currentZones = [];
/** Callback to get current tiled windows order */
let getTiledWindowsCallback = null;
/** Callback to swap window positions and re-tile */
let swapWindowsCallback = null;
// =============================================================================
// INITIALIZATION
// =============================================================================
/**
 * Initialize the drag manager.
 * Call this when tiling is enabled.
 */
export function initDragManager(state, getTiledWindows, swapWindows) {
    console.log("[SLAB-DRAG] Initializing drag manager");
    // Store callbacks
    getTiledWindowsCallback = getTiledWindows;
    swapWindowsCallback = swapWindows;
    // Create overlay
    overlay = new DropZoneOverlay();
    // Connect to display signals
    const display = global.display;
    grabBeginSignalId = display.connect("grab-op-begin", (display, window, grabOp) => {
        handleGrabBegin(state, window, grabOp);
    });
    grabEndSignalId = display.connect("grab-op-end", (display, window, grabOp) => {
        handleGrabEnd(state, window);
    });
    console.log("[SLAB-DRAG] Drag manager initialized");
}
/**
 * Clean up the drag manager.
 * When tiling is disabled or extension is destroyed.
 */
export function cleanupDragManager(state) {
    console.log("[SLAB-DRAG] Cleaning up drag manager");
    // Disconnect display signals
    const display = global.display;
    if (grabBeginSignalId !== null) {
        display.disconnect(grabBeginSignalId);
        grabBeginSignalId = null;
    }
    if (grabEndSignalId !== null) {
        display.disconnect(grabEndSignalId);
        grabEndSignalId = null;
    }
    // Clean up any active drag
    cancelDrag(state);
    // Destroy overlay
    if (overlay) {
        overlay.destroy();
        overlay = null;
    }
    // Clear callbacks
    getTiledWindowsCallback = null;
    swapWindowsCallback = null;
    currentZones = [];
    console.log("[SLAB-DRAG] Drag manager cleaned up");
}
// =============================================================================
// DROP ZONE MANAGEMENT
// =============================================================================
/**
 * Update the available drop zones based on current layout.
 * After layout is calculated.
 */
export function updateDropZones(zones) {
    currentZones = zones;
    console.log(`[SLAB-DRAG] Updated ${zones.length} drop zones`);
}
/**
 * Build drop zones from tiled windows.
 * Each window's current position becomes a drop zone.
 */
export function buildDropZonesFromWindows(windows) {
    const zones = [];
    for (let i = 0; i < windows.length; i++) {
        const window = windows[i];
        const frame = window.get_frame_rect();
        zones.push({
            index: i,
            x: frame.x,
            y: frame.y,
            width: frame.width,
            height: frame.height,
            windowId: window.get_stable_sequence(),
        });
    }
    return zones;
}
// =============================================================================
// DRAG EVENT HANDLERS
// =============================================================================
/**
 * Handle grab begin - start tracking if moving a tiled window.
 */
function handleGrabBegin(state, window, grabOp) {
    // Only handle when tiling is enabled
    if (!state.tilingEnabled) {
        return;
    }
    // Only handle MOVING operations (not resize)
    // Accept regular MOVING, KEYBOARD_MOVING, and MOVING_UNCONSTRAINED (Meta+drag)
    const isMovingOp = grabOp === Meta.GrabOp.MOVING ||
        grabOp === Meta.GrabOp.KEYBOARD_MOVING ||
        grabOp === Meta.GrabOp.MOVING_UNCONSTRAINED;
    if (!isMovingOp) {
        console.log("[SLAB-DRAG] Ignoring non-move grab op:", grabOp);
        return;
    }
    // Check if this window is in our tiled set
    if (!getTiledWindowsCallback) {
        return;
    }
    const tiledWindows = getTiledWindowsCallback();
    const windowId = window.get_stable_sequence();
    const index = tiledWindows.findIndex((w) => w.get_stable_sequence() === windowId);
    if (index === -1) {
        console.log("[SLAB-DRAG] Window not in tiled set, ignoring:", window.title);
        return;
    }
    console.log(`[SLAB-DRAG] Drag started for: ${window.title} (index ${index})`);
    // Build drop zones from current layout
    const zones = buildDropZonesFromWindows(tiledWindows);
    updateDropZones(zones);
    // Set up drag state
    state.dragState = {
        draggedWindow: window,
        originalIndex: index,
        signalIds: [],
    };
    // Connect to position changes during drag
    positionChangedSignalId = window.connect("position-changed", () => {
        handlePositionChanged(state, window);
    });
    state.dragState.signalIds.push(positionChangedSignalId);
}
/**
 * Handle position changed - update drop zone preview.
 */
function handlePositionChanged(state, window) {
    if (!state.dragState || !overlay) {
        return;
    }
    // Get current pointer position (use window center as approximation)
    const frame = window.get_frame_rect();
    const centerX = frame.x + frame.width / 2;
    const centerY = frame.y + frame.height / 2;
    // Find drop zone under pointer
    const zone = getDropZoneAtPosition(centerX, centerY, currentZones);
    if (zone && zone.index !== state.dragState.originalIndex) {
        // Show overlay on target zone (not dragged window's original zone)
        overlay.show(zone);
    }
    else {
        // Hide overlay if over original position or outside zones
        overlay.hide();
    }
}
/**
 * Handle grab end - perform swap if dropped on valid zone.
 */
function handleGrabEnd(state, window) {
    if (!state.dragState || !overlay) {
        return;
    }
    const currentZone = overlay.getCurrentZone();
    const originalIndex = state.dragState.originalIndex;
    const draggedWindow = state.dragState.draggedWindow;
    // Hide overlay
    overlay.hide();
    // Perform swap if dropped on different zone
    if (currentZone && currentZone.index !== originalIndex) {
        console.log(`[SLAB-DRAG] Swapping index ${originalIndex} <-> ${currentZone.index}`);
        if (swapWindowsCallback) {
            swapWindowsCallback(originalIndex, currentZone.index);
        }
    }
    else {
        // No swap - snap the window back to its original layout position
        console.log("[SLAB-DRAG] Drag ended without swap - restoring position");
        const originalZone = currentZones.find((z) => z.index === originalIndex);
        if (originalZone) {
            console.log(`[SLAB-DRAG] Restoring to: ${originalZone.x},${originalZone.y} ${originalZone.width}x${originalZone.height}`);
            draggedWindow.move_resize_frame(true, originalZone.x, originalZone.y, originalZone.width, originalZone.height);
        }
    }
    // Clean up drag state
    cancelDrag(state);
}
/**
 * Cancel active drag and clean up.
 */
function cancelDrag(state) {
    if (state.dragState) {
        // Disconnect position signal
        if (positionChangedSignalId !== null) {
            try {
                state.dragState.draggedWindow.disconnect(positionChangedSignalId);
            }
            catch (e) {
                // Window might be destroyed
            }
            positionChangedSignalId = null;
        }
        state.dragState = null;
    }
    overlay?.hide();
}
