import GLib from "gi://GLib";
import St from "gi://St";
import Clutter from "gi://Clutter";
import * as Main from "resource:///org/gnome/shell/ui/main.js";
import * as PanelMenu from "resource:///org/gnome/shell/ui/panelMenu.js";
import { Extension } from "resource:///org/gnome/shell/extensions/extension.js";

// Import our modular components
import { PriceService } from './services/price.js';
import { AlertService } from './services/alerts.js';
import { HistoryService } from './services/history.js';
import { MenuUI } from './ui/menu.js';
import { SettingsManager } from './utils/settings.js';
import { CONFIG } from './utils/constants.js';

export default class BTCPriceExtension extends Extension {
  enable() {
    log("BTC Extension: Enabling");
    
    // Initialize settings
    this.settings = new SettingsManager(this.getSettings());
    
    // Initialize services
    this.priceService = new PriceService();
    this.alertService = new AlertService(this.settings);
    this.historyService = new HistoryService(this.settings);
    
    // State variables
    this.currentPrice = null;
    this.previousPrice = null;
    this.timeout = null;
    
    // Create UI
    this._createPanelButton();
    this._initializeMenu();
    
    // Set up periodic updates
    this._startPeriodicUpdates();
    
    // Fetch initial price
    log("BTC Extension: Starting first fetch");
    this._fetchPrice();
  }

  disable() {
    log("BTC Extension: Disabling");
    
    // Clean up periodic updates
    this._stopPeriodicUpdates();
    
    // Clean up services
    if (this.priceService) {
      this.priceService.destroy();
      this.priceService = null;
    }
    
    // Clean up UI
    if (this.indicator) {
      this.indicator.destroy();
      this.indicator = null;
    }
    
    // Clear state
    this.currentPrice = null;
    this.previousPrice = null;
    this.settings = null;
    this.alertService = null;
  }

  _createPanelButton() {
    // Create panel button with label
    this.indicator = new PanelMenu.Button(0.0, "BTC Price Indicator", false);
    this.label = new St.Label({
      text: "Loading: ...",
      style: `color: ${CONFIG.UI.DEFAULT_COLOR}; font-weight: bold;`,
      y_align: Clutter.ActorAlign.CENTER,
    });
    this.indicator.add_child(this.label);

    // Add to panel
    Main.panel.addToStatusArea("btc-price-indicator", this.indicator);
  }

  _initializeMenu() {
    this.menuUI = new MenuUI(
      this.indicator,
      this.settings,
      () => this._fetchPrice(),
      (currency) => this._handleCurrencyChange(currency),
      (price) => this._handleSetAlert(price),
      () => this._handleClearAlert(),
      (format) => this._handleDisplayFormatChange(format)
    );
    
    // Initialize alert status if there's an existing alert
    this._updateInitialAlertStatus();
  }

  _startPeriodicUpdates() {
    this.timeout = GLib.timeout_add_seconds(
      CONFIG.UPDATE.PRIORITY,
      CONFIG.UPDATE.INTERVAL_SECONDS,
      () => {
        log("BTC Extension: Periodic fetch triggered");
        this._fetchPrice();
        return GLib.SOURCE_CONTINUE;
      }
    );
  }

  _stopPeriodicUpdates() {
    if (this.timeout) {
      GLib.Source.remove(this.timeout);
      this.timeout = null;
    }
  }

  async _fetchPrice() {
    const currency = this.settings.getCurrency();
    
    try {
      const price = await this.priceService.fetchPrice(currency);
      
       // Update price history
       this.previousPrice = this.currentPrice;
       this.currentPrice = price;
       
       // Save to settings
       this.settings.setLastPrice(price);
       
       // Add to history
       this.historyService.addPricePoint(price);
       
       // Update displays
       this._updateDisplay();
       this.menuUI.updatePriceDisplay(price);
      
      // Check alerts
      this._checkAlert();
      
      log(`BTC Extension: Price updated to ${price}`);
      
    } catch (error) {
      log(`BTC Extension: Error fetching price: ${error.message}`);
      this._handleFetchError(error);
    }
  }

  _updateDisplay() {
    if (!this.currentPrice) return;
    
    const priceStr = this.priceService.formatPrice(this.currentPrice, this.settings.getCurrency());
    const currencySymbol = this.settings.getCurrencySymbol();
    const displayFormat = this.settings.getDisplayFormat();
    
    // Determine color based on price change
    let labelColor = CONFIG.UI.DEFAULT_COLOR;
    if (this.previousPrice && this.previousPrice > 0) {
      if (this.currentPrice > this.previousPrice) {
        labelColor = CONFIG.UI.UP_COLOR;
      } else if (this.currentPrice < this.previousPrice) {
        labelColor = CONFIG.UI.DOWN_COLOR;
      }
    }
    
    // Format display based on selected format
    let displayText = `${currencySymbol}${priceStr}`;
    
    if (displayFormat === CONFIG.DISPLAY_FORMATS.PRICE_WITH_CHANGE && this.previousPrice && this.previousPrice > 0) {
      const change = this.currentPrice - this.previousPrice;
      const changeStr = Math.abs(change).toLocaleString("en-US", {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2,
      });
      const changeSymbol = change >= 0 ? "↑" : "↓";
      displayText += ` ${changeSymbol}${currencySymbol}${changeStr}`;
    } else if (displayFormat === CONFIG.DISPLAY_FORMATS.PRICE_WITH_PERCENTAGE && this.previousPrice && this.previousPrice > 0) {
      const change = ((this.currentPrice - this.previousPrice) / this.previousPrice) * 100;
      const changeStr = Math.abs(change).toLocaleString("en-US", {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2,
      });
      const changeSymbol = change >= 0 ? "↑" : "↓";
      displayText += ` ${changeSymbol}${changeStr}%`;
    }
    
    // Update panel label
    this.label.set_text(displayText);
    this.label.set_style(`color: ${labelColor}; font-weight: bold;`);
    
    log(`BTC Extension: Display updated to ${displayText} with color ${labelColor}`);
  }

  _checkAlert() {
    const result = this.alertService.checkAlert(this.previousPrice, this.currentPrice);
    if (result) {
      // Set label to alert color when triggered
      this.label.set_style(`color: ${CONFIG.UI.ALERT_COLOR}; font-weight: bold;`);
    }
  }

  _handleFetchError(error) {
    if (error.message.includes("No HTTP session")) {
      this.label.set_text("BTC: Kein HTTP");
    } else if (error.message.includes("No data")) {
      this.label.set_text("BTC: Keine Daten");
    } else if (error.message.includes("HTTP")) {
      this.label.set_text(`BTC: ${error.message}`);
    } else {
      this.label.set_text("BTC: Ungültig");
    }
  }

  _handleCurrencyChange(currency) {
    if (currency !== this.settings.getCurrency()) {
      this.settings.setCurrency(currency);
      
      // Update menu display
      this.menuUI.updateCurrencyDisplay(currency);
      
      // Clear current data and fetch new price
      this.currentPrice = null;
      this.previousPrice = null;
      this.settings.setLastPrice(0);
      
      // Clear alert when currency changes
      this.alertService.clearAlert();
      this.menuUI.updateAlertStatus("No alert set");
      
      // Fetch new price immediately
      this._fetchPrice();
      
      log(`BTC Extension: Currency changed to ${currency}`);
    }
  }

  _handleSetAlert(price) {
    const result = this.alertService.setAlert(price, this.currentPrice);
    if (result.success) {
      this.menuUI.updateAlertStatus(result.message);
      return result;
    } else {
      return { success: false, message: "Invalid price" };
    }
  }

  _handleClearAlert() {
    this.alertService.clearAlert();
    this.menuUI.updateAlertStatus("No alert set");
  }

  _updateInitialAlertStatus() {
    if (this.alertService.hasAlert()) {
      const status = this.alertService.getAlertStatus(this.currentPrice);
      this.menuUI.updateAlertStatus(status);
    }
  }

  _handleDisplayFormatChange(format) {
    this._updateDisplay();
    log(`BTC Extension: Display format changed to ${format}`);
  }
}
