#!/bin/bash

# Script to create and push a release tag for the BTC Ticker GNOME extension

set -e

# Color codes for output
GREEN='\033[0;32m'
BLUE='\033[0;34m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# Check if version argument is provided
if [ -z "$1" ]; then
    echo -e "${RED}Error: Version number required${NC}"
    echo "Usage: ./release.sh <version>"
    echo "Example: ./release.sh 1.2.3"
    exit 1
fi

VERSION="$1"
TAG="v${VERSION}"

echo -e "${BLUE}Creating release for version ${VERSION}${NC}"
echo ""

# Check if tag already exists
if git rev-parse "$TAG" >/dev/null 2>&1; then
    echo -e "${RED}Error: Tag $TAG already exists${NC}"
    exit 1
fi

# Check if working directory is clean
if ! git diff-index --quiet HEAD --; then
    echo -e "${RED}Error: Working directory has uncommitted changes${NC}"
    echo "Please commit or stash your changes first"
    exit 1
fi

# Create the tag
echo -e "${BLUE}Creating git tag: $TAG${NC}"
git tag -a "$TAG" -m "Release version $VERSION"

# Push the tag
echo -e "${BLUE}Pushing tag to remote...${NC}"
git push origin "$TAG"

echo -e "${GREEN}✓ Tag $TAG created and pushed successfully!${NC}"
echo ""
echo "The release workflow will now:"
echo "  1. Extract version from the tag"
echo "  2. Update metadata.json"
echo "  3. Create a zip archive"
echo "  4. Create a release on Codeberg"
echo ""
echo "You can monitor the workflow at:"
echo "  https://codeberg.org/md-weber/btc-ticker/actions"
