import * as Main from "resource:///org/gnome/shell/ui/main.js";
import { CONFIG } from '../utils/constants.js';

export class AlertService {
  constructor(settingsManager) {
    this.settings = settingsManager;
  }

  setAlert(price, currentPrice) {
    if (isNaN(price) || price <= 0) {
      return { success: false, message: "Invalid price" };
    }

    this.settings.setAlertPrice(price);
    this.settings.setAlertTriggered(false);

    const direction = this._determineDirection(price, currentPrice);
    const message = this._formatAlertMessage(price, direction);
    
    log(`BTC Extension: Alert set to ${price} (direction: ${direction})`);
    
    return { success: true, message, direction, price };
  }

  clearAlert() {
    this.settings.clearAlert();
    log("BTC Extension: Alert cleared");
    return "No alert set";
  }

  checkAlert(previousPrice, currentPrice) {
    const alertPrice = this.settings.getAlertPrice();
    
    if (!alertPrice || !currentPrice || !previousPrice || this.settings.getAlertTriggered()) {
      return null;
    }

    const alertTriggered = this._checkThresholdCrossing(alertPrice, previousPrice, currentPrice);
    
    if (alertTriggered) {
      this.settings.setAlertTriggered(true);
      this._showAlertNotification(alertTriggered, alertPrice, currentPrice);
      this._logAlertTriggered(alertTriggered, alertPrice, currentPrice);
      return alertTriggered;
    }
    
    return null;
  }

  getAlertStatus(currentPrice) {
    const alertPrice = this.settings.getAlertPrice();
    
    if (alertPrice <= 0) {
      return "No alert set";
    }

    const direction = this._determineDirection(alertPrice, currentPrice);
    return this._formatAlertMessage(alertPrice, direction);
  }

  _determineDirection(alertPrice, currentPrice) {
    if (!currentPrice) return "above";
    return alertPrice > currentPrice ? "above" : "below";
  }

  _checkThresholdCrossing(alertPrice, previousPrice, currentPrice) {
    // Check if we crossed the alert threshold
    if (alertPrice > previousPrice && alertPrice <= currentPrice) {
      return "above";
    } else if (alertPrice < previousPrice && alertPrice >= currentPrice) {
      return "below";
    }
    return null;
  }

  _formatAlertMessage(price, direction) {
    const currencySymbol = this.settings.getCurrencySymbol();
    const formattedPrice = price.toLocaleString("en-US", { minimumFractionDigits: 2 });
    return `Alert when price goes ${direction}: ${currencySymbol}${formattedPrice}`;
  }

  _showAlertNotification(direction, alertPrice, currentPrice) {
    const currencySymbol = this.settings.getCurrencySymbol();
    const alertPriceStr = alertPrice.toLocaleString("en-US");
    const currentPriceStr = currentPrice.toLocaleString("en-US", { minimumFractionDigits: 2 });
    
    Main.notify(
      "Bitcoin Price Alert",
      `BTC went ${direction} ${currencySymbol}${alertPriceStr}!\n` +
        `Current price: ${currencySymbol}${currentPriceStr}`,
    );
  }

  _logAlertTriggered(direction, alertPrice, currentPrice) {
    log(`BTC Extension: Alert triggered - price went ${direction} ${alertPrice} at ${currentPrice}`);
  }

  hasAlert() {
    return this.settings.getAlertPrice() > 0;
  }

  isAlertTriggered() {
    return this.settings.getAlertTriggered();
  }
}