import { CONFIG } from '../utils/constants.js';
import GLib from 'gi://GLib';

export class HistoryService {
  constructor(settingsManager) {
    this.settings = settingsManager;
  }

  addPricePoint(price) {
    const history = this.getPriceHistory();
    const timestamp = Math.floor(Date.now() / 1000); // Unix timestamp in seconds
    
    // Add new price point
    history.push([timestamp, price]);
    
    // Keep only the most recent points
    const maxPoints = this.getMaxHistoryPoints();
    if (history.length > maxPoints) {
      history.splice(0, history.length - maxPoints);
    }
    
    // Save back to settings
    this.settings.getSettings().set_value("price-history", new GLib.Variant("a(di)", history));
    
    log(`BTC Extension: Added price point ${price} to history (total: ${history.length} points)`);
  }

  getPriceHistory() {
    const history = this.settings.getSettings().get_value("price-history");
    try {
      return history && history.n_children() > 0 ? Array.from(history) : [];
    } catch (e) {
      log(`BTC Extension: Error reading history: ${e.message}`);
      return [];
    }
  }

  getRecentHistory(points = 24) {
    const history = this.getPriceHistory();
    return history.slice(-points);
  }

  clearHistory() {
    this.settings.getSettings().set_value("price-history", new GLib.Variant("a(di)", []));
    log("BTC Extension: Price history cleared");
  }

  getLatestPrice() {
    const history = this.getPriceHistory();
    return history.length > 0 ? history[history.length - 1][1] : null;
  }

  getPriceChange(period = 1) {
    const history = this.getRecentHistory(period + 1);
    if (history.length < 2) return null;
    
    const latest = history[history.length - 1][1];
    const previous = history[0][1];
    
    return {
      absolute: latest - previous,
      percentage: ((latest - previous) / previous) * 100,
      latest,
      previous
    };
  }

  get24HourChange() {
    // Get points from last 24 hours (assuming 1 update per minute as configured)
    const history24h = this.getRecentHistory(1440); // 24 hours * 60 minutes
    return this.getPriceChange(history24h.length);
  }

  getFormattedTime(timestamp) {
    const date = new Date(timestamp * 1000);
    return date.toLocaleTimeString('en-US', { 
      hour: '2-digit', 
      minute: '2-digit' 
    });
  }

  getChartData(points = 24) {
    const history = this.getRecentHistory(points);
    if (history.length === 0) return [];
    
    // Extract just the prices for the chart
    return history.map(point => point[1]);
  }

  getMinPrice(points = 24) {
    const history = this.getRecentHistory(points);
    return history.length > 0 ? Math.min(...history.map(p => p[1])) : null;
  }

  getMaxPrice(points = 24) {
    const history = this.getRecentHistory(points);
    return history.length > 0 ? Math.max(...history.map(p => p[1])) : null;
  }

  getMaxHistoryPoints() {
    return this.settings.getSettings().get_int("history-max-points");
  }

  setMaxHistoryPoints(points) {
    this.settings.getSettings().set_int("history-max-points", points);
    
    // Trim existing history if needed
    const history = this.getPriceHistory();
    if (history.length > points) {
      history.splice(0, history.length - points);
      this.settings.getSettings().set_value("price-history", new GLib.Variant("a(di)", history));
    }
  }
}