import GLib from "gi://GLib";
import Soup from "gi://Soup";
import { CONFIG } from '../utils/constants.js';

export class PriceService {
  constructor(onPriceUpdate) {
    this.httpSession = null;
    this.onPriceUpdate = onPriceUpdate;
    this._initializeHttpSession();
  }

  _initializeHttpSession() {
    try {
      this.httpSession = new Soup.Session({
        timeout: CONFIG.API.TIMEOUT / 1000,
        user_agent: CONFIG.API.USER_AGENT,
      });
    } catch (e) {
      log("BTC Extension: Failed to create Soup session: " + e);
    }
  }

  fetchPrice(currency) {
    return new Promise((resolve, reject) => {
      if (!this.httpSession) {
        reject(new Error("No HTTP session available"));
        return;
      }

      const uri = `${CONFIG.API.BASE_URL}/simple/price?ids=${CONFIG.API.COIN_ID}&vs_currencies=${currency}`;
      const message = Soup.Message.new("GET", uri);
      message.set_property("timeout", CONFIG.API.TIMEOUT / 1000);

      log(`BTC Extension: Fetching price from ${uri}`);

      this.httpSession.send_and_read_async(
        message,
        CONFIG.UPDATE.PRIORITY,
        null,
        (session, result) => {
          try {
            const bytes = session.send_and_read_finish(result);
            
            if (!bytes) {
              reject(new Error("No data received"));
              return;
            }

            const status = message.get_status();
            if (status !== 200) {
              reject(new Error(`HTTP ${status}`));
              return;
            }

            const decoder = new TextDecoder("utf-8");
            const response = decoder.decode(bytes.get_data());
            const data = JSON.parse(response);

            if (data.bitcoin && data.bitcoin[currency]) {
              resolve(data.bitcoin[currency]);
            } else {
              reject(new Error("Invalid data structure"));
            }
          } catch (e) {
            reject(e);
          }
        }
      );
    });
  }

  formatPrice(price, currency) {
    return price.toLocaleString("en-US", {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    });
  }

  getCurrencySymbol(currency) {
    return currency === CONFIG.CURRENCIES.EUR ? "€" : "$";
  }

  getCurrencyDisplay(currency) {
    return currency.toUpperCase();
  }

  destroy() {
    if (this.httpSession) {
      this.httpSession.abort();
      this.httpSession = null;
    }
  }
}