import St from "gi://St";
import Clutter from "gi://Clutter";
import { CONFIG } from '../utils/constants.js';

export class SparklineChart {
  constructor(width = 280, height = 80) {
    this.width = width;
    this.height = height;
    this.drawingArea = new St.DrawingArea({
      width: width,
      height: height,
      style: "margin: 10px 0;",
    });
    
    this.data = [];
    this.minPrice = 0;
    this.maxPrice = 0;
    this.strokeColor = CONFIG.UI.UP_COLOR;
    this.fillColor = CONFIG.UI.UP_COLOR + "20"; // Add transparency
    
    this.drawingArea.connect("repaint", () => this._draw());
  }

  setData(data) {
    this.data = data;
    
    if (data.length > 0) {
      this.minPrice = Math.min(...data);
      this.maxPrice = Math.max(...data);
      
      // Determine stroke color based on trend
      if (data.length > 1) {
        const firstPrice = data[0];
        const lastPrice = data[data.length - 1];
        this.strokeColor = lastPrice >= firstPrice ? CONFIG.UI.UP_COLOR : CONFIG.UI.DOWN_COLOR;
        this.fillColor = this.strokeColor + "20";
      }
    }
    
    this.drawingArea.queue_repaint();
  }

  getActor() {
    return this.drawingArea;
  }

  _draw() {
    if (this.data.length === 0) {
      this._drawEmptyState();
      return;
    }
    
    const cr = this.drawingArea.get_context();
    if (!cr) return;
    
    const width = this.width;
    const height = this.height;
    const padding = 5;
    
    // Calculate drawing area
    const drawWidth = width - 2 * padding;
    const drawHeight = height - 2 * padding;
    
    // Normalize data points
    const priceRange = this.maxPrice - this.minPrice || 1;
    const normalizedData = this.data.map((price, index) => {
      const x = padding + (index / Math.max(this.data.length - 1, 1)) * drawWidth;
      const y = padding + drawHeight - ((price - this.minPrice) / priceRange) * drawHeight;
      return { x, y };
    });
    
    // Simple fill area (no gradient)
    cr.setSourceRgba(
      parseInt(this.fillColor.slice(1, 3), 16) / 255,
      parseInt(this.fillColor.slice(3, 5), 16) / 255,
      parseInt(this.fillColor.slice(5, 7), 16) / 255,
      0.2
    );
    
    // Draw filled area
    cr.moveTo(normalizedData[0].x, height);
    
    normalizedData.forEach(point => {
      cr.lineTo(point.x, point.y);
    });
    
    cr.lineTo(normalizedData[normalizedData.length - 1].x, height);
    cr.closePath();
    cr.fill();
    
    // Draw the line
    cr.setSourceRgba(
      parseInt(this.strokeColor.slice(1, 3), 16) / 255,
      parseInt(this.strokeColor.slice(3, 5), 16) / 255,
      parseInt(this.strokeColor.slice(5, 7), 16) / 255,
      1.0
    );
    cr.setLineWidth(2.0);
    
    if (normalizedData.length === 1) {
      // Draw a single point
      cr.arc(normalizedData[0].x, normalizedData[0].y, 2, 0, 2 * Math.PI);
      cr.fill();
    } else {
      // Draw the line
      cr.moveTo(normalizedData[0].x, normalizedData[0].y);
      
      for (let i = 1; i < normalizedData.length; i++) {
        cr.lineTo(normalizedData[i].x, normalizedData[i].y);
      }
      cr.stroke();
    }
    
    // Draw points for the last data point
    if (normalizedData.length > 0) {
      const lastPoint = normalizedData[normalizedData.length - 1];
      
      // White center
      cr.setSourceRgba(1, 1, 1, 1);
      cr.arc(lastPoint.x, lastPoint.y, 3, 0, 2 * Math.PI);
      cr.fill();
      
      // Colored border
      cr.setSourceRgba(
        parseInt(this.strokeColor.slice(1, 3), 16) / 255,
        parseInt(this.strokeColor.slice(3, 5), 16) / 255,
        parseInt(this.strokeColor.slice(5, 7), 16) / 255,
        1.0
      );
      cr.setLineWidth(1.5);
      cr.arc(lastPoint.x, lastPoint.y, 3, 0, 2 * Math.PI);
      cr.stroke();
    }
    
    cr.$dispose();
  }

  _drawEmptyState() {
    const cr = this.drawingArea.get_context();
    if (!cr) return;
    
    const width = this.width;
    const height = this.height;
    
    // Draw a subtle line to indicate chart area
    cr.setSourceRgba(0.3, 0.3, 0.3, 0.3); // Very subtle gray
    cr.setLineWidth(1);
    cr.moveTo(20, height - 15);
    cr.lineTo(width - 20, height - 15);
    cr.stroke();
    
    cr.$dispose();
  }

  clear() {
    this.data = [];
    this.drawingArea.queue_repaint();
  }
}
