import St from "gi://St";
import Clutter from "gi://Clutter";
import * as PopupMenu from "resource:///org/gnome/shell/ui/popupMenu.js";
import * as Main from "resource:///org/gnome/shell/ui/main.js";
import { CONFIG } from '../utils/constants.js';
import { DisplaySettingsUI } from './display-settings.js';

export class MenuUI {
  constructor(indicator, settingsManager, onRefresh, onCurrencyChange, onAlertSet, onAlertClear, onDisplayFormatChange) {
    this.indicator = indicator;
    this.settings = settingsManager;
    this.onRefresh = onRefresh;
    this.onCurrencyChange = onCurrencyChange;
    this.onAlertSet = onAlertSet;
    this.onAlertClear = onAlertClear;
    this.onDisplayFormatChange = onDisplayFormatChange;
    
    this.priceItem = null;
    this.alertEntry = null;
    this.alertStatus = null;
    this.currencySubmenu = null;
    this.displaySettings = null;
    
    this._createMenu();
  }

  _createMenu() {
    this._createPriceDisplay();
    this._createAlertSection();
    this._createCurrencySection();
    this._createDisplaySettingsSection();
    this._createRefreshButton();
    this._createDonationSection();
  }

  _createPriceDisplay() {
    this.priceItem = new PopupMenu.PopupMenuItem("Current price: Loading...", {
      reactive: false,
    });
    this.priceItem.label.set_style(
      "color: #ffffff; font-weight: bold; font-size: 14px;",
    );
    this.indicator.menu.addMenuItem(this.priceItem);
    this.indicator.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
  }

  _createAlertSection() {
    // Alert section label
    const alertLabel = new PopupMenu.PopupMenuItem("Set price alert:", {
      reactive: false,
    });
    alertLabel.label.set_style("color: #ffffff; font-weight: bold;");
    this.indicator.menu.addMenuItem(alertLabel);

    // Alert input section
    const alertBox = new St.BoxLayout({
      style_class: "alert-box",
      vertical: false,
    });

    this.alertEntry = new St.Entry({
      hint_text: `Price in ${this.settings.getCurrencySymbol()}`,
      style: "width: 120px; margin: 5px;",
      can_focus: true,
    });

    const setButton = new St.Button({
      label: "Set",
      style: "margin: 5px;",
    });
    setButton.connect("clicked", () => this._handleSetAlert());

    const clearButton = new St.Button({
      label: "Delete",
      style: "margin: 5px;",
    });
    clearButton.connect("clicked", () => this._handleClearAlert());

    alertBox.add_child(this.alertEntry);
    alertBox.add_child(setButton);
    alertBox.add_child(clearButton);

    const alertBoxItem = new PopupMenu.PopupBaseMenuItem({
      reactive: false,
    });
    alertBoxItem.add_child(alertBox);
    this.indicator.menu.addMenuItem(alertBoxItem);

    // Alert status
    this.alertStatus = new PopupMenu.PopupMenuItem("No alert set", {
      reactive: false,
    });
    this.alertStatus.label.set_style("color: #ffffff;");
    this.indicator.menu.addMenuItem(this.alertStatus);
    this.indicator.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
  }

  _createCurrencySection() {
    // Currency selection label
    const currencyLabel = new PopupMenu.PopupMenuItem("Currency:", {
      reactive: false,
    });
    currencyLabel.label.set_style("color: #ffffff; font-weight: bold;");
    this.indicator.menu.addMenuItem(currencyLabel);

    // Currency selection submenu
    this.currencySubmenu = new PopupMenu.PopupSubMenuMenuItem("Select Currency");
    this.currencySubmenu.label.set_style("color: #ffffff;");
    
    const eurItem = new PopupMenu.PopupMenuItem("EUR (€)");
    eurItem.connect("activate", () => this.onCurrencyChange("eur"));
    
    const usdItem = new PopupMenu.PopupMenuItem("USD ($)");
    usdItem.connect("activate", () => this.onCurrencyChange("usd"));
    
    this.currencySubmenu.menu.addMenuItem(eurItem);
    this.currencySubmenu.menu.addMenuItem(usdItem);
    this.indicator.menu.addMenuItem(this.currencySubmenu);
    this.indicator.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());

    // Initialize currency submenu label
    this.currencySubmenu.label.set_text(`Select Currency (${this.settings.getCurrencyDisplay()})`);
  }

  _createRefreshButton() {
    const refreshItem = new PopupMenu.PopupMenuItem("Update now");
    refreshItem.label.set_style("color: #ffffff;");
    refreshItem.connect("activate", () => {
      log("BTC Extension: Manual refresh triggered");
      this.onRefresh();
    });
    this.indicator.menu.addMenuItem(refreshItem);
    this.indicator.menu.addMenuItem(new PopupMenu.PopupSeparatorMenuItem());
  }

  _createDisplaySettingsSection() {
    this.displaySettings = new DisplaySettingsUI(
      this.indicator,
      this.settings,
      this.onDisplayFormatChange,
      this.onChartToggleChange
    );
  }

  _createDonationSection() {
    // Donation section
    const donationLabel = new PopupMenu.PopupMenuItem(
      "⚡ Support the developer",
      {
        reactive: false,
      },
    );
    donationLabel.label.set_style(
      "color: #ffd700; font-weight: bold; font-size: 13px;",
    );
    this.indicator.menu.addMenuItem(donationLabel);

    const donationText = new PopupMenu.PopupMenuItem("Lightning Address:", {
      reactive: false,
    });
    donationText.label.set_style("color: #ffffff; font-size: 11px;");
    this.indicator.menu.addMenuItem(donationText);

    // Clickable lightning address
    const addressItem = new PopupMenu.PopupMenuItem(CONFIG.DONATION.LIGHTNING_ADDRESS);
    addressItem.label.set_style(
      "color: #4da6ff; font-family: monospace; font-size: 11px;",
    );
    addressItem.connect("activate", () => {
      St.Clipboard.get_default().set_text(
        St.ClipboardType.CLIPBOARD,
        CONFIG.DONATION.LIGHTNING_ADDRESS,
      );
      Main.notify("BTC Extension", "Lightning address copied! ⚡");
    });
    this.indicator.menu.addMenuItem(addressItem);
  }

  _handleSetAlert() {
    const text = this.alertEntry.get_text();
    const price = parseFloat(text.replace(",", "."));
    
    const result = this.onAlertSet(price);
    if (result.success) {
      this.alertStatus.label.set_text(result.message);
      this.alertEntry.set_text("");
    } else {
      this.alertStatus.label.set_text(result.message);
    }
  }

  _handleClearAlert() {
    this.onAlertClear();
    this.alertEntry.set_text("");
  }

  updatePriceDisplay(price) {
    if (!price) return;
    
    const priceStr = price.toLocaleString("en-US", {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    });
    
    const currencySymbol = this.settings.getCurrencySymbol();
    this.priceItem.label.set_text(`Current price: ${currencySymbol}${priceStr}`);
  }

  updateAlertStatus(message) {
    this.alertStatus.label.set_text(message);
  }

  updateCurrencyDisplay(currency) {
    this.currencySubmenu.label.set_text(`Select Currency (${currency.toUpperCase()})`);
    this.alertEntry.hint_text = `Price in ${this.settings.getCurrencySymbol()}`;
  }

  getAlertEntryText() {
    return this.alertEntry.get_text();
  }
}
