import St from 'gi://St';
import Soup from 'gi://Soup?version=3.0'; // Helper for Internet calls
import GLib from 'gi://GLib';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import { Extension } from 'resource:///org/gnome/shell/extensions/extension.js';

export default class DictionaryExtension extends Extension {
    enable() {
        // 1. Create a Session for Internet calls
        this._session = new Soup.Session();

        // 2. Create the Button
        this._button = new PanelMenu.Button(0.0, this.metadata.name, false);

        // 3. Add an Icon (Magnifying Glass = Search)
        this._icon = new St.Icon({
            icon_name: 'system-search-symbolic',
            style_class: 'system-status-icon',
        });
        this._button.add_child(this._icon);

        // 4. Handle Click: Grab Selection & Define
        this._button.connect('button-press-event', () => {
            this._lookupSelection();
        });

        // 5. Add to Panel
        Main.panel.addToStatusArea(this.uuid, this._button);
    }

    disable() {
        // Clean up everything
        if (this._button) {
            this._button.destroy();
            this._button = null;
        }
        this._session = null;
    }

    _lookupSelection() {
        // Get the Clipboard system
        let clipboard = St.Clipboard.get_default();
        
        // PRIMARY = The text currently highlighted (not Ctrl+C)
        clipboard.get_text(St.ClipboardType.PRIMARY, (clipboard, text) => {
            if (!text || text.trim().length === 0) {
                Main.notify("Dictionary", "No text selected!");
                return;
            }

            // Clean the text
            let word = text.trim();
            
            // Limit to one word (optional, but APIs usually fail on sentences)
            if (word.includes(" ")) {
                // Take just the first word if they selected a sentence
                word = word.split(" ")[0]; 
            }

            // Visual Feedback: Let user know we are searching
            Main.notify("Searching...", `Looking up: ${word}`);
            
            this._fetchDefinition(word);
        });
    }

    _fetchDefinition(word) {
        let url = `https://api.dictionaryapi.dev/api/v2/entries/en/${word}`;
        let message = Soup.Message.new('GET', url);

        // Send Request
        this._session.send_and_read_async(message, GLib.PRIORITY_DEFAULT, null, (session, result) => {
            try {
                if (message.get_status() === Soup.Status.OK) {
                    let bytes = session.send_and_read_finish(result);
                    let decoder = new TextDecoder('utf-8');
                    let jsonString = decoder.decode(bytes.get_data());
                    let data = JSON.parse(jsonString);

                    if (data.length > 0 && data[0].meanings.length > 0) {
                        // Extract data
                        let partOfSpeech = data[0].meanings[0].partOfSpeech; // noun, verb, etc.
                        let definition = data[0].meanings[0].definitions[0].definition;
                        
                        // Show Result
                        Main.notify(`${word} (${partOfSpeech})`, definition);
                    } else {
                        Main.notify("Dictionary", `No definition found for ${word}`);
                    }
                } else {
                    Main.notify("Dictionary Error", "Word not found or API error.");
                }
            } catch (e) {
                console.error(e);
                Main.notify("Error", "Failed to parse dictionary data.");
            }
        });
    }
}