import St from "gi://St";
import GObject from "gi://GObject";
import GLib from "gi://GLib";
import Clutter from "gi://Clutter";
import { ControlButtons } from "./ControlButtons.js";
import { AlbumArt } from "./AlbumArt.js";
import { ProgressSlider } from "./ProgressSlider.js";
import { PlayerTabs } from "./PlayerTabs.js";

export const MediaControls = GObject.registerClass(
  {
    Signals: {
      "play-pause": {},
      next: {},
      previous: {},
      shuffle: {},
      repeat: {},
      seek: { param_types: [GObject.TYPE_DOUBLE] },
      "player-changed": { param_types: [GObject.TYPE_STRING] },
    },
  },
  class MediaControls extends St.BoxLayout {
    _init() {
      super._init({
        vertical: true,
        style_class: "media-controls-modern",
      });

      this._currentPlayerName = null;
      this._playerStates = new Map();

      this._buildUI();
    }

    _buildUI() {
      const headerBox = new St.BoxLayout({
        style: "margin-bottom: 20px; spacing: 8px;",
        x_align: Clutter.ActorAlign.CENTER,
      });

      this._playerTabs = new PlayerTabs();
      this._playerTabs.connect("player-changed", (_, name) => {
        this._switchToPlayer(name);
      });
      headerBox.add_child(this._playerTabs);
      this.add_child(headerBox);

      this._albumArt = new AlbumArt();
      this.add_child(this._albumArt);

      const infoBox = new St.BoxLayout({
        vertical: true,
        style: "spacing: 6px; margin-bottom: 24px;",
        x_align: Clutter.ActorAlign.CENTER,
      });

      this._titleLabel = new St.Label({
        text: "No media playing",
        style:
          "font-weight: 700; font-size: 16px; color: rgba(255,255,255,0.95);",
      });
      this._titleLabel.clutter_text.ellipsize = 3;
      infoBox.add_child(this._titleLabel);

      this._artistLabel = new St.Label({
        text: "",
        style:
          "font-size: 13px; font-weight: 500; color: rgba(255,255,255,0.6);",
      });
      this._artistLabel.clutter_text.ellipsize = 3;
      infoBox.add_child(this._artistLabel);
      this.add_child(infoBox);

      this._progressSlider = new ProgressSlider();
      this._progressSlider.connect("seek", (_, position) =>
        this.emit("seek", position),
      );
      this._progressSlider.connect("drag-begin", () => {
        this._saveCurrentPlayerState();
        this.stopPositionUpdate();
      });
      this._progressSlider.connect("drag-end", () => {
        this._saveCurrentPlayerState();
      });
      this.add_child(this._progressSlider);

      this._controlButtons = new ControlButtons();
      this._controlButtons.connect("play-pause", () => this.emit("play-pause"));
      this._controlButtons.connect("next", () => this.emit("next"));
      this._controlButtons.connect("previous", () => this.emit("previous"));
      this._controlButtons.connect("shuffle", () => this.emit("shuffle"));
      this._controlButtons.connect("repeat", () => this.emit("repeat"));
      this.add_child(this._controlButtons);
    }

    _switchToPlayer(playerName) {
      if (playerName === this._currentPlayerName) return;

      this._saveCurrentPlayerState();

      this._currentPlayerName = playerName;
      this.emit("player-changed", playerName);

      this._restorePlayerState(playerName);
    }

    _saveCurrentPlayerState() {
      if (!this._currentPlayerName) return;

      const state = {
        position: this._progressSlider.currentPosition,
        length: this._progressSlider.trackLength,
        sliderValue: this._progressSlider.sliderValue,
        isPlaying: this._progressSlider.isPlaying,
        lastUpdateTime: this._progressSlider.lastUpdateTime,
        artUrl: this._playerStates.get(this._currentPlayerName)?.artUrl || null,
        title: this._titleLabel.text,
        artist: this._artistLabel.text,
      };

      this._playerStates.set(this._currentPlayerName, state);
    }

    _restorePlayerState(playerName) {
      const savedState = this._playerStates.get(playerName);

      if (savedState) {
        this._progressSlider.restoreState(
          savedState.position,
          savedState.length,
          savedState.sliderValue,
          savedState.isPlaying,
          savedState.lastUpdateTime,
        );

        if (savedState.artUrl) {
          this._albumArt.loadCover(savedState.artUrl, false);
        } else {
          this._albumArt.setDefaultCover();
        }

        this._titleLabel.text = savedState.title || "Unknown";
        if (savedState.artist) {
          this._artistLabel.text = savedState.artist;
          this._artistLabel.show();
        } else {
          this._artistLabel.hide();
        }
      }
    }

    update(info, playerName, manager) {
      if (!info) return;

      const isPlayerSwitch = this._currentPlayerName !== playerName;

      if (this._currentPlayerName === playerName) {
        this._saveCurrentPlayerState();
      }

      this._currentPlayerName = playerName;

      const currentState = this._playerStates.get(playerName) || {};

      if (this._currentPlayerName === playerName) {
        if (info.artUrl !== currentState.artUrl) {
          if (info.artUrl) {
            this._albumArt.loadCover(info.artUrl, isPlayerSwitch);
          } else {
            this._albumArt.setDefaultCover();
          }
        }

        this._titleLabel.text = info.title || "Unknown";

        if (info.artists && info.artists.length > 0) {
          this._artistLabel.text = info.artists.join(", ");
          this._artistLabel.show();
        } else {
          this._artistLabel.hide();
        }

        this._controlButtons.updateButtons(info);

        if (!this._progressSlider.isDragging) {
          this._progressSlider.updateFromExternalSource(
            info.position || 0,
            info.length || 0,
            info.status === "Playing",
            isPlayerSwitch,
          );
        }
      }

      this._playerStates.set(playerName, {
        position: info.position || 0,
        length: info.length || 0,
        sliderValue: (info.position || 0) / (info.length || 1),
        isPlaying: info.status === "Playing",
        lastUpdateTime: GLib.get_monotonic_time(),
        artUrl: info.artUrl,
        title: info.title,
        artist: info.artists ? info.artists.join(", ") : "",
      });
    }

    updateTabs(players, currentPlayer, manager) {
      this._playerTabs.updateTabs(players, currentPlayer, manager);
    }

    startPositionUpdate() {
      if (this._currentPlayerName) {
        this._progressSlider.startPositionUpdate();
      }
    }

    stopPositionUpdate() {
      this._progressSlider.stopPositionUpdate();
    }

    onSeeked(position) {
      this._progressSlider.onSeeked(position);

      if (this._currentPlayerName) {
        const state = this._playerStates.get(this._currentPlayerName);
        if (state) {
          state.position = position;
          state.lastUpdateTime = GLib.get_monotonic_time();
          state.sliderValue = position / state.length;
        }
      }
    }

    destroy() {
      this.stopPositionUpdate();

      this._playerStates.clear();

      if (this._albumArt) {
        this._albumArt.destroy();
      }

      super.destroy();
    }
  },
);
