import GObject from "gi://GObject";
import St from "gi://St";
import { Extension } from "resource:///org/gnome/shell/extensions/extension.js";
import * as Main from "resource:///org/gnome/shell/ui/main.js";
import { Button } from "resource:///org/gnome/shell/ui/panelMenu.js";
import GLib from "gi://GLib";
import { PopupMenuItem, PopupSeparatorMenuItem } from "resource:///org/gnome/shell/ui/popupMenu.js";
const HOME_DIR = GLib.get_home_dir();
const XDG_CONFIG_HOME = GLib.getenv("XDG_CONFIG_HOME") || HOME_DIR + "/.config";
const CONFIG_PATHS = [
  // === Firefox (native installations) ===
  {
    label: "Firefox",
    path: XDG_CONFIG_HOME + "/mozilla/firefox/profiles.ini",
    command: "firefox"
  },
  {
    label: "Firefox (classic)",
    path: HOME_DIR + "/.mozilla/firefox/profiles.ini",
    command: "firefox"
  },
  // === Firefox (flatpak) ===
  {
    label: "Firefox (flatpak)",
    path: HOME_DIR + "/.var/app/org.mozilla.firefox/.mozilla/firefox/profiles.ini",
    command: "flatpak run org.mozilla.firefox"
  },
  // === Firefox (snap) ===
  {
    label: "Firefox (snap)",
    path: HOME_DIR + "/snap/firefox/common/.mozilla/firefox/profiles.ini",
    command: "snap run firefox"
  },
  // === Waterfox (Firefox fork) ===
  {
    label: "Waterfox",
    path: XDG_CONFIG_HOME + "/waterfox/profiles.ini",
    command: "waterfox"
  },
  {
    label: "Waterfox (classic)",
    path: HOME_DIR + "/.waterfox/profiles.ini",
    command: "waterfox"
  },
  // === LibreWolf (Privacy-focused Firefox) ===
  {
    label: "LibreWolf",
    path: XDG_CONFIG_HOME + "/librewolf/profiles.ini",
    command: "librewolf"
  },
  {
    label: "LibreWolf (classic)",
    path: HOME_DIR + "/.librewolf/profiles.ini",
    command: "librewolf"
  },
  // === Floorp (Firefox fork) ===
  {
    label: "Floorp",
    path: XDG_CONFIG_HOME + "/floorp/profiles.ini",
    command: "floorp"
  },
  {
    label: "Floorp (classic)",
    path: HOME_DIR + "/.floorp/profiles.ini",
    command: "floorp"
  },
  {
    label: "Floorp (flatpak)",
    path: HOME_DIR + "/.var/app/one.ablaze.floorp/.floorp/profiles.ini",
    command: "flatpak run one.ablaze.floorp"
  },
  // === Zen Browser (Gecko-based) ===
  {
    label: "Zen",
    path: XDG_CONFIG_HOME + "/zen/profiles.ini",
    command: "zen-browser"
  },
  {
    label: "Zen (classic)",
    path: HOME_DIR + "/.zen/profiles.ini",
    command: "zen-browser"
  },
  {
    label: "Zen (flatpak)",
    path: HOME_DIR + "/.var/app/app.zen_browser.zen/.zen/profiles.ini",
    command: "flatpak run app.zen_browser.zen"
  },
  // === IceCat (GNU Firefox) ===
  {
    label: "IceCat",
    path: HOME_DIR + "/.icecat/profiles.ini",
    command: "icecat"
  },
  // === Palemoon (legacy fork) ===
  {
    label: "Palemoon",
    path: HOME_DIR + "/.moonchild productions/pale moon/profiles.ini",
    command: "palemoon"
  }
];
function getFirefoxProfiles() {
  const browsers = CONFIG_PATHS.filter(
    (browser) => GLib.file_test(browser.path, GLib.FileTest.EXISTS)
  );
  if (browsers.length === 0) {
    return [];
  }
  return browsers.map(
    (browser) => ({
      ...browser,
      profiles: getProfilesFromConfigFile(browser.path)
    })
  );
}
function getProfilesFromConfigFile(path) {
  let fileContent = GLib.file_get_contents(path)[1];
  let content = fileContent.toString();
  let namePattern = /Name=(.*)/g;
  let profiles = [];
  let match;
  while ((match = namePattern.exec(content)) !== null) {
    profiles.push(match[1]);
  }
  return profiles;
}
function openFirefoxProfile({
  command,
  profile,
  title,
  notify
}) {
  const fullCommand = `${command} -P ${profile} -no-remote`;
  try {
    const success = GLib.spawn_command_line_async(fullCommand);
    if (!success) {
      notify(title, `Failed to start Firefox with the "${profile}" profile.`);
    }
  } catch (e) {
    const message = `An error occurred while launching Firefox with the "${profile}" profile.`;
    logError(e, `[${title}] ${message}`);
    notify(title, message);
  }
}
function fillMenu({
  title,
  menu,
  profiles,
  notify
}) {
  if ("removeAll" in menu) {
    menu.removeAll();
  }
  if ("addMenuItem" in menu) {
    fillProfilesInMenu(menu, profiles, title, notify);
  }
}
function fillProfilesInMenu(menu, profiles, title, notify) {
  if (!("addMenuItem" in menu)) {
    return;
  }
  if (profiles.length === 0) {
    menu.addMenuItem(new PopupMenuItem("No profiles found", { reactive: false }));
    return;
  }
  profiles.forEach((browser) => {
    const section = new PopupSeparatorMenuItem(browser.label);
    menu.addMenuItem(section);
    browser.profiles.forEach((profile) => {
      const item = new PopupMenuItem(profile);
      item.connect(
        "activate",
        () => openFirefoxProfile({
          command: browser.command,
          profile,
          title,
          notify
        })
      );
      menu.addMenuItem(item);
    });
  });
}
class FirefoxProfilesExtension extends Extension {
  _indicator = null;
  enable() {
    this._indicator = new GFirefoxProfilesIndicator(this.metadata.name);
    Main.panel.addToStatusArea(this.uuid, this._indicator);
  }
  disable() {
    if (this._indicator) {
      this._indicator.destroy();
      this._indicator = null;
    }
  }
}
class FirefoxProfilesIndicator extends Button {
  title;
  constructor(title) {
    super(0, title);
    this.title = title;
    this.add_child(
      new St.Icon({
        icon_name: "firefox-symbolic",
        // white version of the Firefox icon
        style_class: "system-status-icon"
      })
    );
    this.refreshProfiles();
  }
  /**
   * Refresh the profiles list and update the menu.
   * This is called on initial load and when the refresh button is clicked.
   */
  refreshProfiles() {
    const profiles = getFirefoxProfiles();
    fillMenu({
      title: this.title,
      menu: this.menu,
      profiles,
      notify: Main.notify
    });
  }
}
const GFirefoxProfilesIndicator = GObject.registerClass(FirefoxProfilesIndicator);
export {
  FirefoxProfilesExtension as default
};
