import St from 'gi://St';
import Clutter from 'gi://Clutter';
import Pango from 'gi://Pango';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import * as PopupMenu from 'resource:///org/gnome/shell/ui/popupMenu.js';
import { Extension } from 'resource:///org/gnome/shell/extensions/extension.js';
import GLib from 'gi://GLib';
import Gio from 'gi://Gio';

export default class NotesExtension extends Extension {

    enable() {
        this._settings = this.getSettings();
        this._notesDir = this._settings.get_string('notes-path') || 
            GLib.build_filenamev([GLib.get_home_dir(), 'NotesExtension']);
        this._lightTheme = this._settings.get_string('theme') === 'light';
        
        this._indicator = new PanelMenu.Button(0.0, this.metadata.name, false);
        this._indicator.add_child(new St.Icon({
            icon_name: 'folder-documents-symbolic',
            style_class: 'system-status-icon',
        }));

        this._indicator.menu.box.style_class = 'notes-menu';
        if (this._lightTheme) this._indicator.menu.box.add_style_class_name('light');

        this._foldersSection = new PopupMenu.PopupMenuSection();
        this._indicator.menu.addMenuItem(this._foldersSection);

        this._folderEntry = new St.Entry({ 
            hint_text: 'New folder...',
            style_class: 'notes-input'
        });
        
        this._folderEntry.clutter_text.connect('activate', () => this._addFolder());

        this._folderInputItem = this._createInputRow(this._folderEntry, 
            this._createButton('Add Folder', 'btn-primary', () => this._addFolder()));
        this._indicator.menu.addMenuItem(this._folderInputItem);

        this._editorSection = new PopupMenu.PopupMenuSection();
        this._indicator.menu.addMenuItem(this._editorSection);

        Main.panel.addToStatusArea(this.uuid, this._indicator);

        this._ensureNotesDir();
        this._refreshFolders();
    }

    _createButton(label, styleClass, callback) {
        let button = new St.Button({ label, style_class: styleClass });
        button.connect('clicked', callback);
        return button;
    }

    _createInputRow(entry, button) {
        let box = new St.BoxLayout({ x_expand: true });
        box.add_child(entry);
        // Spacer pushes button to right edge (x_expand on button itself doesn't work)
        box.add_child(new St.Widget({ x_expand: true }));
        box.add_child(button);
        
        let item = new PopupMenu.PopupBaseMenuItem({ reactive: false });
        item.add_child(box);
        return item;
    }

    _getPath(...parts) {
        return GLib.build_filenamev([this._notesDir, ...parts]);
    }

    _ensureNotesDir() {
        let dir = Gio.File.new_for_path(this._notesDir);
        if (!dir.query_exists(null)) {
            dir.make_directory_with_parents(null);
        }
    }

    _refreshFolders() {
        this._foldersSection.removeAll();

        let enumerator = Gio.File.new_for_path(this._notesDir)
            .enumerate_children('standard::name,standard::type', Gio.FileQueryInfoFlags.NONE, null);

        let info;
        while ((info = enumerator.next_file(null)) !== null) {
            if (info.get_file_type() === Gio.FileType.DIRECTORY) {
                this._addFolderItem(info.get_name());
            }
        }
    }

    _addFolderItem(folderName) {
        let headerBox = new St.BoxLayout({ 
            style: 'spacing: 10px;',
            x_expand: true,
            y_align: Clutter.ActorAlign.CENTER
        });
        
        let nameContainer = new St.BoxLayout({
            style_class: 'folder-name-container',
            x_expand: true,
            y_align: Clutter.ActorAlign.CENTER
        });
        nameContainer.add_child(new St.Label({ text: folderName }));
        
        headerBox.add_child(nameContainer);
        headerBox.add_child(this._createButton('Rename', 'btn-warning',
            () => this._renameItem(folderName, null)));
        headerBox.add_child(this._createButton('Delete', 'btn-danger',
            () => this._deleteFolder(folderName)));
        
        let headerItem = new PopupMenu.PopupBaseMenuItem({ reactive: false });
        headerItem.actor.x_expand = true;
        headerItem.add_child(headerBox);

        // PopupSubMenuMenuItem doesn't allow customizing header layout,
        // so I create with empty label and inject custom header with buttons
        let folderItem = new PopupMenu.PopupSubMenuMenuItem('', { reactive: true });
        folderItem.actor.insert_child_at_index(headerItem.actor, 0);
        folderItem.menu.box.style_class = 'notes-submenu';
        if (this._lightTheme) folderItem.menu.box.add_style_class_name('light');

        let noteEntry = new St.Entry({ 
            hint_text: 'New note...',
            style_class: 'notes-input'
        });
        
        noteEntry.clutter_text.connect('activate', () => this._addNote(folderName, noteEntry));

        folderItem.menu.addMenuItem(this._createInputRow(noteEntry, 
            this._createButton('Add Note', 'btn-primary', () => this._addNote(folderName, noteEntry))));
        
        let notesSection = new PopupMenu.PopupMenuSection();
        this._populateNotes(notesSection, folderName);

        let scrollView = new St.ScrollView({
            hscrollbar_policy: St.PolicyType.NEVER,
            vscrollbar_policy: St.PolicyType.AUTOMATIC,
            style: 'max-height: 300px; min-width: 650px;',
        });
        scrollView.add_child(notesSection.actor);
        
        let scrollItem = new PopupMenu.PopupBaseMenuItem({ reactive: false, can_focus: false });
        scrollItem.add_child(scrollView);
        folderItem.menu.addMenuItem(scrollItem);
        
        this._foldersSection.addMenuItem(folderItem);
    }

    _populateNotes(notesSection, folderName) {
        let enumerator = Gio.File.new_for_path(this._getPath(folderName))
            .enumerate_children('standard::name,standard::type', Gio.FileQueryInfoFlags.NONE, null);

        let info;
        while ((info = enumerator.next_file(null)) !== null) {
            if (info.get_file_type() === Gio.FileType.REGULAR) {
                this._addNoteItem(notesSection, folderName, info.get_name());
            }
        }
    }

    _addNoteItem(notesSection, folderName, noteFile) {
        let displayName = noteFile.endsWith('.txt') ? noteFile.slice(0, -4) : noteFile;
        
        let box = new St.BoxLayout({ 
            style: 'spacing: 10px;',
            x_expand: true,
            y_align: Clutter.ActorAlign.CENTER
        });
        
        let label = new St.Label({ text: displayName, reactive: true });
        label.connect('button-press-event', () => {
            this._showEditor(folderName, noteFile, displayName);
            return Clutter.EVENT_STOP;
        });
        
        box.add_child(label);
        box.add_child(new St.Widget({ x_expand: true }));
        box.add_child(this._createButton('Rename', 'btn-warning',
            () => this._renameItem(folderName, noteFile)));
        box.add_child(this._createButton('Delete', 'btn-danger',
            () => this._deleteNote(folderName, noteFile)));

        let item = new PopupMenu.PopupBaseMenuItem({ reactive: false });
        item.add_child(box);
        notesSection.addMenuItem(item);
    }

    _showEditor(folderName, noteFile, displayName) {
        let content = '';
        try {
            let [, data] = Gio.File.new_for_path(this._getPath(folderName, noteFile)).load_contents(null);
            content = new TextDecoder().decode(data);
        } catch (e) {}

        this._switchToEditor();
        let titleItem = new PopupMenu.PopupMenuItem(displayName, { reactive: false });
        titleItem.actor.add_style_class_name('notes-editor-title');
        this._editorSection.addMenuItem(titleItem);

        let text = new Clutter.Text({
            editable: true,
            single_line_mode: false,
            line_wrap: true,
            line_wrap_mode: Pango.WrapMode.WORD_CHAR,
            text: content,
            reactive: true
        });

        this._setupTextColors(text);
        this._setupKeyBindings(text);

        // Wrap in BoxLayout to capture clicks and focus text
        let layout = new St.BoxLayout({ reactive: true });
        layout.add_child(text);
        layout.connect('button-press-event', () => {
            text.grab_key_focus();
            return Clutter.EVENT_STOP;
        });

        let scroll = new St.ScrollView({
            hscrollbar_policy: St.PolicyType.NEVER,
            vscrollbar_policy: St.PolicyType.AUTOMATIC,
            width: 600,
            height: 300,
            reactive: true
        });
        scroll.add_child(layout);

        let scrollItem = new PopupMenu.PopupBaseMenuItem({ reactive: false });
        scrollItem.add_child(scroll);
        this._editorSection.addMenuItem(scrollItem);

        let buttonBox = new St.BoxLayout({
            style: 'margin-top: 15px; spacing: 10px;',
            x_expand: true
        });
        buttonBox.add_child(new St.Widget({ x_expand: true }));

        buttonBox.add_child(this._createButton('Save', 'btn-primary', () => {
            this._saveNote(this._getPath(folderName, noteFile), text.text);
            this._switchToFolders();
        }));
        buttonBox.add_child(this._createButton('Close', 'btn-danger', () => this._switchToFolders()));

        let buttonItem = new PopupMenu.PopupBaseMenuItem({ reactive: false });
        buttonItem.add_child(buttonBox);
        this._editorSection.addMenuItem(buttonItem);

        text.grab_key_focus();
        this._indicator.menu.open(true);
    }

    _setupTextColors(text) {
        // Workaround: Clutter.Text ignored CSS, so I pass theme colors from St.Entry
        let tempEntry = new St.Entry({ style_class: 'notes-input' });
        this._indicator.menu.box.add_child(tempEntry);

        let themeNode = tempEntry.get_theme_node();
        text.set_color(themeNode.get_foreground_color());

        let bg = themeNode.get_background_color();
        bg.alpha = 180; // 70% opacity for text selection highlight
        text.set_selection_color(bg);

        this._indicator.menu.box.remove_child(tempEntry);
        tempEntry.destroy();
        tempEntry = null;
    }

    _setupKeyBindings(text) {
        if (!this._settings.get_boolean('enable-clipboard-shortcuts')) {
            return; // Skip if user disabled clipboard shortcuts
        }
        text.connect('key-press-event', (_, event) => {
            if (!(event.get_state() & Clutter.ModifierType.CONTROL_MASK)) 
                return Clutter.EVENT_PROPAGATE;

            let key = event.get_key_symbol();
            if (key === Clutter.KEY_c) {
                let cursor = text.get_cursor_position();
                let selectionBound = text.get_selection_bound();
                
                if (cursor !== selectionBound) {
                    let start = Math.min(cursor, selectionBound);
                    let end = Math.max(cursor, selectionBound);
                    let selected = text.get_text().substring(start, end);
                    
                    // Set both CLIPBOARD (Ctrl+V) and PRIMARY (middle-click) for X11 compatibility
                    St.Clipboard.get_default().set_text(St.ClipboardType.CLIPBOARD, selected);
                    St.Clipboard.get_default().set_text(St.ClipboardType.PRIMARY, selected);
                }
                return Clutter.EVENT_STOP;
            }
            if (key === Clutter.KEY_v) {
                St.Clipboard.get_default().get_text(St.ClipboardType.CLIPBOARD, (_, clipText) => {
                    if (clipText) {
                        let pos = text.get_cursor_position();
                        let current = text.get_text();
                        text.set_text(current.slice(0, pos) + clipText + current.slice(pos));
                        let newPos = pos + clipText.length;
                        text.set_cursor_position(newPos);
                        text.set_selection_bound(newPos);
                    }
                });
                return Clutter.EVENT_STOP;
            }
            return Clutter.EVENT_PROPAGATE;
        });
    }

    _deleteFolder(folderName) {
        let dialog = new St.BoxLayout({ vertical: true, style: 'spacing: 10px; padding: 20px;' });
        dialog.add_child(new St.Label({ text: `Delete "${folderName}"?` }));

        let buttons = new St.BoxLayout({ style: 'spacing: 10px;' });
        buttons.add_child(this._createButton('Delete', 'btn-danger', () => {
            let folder = Gio.File.new_for_path(this._getPath(folderName));
            let enumerator = folder.enumerate_children('standard::name', Gio.FileQueryInfoFlags.NONE, null);
            let info;
            while ((info = enumerator.next_file(null)) !== null) {
                folder.get_child(info.get_name()).delete(null);
            }
            folder.delete(null);
            this._switchToFolders();
            this._refreshFolders();
        }));
        buttons.add_child(this._createButton('Cancel', 'btn-primary', () => {
            this._switchToFolders();
        }));

        dialog.add_child(buttons);

        this._switchToEditor();
        let item = new PopupMenu.PopupBaseMenuItem({ reactive: false });
        item.add_child(dialog);
        this._editorSection.addMenuItem(item);
        this._indicator.menu.open(true);
    }

    _saveNote(path, content) {
        Gio.File.new_for_path(path).replace_contents(content, null, false, 
            Gio.FileCreateFlags.REPLACE_DESTINATION, null);
    }

    _deleteNote(folderName, noteFile) {
        Gio.File.new_for_path(this._getPath(folderName, noteFile)).delete(null);
        this._refreshFolders();
    }

    _addFolder() {
        let name = this._folderEntry.get_text().trim();
        if (!name) return;

        let dir = Gio.File.new_for_path(this._getPath(name));
        if (!dir.query_exists(null)) dir.make_directory(null);

        this._folderEntry.set_text('');
        this._refreshFolders();
    }

    _addNote(folderName, entry) {
        let name = entry.get_text().trim();
        if (!name) return;
        
        let file = Gio.File.new_for_path(
            this._getPath(folderName, `${name.replace(/[^a-zA-Z0-9_\- ]/g, '_')}.txt`)
        );

        if (!file.query_exists(null)) {
            file.create(Gio.FileCreateFlags.NONE, null)?.close(null);
        }

        entry.set_text('');
        this._refreshFolders();
    }

    _renameItem(folderName, noteFile) {
        let isNote = noteFile !== null;
        let currentName = isNote ? noteFile.slice(0, -4) : folderName;
        
        let entry = new St.Entry({ text: currentName, style_class: 'notes-input' });
        
        let confirm = this._createButton('Confirm', 'btn-primary', () => {
            let newName = entry.get_text().trim();
            if (newName && newName !== currentName) {
                let oldPath = isNote ? this._getPath(folderName, noteFile) : this._getPath(folderName);
                let finalName = isNote ? `${newName}.txt` : newName;
                Gio.File.new_for_path(oldPath).set_display_name(finalName, null);
                this._refreshFolders();
            }
            this._switchToFolders();
        });

        this._switchToEditor();
        this._editorSection.addMenuItem(this._createInputRow(entry, confirm));
        this._indicator.menu.open(true);
        entry.grab_key_focus();
    }

    _switchToEditor() {
        this._foldersSection.actor.hide();
        this._folderInputItem.actor.hide();
        this._editorSection.removeAll();
    }

    _switchToFolders() {
        this._foldersSection.actor.show();
        this._folderInputItem.actor.show();
        this._editorSection.removeAll();
    }

    disable() {
        Main.panel.statusArea[this.uuid]?.destroy();

        if (this._editorSection) {
            this._editorSection.removeAll();
            this._editorSection.destroy();
            this._editorSection = null;
        }

        if (this._foldersSection) {
            this._foldersSection.removeAll();
            this._foldersSection.destroy();
            this._foldersSection = null;
        }

        if (this._folderInputItem) {
            this._folderInputItem.destroy();
            this._folderInputItem = null;
        }

        if (this._folderEntry) {
            this._folderEntry.destroy();
            this._folderEntry = null;
        }

        if (this._indicator) {
            this._indicator.destroy();
            this._indicator = null;
        }

        this._settings = null;
    }
}