const { Gio, GLib } = imports.gi;
const { main } = imports.ui;
const ExtensionUtils = imports.misc.extensionUtils;
const Me = ExtensionUtils.getCurrentExtension();

const SEARCH_PROVIDER_ID = 'bang-search-provider';

const DEFAULT_BANGS = {
    '!gh': 'https://github.com/search?q=',
    '!so': 'https://stackoverflow.com/search?q=',
    '!yt': 'https://www.youtube.com/results?search_query=',
    '!wiki': 'https://en.wikipedia.org/wiki/Special:Search?search=',
    '!map': 'https://www.google.com/maps/search/',
    '!lc': 'https://lcnum.ishdn.me/'
};

class BangSearchProvider {
    constructor() {
        this._settings = ExtensionUtils.getSettings('org.gnome.shell.extensions.bang-search');

        this._ensureDefaults();
    }

    get appInfo() {
        return {
            get_name: () => `Web Search`,
            get_icon: () => {
                return Gio.icon_new_for_string(`${Me.dir.get_path()}/logo.svg`);
            },
            get_id: () => SEARCH_PROVIDER_ID,
            should_show: () => true,
        };
    }

    _ensureDefaults() {
        const currentBangs = this._settings.get_value('bang-shortcuts').recursiveUnpack();
        if (Object.keys(currentBangs).length === 0) {
            this._settings.set_value('bang-shortcuts', new GLib.Variant('a{ss}', DEFAULT_BANGS));
        }
    }

    get _customBangs() {
        const bangDict = this._settings.get_value('bang-shortcuts').recursiveUnpack();
        if (Object.keys(bangDict).length === 0) {
            return DEFAULT_BANGS;
        }
        return bangDict;
    }

    getResultMeta(id) {
        const { type, answer, query } = JSON.parse(id);

        if (type === "bang") {
            let hostname = 'url';
            try {
                const urlObj = new URL(answer);
                hostname = urlObj.hostname;
            } catch (e) {
                // If URL parsing fails, we just fallback to generic text
            }

            return {
                id,
                name: hostname !== 'url' ? `Search ${hostname} for ${query}` : `Search for ${query}`,
                description: `!bang Redirect`,
                clipboardText: answer,
                createIcon: () => null,
            };
        }

        return {
            id,
            name: answer,
            description: `Search Google`,
            clipboardText: answer,
            createIcon: () => null,
        };
    }

    getResultMetas(results, cb) {
        cb(results.map(this.getResultMeta.bind(this)));
    }

    activateResult(result) {
        const { type, answer } = JSON.parse(result);
        let url = "";

        if (type === "query") {
            url = `https://google.com/search?q=${answer}`;
        } else if (type === "bang") {
            url = answer;
        }

        if (url) {
            const context = global.create_app_launch_context(0, -1);
            Gio.app_info_launch_default_for_uri(url, context);
        }
    }

    filterResults(providerResults, maxResults) {
        return providerResults.slice(0, maxResults);
    }

    getInitialResultSet(terms, cb) {
        const query = terms.join(" ");
        const results = [JSON.stringify({ type: "query", answer: query })];

        if (query.startsWith("!")) {
            const bangCommand = terms[0].toLowerCase();
            const searchQuery = terms.slice(1).join(" ");

            const customBangs = this._customBangs;

            if (customBangs[bangCommand]) {
                const redirectUrl = customBangs[bangCommand] + encodeURIComponent(searchQuery);
                results.unshift(JSON.stringify({
                    type: "bang",
                    answer: redirectUrl,
                    query: searchQuery,
                }));
            }
        }

        cb(results);
    }

    getSubsearchResultSet(_, terms, cb) {
        this.getInitialResultSet(terms, cb);
    }
}

let providerInstance = null;

function init() {
}

function getOverviewSearchResult() {
    return main.overview.viewSelector
        ? main.overview.viewSelector._searchResults
        : main.overview._overview.controls._searchController._searchResults;
}

function enable() {
    if (!providerInstance) {
        providerInstance = new BangSearchProvider();
    }
    getOverviewSearchResult()._registerProvider(providerInstance);
}

function disable() {
    if (providerInstance) {
        getOverviewSearchResult()._unregisterProvider(providerInstance);
        providerInstance = null;
    }
}