'use strict';

imports.gi.versions.Gtk = '4.0';
const { Gtk, Gio, GLib, GObject } = imports.gi;
const ExtensionUtils = imports.misc.extensionUtils;

function getSettings() {
    return ExtensionUtils.getSettings('org.gnome.shell.extensions.bang-search');
}

function loadBangs() {
    const value = getSettings().get_value('bang-shortcuts');
    return value.recursiveUnpack();
}

function saveBangs(bangs) {
    getSettings().set_value('bang-shortcuts', new GLib.Variant('a{ss}', bangs));
    return true;
}

function init() {
}

function buildPrefsWidget() {
    const widget = new Gtk.Box({
        orientation: Gtk.Orientation.VERTICAL,
        spacing: 12,
        margin_top: 20,
        margin_bottom: 20,
        margin_start: 20,
        margin_end: 20,
    });

    const headerLabel = new Gtk.Label({
        label: '<b>Custom Bang Shortcuts</b>',
        use_markup: true,
        halign: Gtk.Align.START,
    });
    widget.append(headerLabel);

    const descLabel = new Gtk.Label({
        label: 'Configure custom bang shortcuts (e.g., !gh for GitHub search)',
        halign: Gtk.Align.START,
        css_classes: ['dim-label'],
    });
    widget.append(descLabel);

    const scrolledWindow = new Gtk.ScrolledWindow({
        vexpand: true,
        hexpand: true,
        min_content_height: 300,
    });

    const listBox = new Gtk.ListBox({
        selection_mode: Gtk.SelectionMode.NONE,
        css_classes: ['boxed-list'],
    });
    scrolledWindow.set_child(listBox);
    widget.append(scrolledWindow);

    let bangs = loadBangs();

    function refreshList() {
        let child = listBox.get_first_child();
        while (child) {
            const next = child.get_next_sibling();
            listBox.remove(child);
            child = next;
        }

        for (const [shortcut, url] of Object.entries(bangs)) {
            const row = createBangRow(shortcut, url);
            listBox.append(row);
        }
    }

    function createBangRow(shortcut, url) {
        const row = new Gtk.ListBoxRow({
            activatable: false,
        });

        const box = new Gtk.Box({
            orientation: Gtk.Orientation.HORIZONTAL,
            spacing: 10,
            margin_top: 8,
            margin_bottom: 8,
            margin_start: 12,
            margin_end: 12,
        });

        const shortcutEntry = new Gtk.Entry({
            text: shortcut,
            placeholder_text: '!bang',
            width_chars: 12,
            tooltip_text: 'Bang shortcut (e.g., !gh)',
        });

        const urlEntry = new Gtk.Entry({
            text: url,
            placeholder_text: 'https://example.com/search?q=',
            hexpand: true,
            tooltip_text: 'URL template (query will be appended)',
        });

        const updateButton = new Gtk.Button({
            icon_name: 'document-save-symbolic',
            tooltip_text: 'Save changes',
            css_classes: ['flat'],
        });

        const deleteButton = new Gtk.Button({
            icon_name: 'user-trash-symbolic',
            tooltip_text: 'Delete this bang',
            css_classes: ['flat', 'destructive-action'],
        });

        updateButton.connect('clicked', () => {
            const newShortcut = shortcutEntry.get_text().trim().toLowerCase();
            const newUrl = urlEntry.get_text().trim();

            if (!newShortcut || !newUrl) {
                return;
            }

            // Ensure shortcut starts with !
            const finalShortcut = newShortcut.startsWith('!') ? newShortcut : '!' + newShortcut;

            // Remove old entry if shortcut changed
            if (finalShortcut !== shortcut) {
                delete bangs[shortcut];
            }

            bangs[finalShortcut] = newUrl;
            saveBangs(bangs);
            refreshList();
        });

        deleteButton.connect('clicked', () => {
            delete bangs[shortcut];
            saveBangs(bangs);
            refreshList();
        });

        box.append(shortcutEntry);
        box.append(urlEntry);
        box.append(updateButton);
        box.append(deleteButton);
        row.set_child(box);

        return row;
    }

    const addBox = new Gtk.Box({
        orientation: Gtk.Orientation.HORIZONTAL,
        spacing: 10,
        css_classes: ['toolbar'],
    });

    const newShortcutEntry = new Gtk.Entry({
        placeholder_text: '!bang',
        width_chars: 12,
        tooltip_text: 'New bang shortcut',
    });

    const newUrlEntry = new Gtk.Entry({
        placeholder_text: 'https://example.com/search?q=',
        hexpand: true,
        tooltip_text: 'URL template (query will be appended)',
    });

    const addButton = new Gtk.Button({
        label: 'Add Bang',
        css_classes: ['suggested-action'],
    });

    addButton.connect('clicked', () => {
        let shortcut = newShortcutEntry.get_text().trim().toLowerCase();
        const url = newUrlEntry.get_text().trim();

        if (!shortcut || !url) {
            return;
        }

        // Ensure shortcut starts with !
        if (!shortcut.startsWith('!')) {
            shortcut = '!' + shortcut;
        }

        bangs[shortcut] = url;
        saveBangs(bangs);

        newShortcutEntry.set_text('');
        newUrlEntry.set_text('');

        refreshList();
    });

    addBox.append(newShortcutEntry);
    addBox.append(newUrlEntry);
    addBox.append(addButton);
    widget.append(addBox);

    const infoLabel = new Gtk.Label({
        label: '<i>Note: Changes are saved automatically to GSettings.</i>',
        use_markup: true,
        halign: Gtk.Align.START,
        css_classes: ['dim-label'],
        margin_top: 10,
    });
    widget.append(infoLabel);

    refreshList();

    return widget;
}
