// extension.js - Main extension logic with idle monitoring
import {Extension} from 'resource:///org/gnome/shell/extensions/extension.js';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';

import {ProcessManager} from './lib/processManager.js';
import {discoverThemes} from './lib/themeManager.js';

export default class PlymouthScreensaver extends Extension {
    constructor(metadata) {
        super(metadata);
        this._settings = null;
        this._processManager = null;
        this._idleMonitor = null;
        this._idleWatchId = null;
        this._activeWatchId = null;
        this._isScreensaverRunning = false;
        this._sessionModeChangedId = null;
        this._settingsChangedIds = [];
        this._isEnabled = true;
    }

    enable() {
        // Load settings
        this._settings = this.getSettings();

        // Initialize process manager
        this._processManager = new ProcessManager(this.path);

        // Get idle monitor (GNOME 45+ uses backend.get_core_idle_monitor())
        this._idleMonitor = global.backend.get_core_idle_monitor();

        // Set up idle monitoring
        this._setupIdleMonitoring();

        // Watch for session mode changes (e.g., lock screen)
        this._sessionModeChangedId = Main.sessionMode.connect('updated', () => {
            this._onSessionModeChanged();
        });

        // Watch for settings changes
        this._settingsChangedIds.push(
            this._settings.connect('changed::enabled', () => {
                this._onEnabledChanged();
            })
        );

        this._settingsChangedIds.push(
            this._settings.connect('changed::idle-time-minutes', () => {
                this._onIdleTimeChanged();
            })
        );

        // Check if extension is enabled in settings
        this._isEnabled = this._settings.get_boolean('enabled');
    }

    disable() {

        // Remove idle watches
        if (this._idleWatchId) {
            this._idleMonitor.remove_watch(this._idleWatchId);
            this._idleWatchId = null;
        }

        if (this._activeWatchId) {
            this._idleMonitor.remove_watch(this._activeWatchId);
            this._activeWatchId = null;
        }

        // Terminate any running screensaver
        if (this._processManager) {
            this._processManager.terminate();
            this._processManager.destroy();
            this._processManager = null;
        }

        // Disconnect session mode signal
        if (this._sessionModeChangedId) {
            Main.sessionMode.disconnect(this._sessionModeChangedId);
            this._sessionModeChangedId = null;
        }

        // Disconnect settings signals
        for (const id of this._settingsChangedIds) {
            this._settings.disconnect(id);
        }
        this._settingsChangedIds = [];

        // Clean up references
        this._idleMonitor = null;
        this._settings = null;
        this._isScreensaverRunning = false;
    }

    _setupIdleMonitoring() {
        // Remove existing watches
        if (this._idleWatchId) {
            this._idleMonitor.remove_watch(this._idleWatchId);
            this._idleWatchId = null;
        }

        if (this._activeWatchId) {
            this._idleMonitor.remove_watch(this._activeWatchId);
            this._activeWatchId = null;
        }

        // Only set up monitoring if enabled
        if (!this._isEnabled) {
            return;
        }

        // Get idle time from settings (convert minutes to milliseconds)
        const idleTimeMinutes = this._settings.get_uint('idle-time-minutes');
        const idleTimeMs = idleTimeMinutes * 60 * 1000;

        // Add idle watch (fires once when idle)
        this._idleWatchId = this._idleMonitor.add_idle_watch(idleTimeMs, () => {
            this._onIdleDetected();
        });

        // Add active watch (fires on every activity, persists automatically)
        this._activeWatchId = this._idleMonitor.add_user_active_watch(() => {
            this._onUserActive();
        });
    }

    _onIdleDetected() {
        // Check if screensaver is already running
        if (this._isScreensaverRunning) {
            this._reAddIdleWatch();
            return;
        }

        // Check if we're in a session mode where screensaver shouldn't run
        const restrictedModes = ['unlock-dialog', 'lock-screen', 'gdm', 'initial-setup'];
        if (restrictedModes.includes(Main.sessionMode.currentMode)) {
            this._reAddIdleWatch();
            return;
        }

        // Get theme - either random or selected
        let themePath, themeName;
        const useRandomTheme = this._settings.get_boolean('random-theme');

        if (useRandomTheme) {
            const themes = discoverThemes();
            if (themes.length > 0) {
                const randomIndex = Math.floor(Math.random() * themes.length);
                const randomTheme = themes[randomIndex];
                themePath = randomTheme.themePath;
                themeName = randomTheme.name;
            } else {
                // Fallback to selected theme if no themes found
                themePath = this._settings.get_string('selected-theme-path');
                themeName = this._settings.get_string('selected-theme-name');
            }
        } else {
            themePath = this._settings.get_string('selected-theme-path');
            themeName = this._settings.get_string('selected-theme-name');
        }

        // Build settings object
        const settings = {
            fps: this._settings.get_double('fps'),
            scale: this._settings.get_int('scale'),
            backend: this._settings.get_string('backend')
        };

        // Launch screensaver
        const success = this._processManager.launch(themePath, () => {
            this._isScreensaverRunning = false;
        }, false, settings);

        if (success) {
            this._isScreensaverRunning = true;
        }

        // Re-add idle watch for next cycle (idle watch fires only once)
        this._reAddIdleWatch();
    }

    _onUserActive() {
        // If screensaver is running, terminate it
        if (this._isScreensaverRunning) {
            this._processManager.terminate();
            this._isScreensaverRunning = false;
        }

        // Re-add the user active watch (it's one-shot)
        this._activeWatchId = this._idleMonitor.add_user_active_watch(() => {
            this._onUserActive();
        });
    }

    _reAddIdleWatch() {
        // Remove old watch
        if (this._idleWatchId) {
            this._idleMonitor.remove_watch(this._idleWatchId);
        }

        // Add new watch
        const idleTimeMinutes = this._settings.get_uint('idle-time-minutes');
        const idleTimeMs = idleTimeMinutes * 60 * 1000;

        this._idleWatchId = this._idleMonitor.add_idle_watch(idleTimeMs, () => {
            this._onIdleDetected();
        });
    }

    _onSessionModeChanged() {
        // If we're in a restricted mode (e.g., lock screen), terminate screensaver
        const restrictedModes = ['unlock-dialog', 'lock-screen', 'gdm', 'initial-setup'];
        if (restrictedModes.includes(Main.sessionMode.currentMode) && this._isScreensaverRunning) {
            this._processManager.terminate();
            this._isScreensaverRunning = false;
        }
    }

    _onEnabledChanged() {
        this._isEnabled = this._settings.get_boolean('enabled');

        if (this._isEnabled) {
            // Re-enable idle monitoring
            this._setupIdleMonitoring();
        } else {
            // Disable idle monitoring and terminate screensaver
            if (this._idleWatchId) {
                this._idleMonitor.remove_watch(this._idleWatchId);
                this._idleWatchId = null;
            }
            if (this._activeWatchId) {
                this._idleMonitor.remove_watch(this._activeWatchId);
                this._activeWatchId = null;
            }
            if (this._isScreensaverRunning) {
                this._processManager.terminate();
                this._isScreensaverRunning = false;
            }
        }
    }

    _onIdleTimeChanged() {
        this._setupIdleMonitoring();
    }
}
