// processManager.js - Process lifecycle management for plymouth_saver.py
import GLib from 'gi://GLib';
import Gio from 'gi://Gio';

export class ProcessManager {
    constructor(extensionPath) {
        this._extensionPath = extensionPath;
        this._process = null;
        this._pid = null;
        this._watchId = null;
    }

    /**
     * Launch plymouth_saver.py with specified theme
     * @param {string} themePath - Full path to Plymouth theme directory
     * @param {Function} callback - Callback function when process exits
     * @param {boolean} windowed - Launch in windowed mode (for preview)
     * @param {Object} settings - Optional settings object with fps, scale, backend
     * @returns {boolean} True if launch successful
     */
    launch(themePath, callback, windowed = false, settings = null) {
        // Clean up any existing process
        if (this._process) {
            this.terminate();
        }

        try {
            const scriptPath = GLib.build_filenamev([this._extensionPath, 'plymouth_saver.py']);

            // Build command arguments
            const args = ['python3', scriptPath, '--theme-dir', themePath];

            // Add backend if specified
            if (settings && settings.backend) {
                args.push('--backend', settings.backend);
            } else {
                args.push('--backend', 'auto');
            }

            // Add FPS if specified
            if (settings && settings.fps) {
                args.push('--fps', settings.fps.toString());
            }

            // Add scale if specified
            if (settings && settings.scale) {
                args.push('--scale', settings.scale.toString());
            }

            // Add windowed mode flags for preview
            if (windowed) {
                args.push('--windowed');
                args.push('--duration-seconds', '5');
            }


            // Launch process using Gio.Subprocess
            this._process = Gio.Subprocess.new(
                args,
                Gio.SubprocessFlags.STDOUT_PIPE | Gio.SubprocessFlags.STDERR_PIPE
            );

            // Get PID
            this._pid = this._process.get_identifier();

            // Watch for process exit
            this._process.wait_async(null, (proc, result) => {
                try {
                    proc.wait_finish(result);
                } catch (e) {
                    // Process wait failed
                }

                // Clean up
                this._process = null;
                this._pid = null;

                // Call callback if provided
                if (callback) {
                    callback();
                }
            });

            return true;

        } catch (e) {
            this._process = null;
            this._pid = null;
            return false;
        }
    }

    /**
     * Terminate the running process
     * @returns {boolean} True if termination initiated
     */
    terminate() {
        if (!this._process) {
            return false;
        }

        try {
            // Send SIGTERM (15) for graceful shutdown
            this._process.send_signal(15);

            // Set up a timeout to force kill if needed
            GLib.timeout_add(GLib.PRIORITY_DEFAULT, 2000, () => {
                if (this._process) {
                    try {
                        this._process.force_exit();
                    } catch (e) {
                        // Force exit failed
                    }
                    this._process = null;
                    this._pid = null;
                }
                return GLib.SOURCE_REMOVE;
            });

            return true;

        } catch (e) {
            this._process = null;
            this._pid = null;
            return false;
        }
    }

    /**
     * Launch in preview mode (windowed, 5-second duration)
     * @param {string} themePath - Full path to Plymouth theme directory
     * @param {Function} callback - Callback function when preview exits
     * @param {Object} settings - Optional settings object
     * @returns {boolean} True if launch successful
     */
    preview(themePath, callback, settings = null) {
        return this.launch(themePath, callback, true, settings);
    }

    /**
     * Check if a process is currently running
     * @returns {boolean} True if process is running
     */
    isRunning() {
        return this._process !== null;
    }

    /**
     * Get the current process PID
     * @returns {number|null} PID or null if not running
     */
    getPid() {
        return this._pid;
    }

    /**
     * Clean up resources
     */
    destroy() {
        this.terminate();
        this._extensionPath = null;
    }
}
