// themeManager.js - Plymouth theme discovery and parsing
import GLib from 'gi://GLib';
import Gio from 'gi://Gio';

const PLYMOUTH_THEMES_DIR = '/usr/share/plymouth/themes';

// Text-only theme modules to filter out
const TEXT_ONLY_MODULES = [
    'text',
    'tribar',
    'details',
    'ubuntu-text',
    'fade-throbber',
    'space-flares'
];

// Animated theme modules to include
const ANIMATED_MODULES = [
    'script',
    'two-step'
];

/**
 * Discover all animated Plymouth themes
 * @returns {Array} Array of theme objects
 */
export function discoverThemes() {
    const themes = [];

    try {
        const themesDir = Gio.File.new_for_path(PLYMOUTH_THEMES_DIR);

        if (!themesDir.query_exists(null)) {
            return themes;
        }

        const enumerator = themesDir.enumerate_children(
            'standard::name,standard::type',
            Gio.FileQueryInfoFlags.NONE,
            null
        );

        let info;
        while ((info = enumerator.next_file(null)) !== null) {
            if (info.get_file_type() === Gio.FileType.DIRECTORY) {
                const themeName = info.get_name();
                const themePath = GLib.build_filenamev([PLYMOUTH_THEMES_DIR, themeName]);

                const theme = _parseTheme(themePath, themeName);
                if (theme && theme.isAnimated && !theme.isText) {
                    themes.push(theme);
                }
            }
        }
    } catch (e) {
        // Theme discovery failed
    }

    // Sort themes alphabetically by name
    themes.sort((a, b) => a.name.localeCompare(b.name));

    return themes;
}

/**
 * Parse a Plymouth theme directory
 * @param {string} path - Full path to theme directory
 * @param {string} themeName - Theme directory name
 * @returns {Object|null} Theme object or null if invalid
 */
function _parseTheme(path, themeName) {
    try {
        // Look for .plymouth file
        const plymouthFile = GLib.build_filenamev([path, `${themeName}.plymouth`]);
        const file = Gio.File.new_for_path(plymouthFile);

        if (!file.query_exists(null)) {
            return null;
        }

        // Parse the .plymouth INI file
        const keyFile = new GLib.KeyFile();
        keyFile.load_from_file(plymouthFile, GLib.KeyFileFlags.NONE);

        const SECTION = 'Plymouth Theme';

        // Extract theme metadata
        let name = themeName;
        let description = '';
        let moduleName = '';
        let imageDir = '';

        try {
            name = keyFile.get_string(SECTION, 'Name');
        } catch (e) {
            // Use directory name as fallback
        }

        try {
            description = keyFile.get_string(SECTION, 'Description');
        } catch (e) {
            description = '';
        }

        try {
            moduleName = keyFile.get_string(SECTION, 'ModuleName');
        } catch (e) {
            return null;
        }

        try {
            imageDir = keyFile.get_string(SECTION, 'ImageDir');
        } catch (e) {
            // ImageDir is optional, defaults to theme directory
            imageDir = path;
        }

        // Determine if theme is animated or text-only
        const isAnimated = ANIMATED_MODULES.includes(moduleName);
        const isText = TEXT_ONLY_MODULES.includes(moduleName);

        // Find first frame for thumbnail
        const firstFramePath = _findFirstFrame(path, imageDir);

        return {
            name: name,
            description: description,
            moduleName: moduleName,
            themePath: path,
            themeName: themeName,
            firstFramePath: firstFramePath,
            isAnimated: isAnimated,
            isText: isText
        };

    } catch (e) {
        return null;
    }
}

/**
 * Find the first animation frame for thumbnail
 * @param {string} themePath - Theme directory path
 * @param {string} imageDir - Image directory path (may be relative)
 * @returns {string|null} Path to first frame or null
 */
function _findFirstFrame(themePath, imageDir) {
    // Resolve image directory
    let imgDir = imageDir;
    if (!GLib.path_is_absolute(imageDir)) {
        imgDir = GLib.build_filenamev([themePath, imageDir]);
    }

    const imgDirFile = Gio.File.new_for_path(imgDir);
    if (!imgDirFile.query_exists(null)) {
        imgDir = themePath; // Fallback to theme directory
    }

    // Common frame naming patterns
    const patterns = [
        'progress-0.png',
        'progress-00.png',
        'progress-000.png',
        'animation-0001.png',
        'animation-0000.png',
        'animation-001.png',
        'animation-01.png',
        'animation-1.png',
        'throbber-0.png',
        'throbber-00.png',
        'throbber-01.png',
        'spinner-0.png',
        'frame-0.png',
        'frame-00.png',
        'frame-001.png',
        'boot-0.png',
        // Fallback to any PNG
        '*.png'
    ];

    for (const pattern of patterns) {
        if (pattern.includes('*')) {
            // Glob pattern - find first PNG
            try {
                const dir = Gio.File.new_for_path(imgDir);
                const enumerator = dir.enumerate_children(
                    'standard::name,standard::type',
                    Gio.FileQueryInfoFlags.NONE,
                    null
                );

                let info;
                while ((info = enumerator.next_file(null)) !== null) {
                    const fileName = info.get_name();
                    if (fileName.endsWith('.png') && info.get_file_type() === Gio.FileType.REGULAR) {
                        return GLib.build_filenamev([imgDir, fileName]);
                    }
                }
            } catch (e) {
                continue;
            }
        } else {
            // Exact file check
            const framePath = GLib.build_filenamev([imgDir, pattern]);
            const file = Gio.File.new_for_path(framePath);
            if (file.query_exists(null)) {
                return framePath;
            }
        }
    }

    return null;
}
