#!/usr/bin/env python3
import argparse
import glob
import os
import re
import signal
import sys
import time


FRAME_RE = re.compile(r"^(?P<prefix>.+)-(?P<num>\d+)\.png$")


def find_background(theme_dir):
    candidates = [
        "background.png",
        "background.jpg",
        "background.jpeg",
        "bgrt-fallback.png",
        "watermark.png",
    ]
    for name in candidates:
        path = os.path.join(theme_dir, name)
        if os.path.isfile(path):
            return path
    return None


def find_frames(theme_dir):
    groups = {}
    for name in os.listdir(theme_dir):
        match = FRAME_RE.match(name)
        if not match:
            continue
        prefix = match.group("prefix")
        num = int(match.group("num"))
        groups.setdefault(prefix, []).append((num, name))
    if not groups:
        return []
    best_prefix = max(groups.keys(), key=lambda p: len(groups[p]))
    frames = sorted(groups[best_prefix], key=lambda item: item[0])
    return [os.path.join(theme_dir, name) for _, name in frames]


def run_tkinter(frame_paths, background_path, args):
    try:
        import tkinter as tk
    except ModuleNotFoundError as exc:
        raise RuntimeError("tkinter is not available") from exc

    root = tk.Tk()
    root.configure(bg="black")
    root.attributes("-topmost", True)
    if not args.windowed:
        root.attributes("-fullscreen", True)
    root.bind("<Escape>", lambda event: root.destroy())
    root.bind("q", lambda event: root.destroy())
    root.bind("<Button-1>", lambda event: root.destroy())

    screen_w = root.winfo_screenwidth()
    screen_h = root.winfo_screenheight()
    canvas = tk.Canvas(root, width=screen_w, height=screen_h, highlightthickness=0)
    canvas.pack(fill="both", expand=True)

    bg_image = None
    if background_path:
        bg_image = tk.PhotoImage(file=background_path)
        canvas.create_image(
            screen_w // 2, screen_h // 2, image=bg_image, anchor="center"
        )
    else:
        canvas.configure(bg="black")

    frames = []
    for path in frame_paths:
        img = tk.PhotoImage(file=path)
        if args.scale > 1:
            img = img.zoom(args.scale, args.scale)
        frames.append(img)

    x = screen_w // 2 + args.offset_x
    y = screen_h // 2 + args.offset_y
    frame_item = canvas.create_image(x, y, image=frames[0], anchor="center")

    delay_ms = max(1, int(1000 / max(1.0, args.fps)))
    state = {"index": 0}
    start_time = time.perf_counter()

    def tick():
        if args.duration_seconds > 0:
            if time.perf_counter() - start_time >= args.duration_seconds:
                root.destroy()
                return
        state["index"] = (state["index"] + 1) % len(frames)
        canvas.itemconfigure(frame_item, image=frames[state["index"]])
        root.after(delay_ms, tick)

    def handle_sigterm(_signum, _frame):
        root.destroy()

    signal.signal(signal.SIGTERM, handle_sigterm)
    root.after(delay_ms, tick)
    root.mainloop()
    return 0


def run_pygame(frame_paths, background_path, args):
    try:
        import pygame
    except ModuleNotFoundError as exc:
        raise RuntimeError("pygame is not available") from exc

    pygame.init()
    pygame.display.set_caption("plymouth_saver")

    first = pygame.image.load(frame_paths[0])
    frame_w = first.get_width()
    frame_h = first.get_height()
    scaled_w = frame_w * max(1, args.scale)
    scaled_h = frame_h * max(1, args.scale)

    if args.windowed:
        size = (max(scaled_w, 640), max(scaled_h, 480))
        flags = 0
    else:
        size = (0, 0)
        flags = pygame.FULLSCREEN

    screen = pygame.display.set_mode(size, flags)
    screen_w, screen_h = screen.get_size()

    def load_frame(path):
        img = pygame.image.load(path).convert_alpha()
        if args.scale > 1:
            img = pygame.transform.scale(
                img, (img.get_width() * args.scale, img.get_height() * args.scale)
            )
        return img

    frames = [load_frame(path) for path in frame_paths]

    background = None
    if background_path:
        bg = pygame.image.load(background_path).convert()
        background = pygame.transform.smoothscale(bg, (screen_w, screen_h))

    # Create black overlay for fade effects
    overlay = pygame.Surface((screen_w, screen_h))
    overlay.fill((0, 0, 0))

    x = screen_w // 2 + args.offset_x
    y = screen_h // 2 + args.offset_y
    running = True
    start_time = time.perf_counter()
    clock = pygame.time.Clock()

    # Hide mouse cursor
    pygame.mouse.set_visible(False)

    while running:
        for event in pygame.event.get():
            if event.type == pygame.QUIT:
                running = False
            elif event.type == pygame.KEYDOWN:
                if event.key in (pygame.K_ESCAPE, pygame.K_q):
                    running = False
            elif event.type == pygame.MOUSEBUTTONDOWN:
                running = False

        if args.duration_seconds > 0:
            if time.perf_counter() - start_time >= args.duration_seconds:
                running = False

        elapsed = time.perf_counter() - start_time

        # Draw background or black
        if background:
            screen.blit(background, (0, 0))
        else:
            screen.fill((0, 0, 0))

        # Draw animation frame
        frame_index = int(elapsed * max(1.0, args.fps)) % len(frames)
        frame = frames[frame_index]
        rect = frame.get_rect(center=(x, y))
        screen.blit(frame, rect)

        # Fade in from black
        if args.fade_in_seconds > 0 and elapsed < args.fade_in_seconds:
            alpha = max(0, int(255 * (1 - elapsed / args.fade_in_seconds)))
            overlay.set_alpha(alpha)
            screen.blit(overlay, (0, 0))

        pygame.display.flip()
        clock.tick(120)

    pygame.quit()
    return 0


def main():
    parser = argparse.ArgumentParser(
        description="Fullscreen plymouth theme frame player for screensavers."
    )
    parser.add_argument(
        "--theme-dir",
        default=None,
        help="Path to a plymouth theme directory.",
    )
    parser.add_argument(
        "--frames",
        default=None,
        help="Glob pattern for frames; overrides theme discovery.",
    )
    parser.add_argument("--fps", type=float, default=30.0, help="Frames per second.")
    parser.add_argument(
        "--scale",
        type=int,
        default=1,
        help="Integer scale factor for frames.",
    )
    parser.add_argument(
        "--windowed",
        action="store_true",
        help="Run in a window for testing.",
    )
    parser.add_argument("--offset-x", type=int, default=0, help="Frame X offset.")
    parser.add_argument("--offset-y", type=int, default=0, help="Frame Y offset.")
    parser.add_argument(
        "--duration-seconds",
        type=float,
        default=0.0,
        help="Auto-exit after N seconds; 0 disables.",
    )
    parser.add_argument(
        "--fade-in-seconds",
        type=float,
        default=3.0,
        help="Duration to fade in animation from black (default: 3).",
    )
    parser.add_argument(
        "--backend",
        choices=("auto", "pygame", "tkinter"),
        default="auto",
        help="Rendering backend (default: auto).",
    )
    args = parser.parse_args()

    theme_dir = args.theme_dir
    if theme_dir is None:
        default_dir = "/usr/share/plymouth/themes/owl"
        theme_dir = default_dir if os.path.isdir(default_dir) else os.getcwd()

    if args.frames:
        frame_paths = sorted(glob.glob(args.frames))
        if not frame_paths and theme_dir:
            frame_paths = sorted(glob.glob(os.path.join(theme_dir, args.frames)))
    else:
        frame_paths = find_frames(theme_dir)

    if not frame_paths:
        print("No frame sequence found.", file=sys.stderr)
        return 1

    background_path = find_background(theme_dir)

    backend = args.backend
    if backend == "auto":
        try:
            import pygame  # noqa: F401

            backend = "pygame"
        except ModuleNotFoundError:
            backend = "tkinter"

    if backend == "pygame":
        try:
            return run_pygame(frame_paths, background_path, args)
        except RuntimeError as exc:
            print(str(exc), file=sys.stderr)
            return 1

    try:
        return run_tkinter(frame_paths, background_path, args)
    except RuntimeError as exc:
        print(str(exc), file=sys.stderr)
        return 1


if __name__ == "__main__":
    raise SystemExit(main())
