// prefs.js - Preferences window with Gtk4/Adw
import {ExtensionPreferences} from 'resource:///org/gnome/Shell/Extensions/js/extensions/prefs.js';
import Adw from 'gi://Adw';
import Gtk from 'gi://Gtk';
import Gdk from 'gi://Gdk';
import GLib from 'gi://GLib';
import GdkPixbuf from 'gi://GdkPixbuf';
import Gio from 'gi://Gio';

import {discoverThemes} from './lib/themeManager.js';
import {ProcessManager} from './lib/processManager.js';

export default class PlymouthScreensaverPreferences extends ExtensionPreferences {
    fillPreferencesWindow(window) {
        const settings = this.getSettings();
        const processManager = new ProcessManager(this.path);

        // Create main preferences page
        const page = new Adw.PreferencesPage();
        window.add(page);

        // General settings group
        const generalGroup = new Adw.PreferencesGroup({
            title: 'General',
            description: 'Basic screensaver settings'
        });
        page.add(generalGroup);

        // Enable/disable switch
        const enabledRow = new Adw.SwitchRow({
            title: 'Enable Extension',
            subtitle: 'Turn the Plymouth screensaver on or off'
        });
        settings.bind('enabled', enabledRow, 'active', Gio.SettingsBindFlags.DEFAULT);
        generalGroup.add(enabledRow);

        // Idle time spin row
        const idleTimeRow = new Adw.SpinRow({
            title: 'Idle Time',
            subtitle: 'Minutes of inactivity before screensaver activates',
            adjustment: new Gtk.Adjustment({
                lower: 1,
                upper: 60,
                step_increment: 1
            })
        });
        settings.bind('idle-time-minutes', idleTimeRow, 'value', Gio.SettingsBindFlags.DEFAULT);
        generalGroup.add(idleTimeRow);

        // Random theme switch
        const randomThemeRow = new Adw.SwitchRow({
            title: 'Random Theme',
            subtitle: 'Choose a random theme each time the screensaver activates'
        });
        settings.bind('random-theme', randomThemeRow, 'active', Gio.SettingsBindFlags.DEFAULT);
        generalGroup.add(randomThemeRow);

        // Theme selection group
        const themeGroup = new Adw.PreferencesGroup({
            title: 'Theme Selection',
            description: 'Choose a Plymouth boot animation theme'
        });
        page.add(themeGroup);

        // Discover themes
        const themes = discoverThemes();

        if (themes.length === 0) {
            const noThemesLabel = new Gtk.Label({
                label: 'No animated Plymouth themes found in /usr/share/plymouth/themes/',
                wrap: true,
                margin_top: 12,
                margin_bottom: 12,
                margin_start: 12,
                margin_end: 12
            });
            noThemesLabel.add_css_class('dim-label');
            themeGroup.add(noThemesLabel);
        } else {
            // Theme dropdown
            const themeRow = new Adw.ComboRow({
                title: 'Theme',
                subtitle: 'Select a Plymouth animation theme'
            });

            const themeModel = new Gtk.StringList();
            for (const theme of themes) {
                themeModel.append(theme.name);
            }
            themeRow.set_model(themeModel);

            // Find and set current selection
            const selectedThemePath = settings.get_string('selected-theme-path');
            let selectedIndex = 0;
            for (let i = 0; i < themes.length; i++) {
                if (themes[i].themePath === selectedThemePath) {
                    selectedIndex = i;
                    break;
                }
            }
            themeRow.set_selected(selectedIndex);

            themeGroup.add(themeRow);

            // Preview container
            const previewBox = new Gtk.Box({
                orientation: Gtk.Orientation.HORIZONTAL,
                spacing: 16,
                margin_top: 16,
                margin_bottom: 8,
                margin_start: 12,
                margin_end: 12,
                halign: Gtk.Align.CENTER
            });
            themeGroup.add(previewBox);

            // Large preview image using Gtk.Picture for proper scaling
            const previewImage = new Gtk.Picture({
                width_request: 256,
                height_request: 256,
                valign: Gtk.Align.CENTER,
                content_fit: Gtk.ContentFit.CONTAIN,  // Scale to fit while preserving aspect ratio
            });
            previewImage.add_css_class('card');
            previewBox.append(previewImage);

            // Right side: description and test button
            const rightBox = new Gtk.Box({
                orientation: Gtk.Orientation.VERTICAL,
                spacing: 12,
                valign: Gtk.Align.CENTER
            });
            previewBox.append(rightBox);

            // Description label
            const descLabel = new Gtk.Label({
                label: '',
                xalign: 0,
                wrap: true,
                max_width_chars: 30
            });
            descLabel.add_css_class('dim-label');
            rightBox.append(descLabel);

            // Test button
            const testButton = new Gtk.Button({
                label: 'Test Theme',
                halign: Gtk.Align.START
            });
            testButton.add_css_class('pill');
            testButton.add_css_class('suggested-action');
            rightBox.append(testButton);

            // Function to update preview
            const updatePreview = (index) => {
                const theme = themes[index];
                if (theme) {
                    // Update preview image
                    if (theme.firstFramePath) {
                        try {
                            // Load original image
                            const original = GdkPixbuf.Pixbuf.new_from_file(theme.firstFramePath);

                            // Create texture from pixbuf - Gtk.Picture will handle scaling
                            const texture = Gdk.Texture.new_for_pixbuf(original);
                            previewImage.set_paintable(texture);
                        } catch (e) {
                            previewImage.set_paintable(null);
                        }
                    } else {
                        previewImage.set_paintable(null);
                    }

                    // Update description
                    descLabel.set_label(theme.description || 'No description available');
                }
            };

            // Initial preview update
            updatePreview(selectedIndex);

            // Connect dropdown change signal
            themeRow.connect('notify::selected', () => {
                const index = themeRow.get_selected();
                const theme = themes[index];
                if (theme) {
                    settings.set_string('selected-theme-path', theme.themePath);
                    settings.set_string('selected-theme-name', theme.name);
                    updatePreview(index);
                }
            });

            // Connect test button
            testButton.connect('clicked', () => {
                const index = themeRow.get_selected();
                const theme = themes[index];
                if (theme) {
                    const previewSettings = {
                        fps: settings.get_double('fps'),
                        scale: settings.get_int('scale'),
                        backend: settings.get_string('backend')
                    };
                    processManager.preview(theme.themePath, null, previewSettings);
                }
            });
        }

        // Advanced settings group
        const advancedGroup = new Adw.PreferencesGroup({
            title: 'Advanced',
            description: 'Fine-tune animation settings'
        });
        page.add(advancedGroup);

        // FPS spin row
        const fpsRow = new Adw.SpinRow({
            title: 'Frames Per Second',
            subtitle: 'Animation frame rate',
            adjustment: new Gtk.Adjustment({
                lower: 1,
                upper: 120,
                step_increment: 1
            }),
            digits: 0
        });
        settings.bind('fps', fpsRow, 'value', Gio.SettingsBindFlags.DEFAULT);
        advancedGroup.add(fpsRow);

        // Scale spin row
        const scaleRow = new Adw.SpinRow({
            title: 'Scale Factor',
            subtitle: 'Integer scale multiplier for animation',
            adjustment: new Gtk.Adjustment({
                lower: 1,
                upper: 8,
                step_increment: 1
            })
        });
        settings.bind('scale', scaleRow, 'value', Gio.SettingsBindFlags.DEFAULT);
        advancedGroup.add(scaleRow);

        // Backend combo row
        const backendRow = new Adw.ComboRow({
            title: 'Rendering Backend',
            subtitle: 'Choose between auto-detection, pygame, or tkinter'
        });

        const backendModel = new Gtk.StringList();
        backendModel.append('Auto');
        backendModel.append('Pygame');
        backendModel.append('Tkinter');
        backendRow.set_model(backendModel);

        // Set initial value
        const backend = settings.get_string('backend');
        if (backend === 'auto') {
            backendRow.set_selected(0);
        } else if (backend === 'pygame') {
            backendRow.set_selected(1);
        } else if (backend === 'tkinter') {
            backendRow.set_selected(2);
        }

        // Connect signal
        backendRow.connect('notify::selected', () => {
            const selected = backendRow.get_selected();
            if (selected === 0) {
                settings.set_string('backend', 'auto');
            } else if (selected === 1) {
                settings.set_string('backend', 'pygame');
            } else if (selected === 2) {
                settings.set_string('backend', 'tkinter');
            }
        });

        advancedGroup.add(backendRow);

        // Clean up on window close
        window.connect('close-request', () => {
            processManager.destroy();
        });
    }

}
