/* -*- mode: js2; js2-basic-offset: 4; indent-tabs-mode: nil -*- */

// Strict mode.
"use strict";

import GLib from "gi://GLib";
import GObject from "gi://GObject";
import Gio from "gi://Gio";

import * as Main from "resource:///org/gnome/shell/ui/main.js";
import * as PopupMenu from "resource:///org/gnome/shell/ui/popupMenu.js";
import * as QuickSettings from "resource:///org/gnome/shell/ui/quickSettings.js";
import * as GnomeSession from "resource:///org/gnome/shell/misc/gnomeSession.js";

import { Switch as PrimeSwitch } from "./prime.js";
import * as Icons from "./icons.js";
import * as Log from "./log.js";

/**
 * Prime Indicator Toggle for Quick Settings.
 */
export const Toggle = GObject.registerClass(
  {
    GTypeName: "PrimeIndicatorMenuToggle",
  },
  class Toggle extends QuickSettings.QuickMenuToggle {
    /**
     * Constructor.
     *
     * @param  {Extension} extension
     * @return {Void}
     */
    _init(extension) {
      super._init({
        title: extension.gettext("Prime Select"),
        toggleMode: false,
      });

      this._extension = extension;
      this._settings = extension.getSettings();
      this._switch = new PrimeSwitch();
      this._timeoutSourceId = null;
      this._pending = false;
      this._loggingOut = false;

      // Set icon
      this.gicon = new Icons.Icon(extension.path);

      // Setup menu
      this._setupMenu();

      // Monitor GPU changes
      this._switch.connect("gpu-change", this._handlePrimeGpuChange.bind(this));
      this._switch.monitor();

      // Log warnings
      let switches = this._switch.switches;
      if (!this._switch.command("sudo"))
        this._log("can't find sudo frontend command, switch disabled");
      if (!this._switch.command("select"))
        this._log("can't find prime-select command, query/switch disabled");
      if (!this._switch.command("management"))
        this._log(
          "can't find nvidia-smi command, logout notification disabled",
        );
      if (!this._switch.command("settings"))
        this._log("can't find nvidia-settings command, settings disabled");
      if (!switches.length)
        this._log("can't find any prime switch, select disabled");

      this._refresh();
    }

    /**
     * Proxy for Log.journal.
     *
     * @param  {...String} message
     * @return {Void}
     */
    _log(...message) {
      Log.journal("Menu.Widget", ...message);
    }

    /**
     * Setup menu items.
     *
     * @return {Void}
     */
    _setupMenu() {
      const _ = this._extension.gettext.bind(this._extension);
      this._ui = {};
      let switches = this._switch.switches;

      // Header
      this.menu.setHeader(
        new Icons.Icon(this._extension.path),
        _("Prime Select"),
        null,
      );

      // GPU options
      if (switches.includes("intel")) {
        this._ui.intel = new PopupMenu.PopupMenuItem(_("Intel"));
        this._ui.intel.connect(
          "activate",
          this._handleMenuItemClick.bind(this),
        );
        this.menu.addMenuItem(this._ui.intel);
      }
      if (switches.includes("nvidia")) {
        this._ui.nvidia = new PopupMenu.PopupMenuItem(_("NVidia"));
        this._ui.nvidia.connect(
          "activate",
          this._handleMenuItemClick.bind(this),
        );
        this.menu.addMenuItem(this._ui.nvidia);
      }
      if (switches.includes("on-demand")) {
        this._ui.demand = new PopupMenu.PopupMenuItem(_("NVidia On-Demand"));
        this._ui.demand.connect(
          "activate",
          this._handleMenuItemClick.bind(this),
        );
        this.menu.addMenuItem(this._ui.demand);
      }

      // Separator
      this._ui.separator = new PopupMenu.PopupSeparatorMenuItem();
      this.menu.addMenuItem(this._ui.separator);

      // Status messages
      this._ui.messagePending = new PopupMenu.PopupMenuItem(
        _("Please wait for the operation\nto complete"),
      );
      this._ui.messagePending.setSensitive(false);
      this.menu.addMenuItem(this._ui.messagePending);

      this._ui.messageRestart = new PopupMenu.PopupMenuItem(
        _("Please log out and log back in\nto apply the changes"),
      );
      this._ui.messageRestart.setSensitive(false);
      this.menu.addMenuItem(this._ui.messageRestart);

      this._ui.messageLoggingOut = new PopupMenu.PopupMenuItem(
        _("Logging out..."),
      );
      this._ui.messageLoggingOut.setSensitive(false);
      this.menu.addMenuItem(this._ui.messageLoggingOut);
    }

    /**
     * Settings property getter.
     *
     * @return {Gio.Settings}
     */
    get settings() {
      return this._settings;
    }

    /**
     * Switch property getter.
     *
     * @return {Prime.Switch}
     */
    get switch() {
      return this._switch;
    }

    /**
     * Logout gnome session.
     *
     * @return {Void}
     */
    logout() {
      let sessionManager = new GnomeSession.SessionManager();
      let mode = 1;
      // 0: Normal.
      // 1: No confirmation interface should be shown.
      // 2: Forcefully logout. No confirmation will be shown and any inhibitors will be ignored.

      this._log("gnome session logout");
      sessionManager.LogoutRemote(mode);
    }

    /**
     * Clear any main loop timeout sources.
     *
     * @return {Void}
     */
    _delayClear() {
      if (!this._timeoutSourceId) return;

      GLib.Source.remove(this._timeoutSourceId);
      this._timeoutSourceId = null;
    }

    /**
     * Execute callback with delay.
     *
     * @param  {Number}   timeout
     * @param  {Function} callback
     * @param  {...Mixed} args
     * @return {Void}
     */
    _delayExecute(timeout, callback, ...args) {
      if (this._timeoutSourceId) throw new Error("Timeout already in use");

      this._timeoutSourceId = GLib.timeout_add(
        GLib.PRIORITY_DEFAULT,
        timeout,
        () => {
          this._timeoutSourceId = null;

          if (typeof callback === "function") callback.apply(this, args);

          // Stop repeating.
          return GLib.SOURCE_REMOVE;
        },
      );
    }

    /**
     * Refresh widget menu:
     * set items sensitivity and show/hide logout message.
     *
     * @return {Void}
     */
    _refresh() {
      let query = this._switch.query;
      let pending = this._pending;
      let loggingOut = this._loggingOut;
      let isRestartNeeded = this._switch.isRestartNeeded;
      let sensitive =
        !pending && !loggingOut
          ? this._switch.command("sudo") && this._switch.command("select")
          : false;

      // Update subtitle with current GPU
      this.subtitle =
        query !== "unknown"
          ? query.charAt(0).toUpperCase() + query.slice(1)
          : "";

      if (this._ui.nvidia) {
        this._ui.nvidia.setSensitive(sensitive);
        this._ui.nvidia.setOrnament(
          query === "nvidia"
            ? PopupMenu.Ornament.CHECK
            : PopupMenu.Ornament.NONE,
        );
      }
      if (this._ui.intel) {
        this._ui.intel.setSensitive(sensitive);
        this._ui.intel.setOrnament(
          query === "intel"
            ? PopupMenu.Ornament.CHECK
            : PopupMenu.Ornament.NONE,
        );
      }
      if (this._ui.demand) {
        this._ui.demand.setSensitive(sensitive);
        this._ui.demand.setOrnament(
          query === "on-demand"
            ? PopupMenu.Ornament.CHECK
            : PopupMenu.Ornament.NONE,
        );
      }

      this._ui.separator.visible = pending || isRestartNeeded || loggingOut;
      this._ui.messagePending.visible = loggingOut ? false : pending;
      this._ui.messageRestart.visible =
        loggingOut || pending ? false : isRestartNeeded;
      this._ui.messageLoggingOut.visible = loggingOut;
    }

    /**
     * Switch GPU.
     *
     * @param  {String} gpu
     * @return {Void}
     */
    _switchGpu(gpu) {
      this._pending = true;
      this._refresh();

      this._switch.switch(gpu, (e) => {
        let doRestart =
          e.result &&
          this._settings.get_boolean("auto-logout") &&
          this._switch.isRestartNeeded;
        if (!doRestart) {
          this._pending = false;
          this._refresh();
          return;
        }

        this._log("logout on gpu switch enabled, logging out");
        this._pending = false;
        this._loggingOut = true;
        this._refresh();

        // Logout with delay.
        this._delayExecute(1000, this.logout.bind(this));
      });
    }

    /**
     * Settings changed event handler.
     *
     * @param  {Object} actor
     * @param  {String} key
     * @return {Void}
     */
    _handleSettings(actor, key) {
      // pass
    }

    /**
     * Menu item click event handler.
     *
     * @param  {PopupMenuItem} actor
     * @param  {Clutter.Event} event
     * @return {Void}
     */
    _handleMenuItemClick(actor, event) {
      if (actor._ornament !== PopupMenu.Ornament.NONE) return;

      let gpu = null;
      if (this._ui.nvidia && this._ui.nvidia === actor) gpu = "nvidia";
      else if (this._ui.intel && this._ui.intel === actor) gpu = "intel";
      else if (this._ui.demand && this._ui.demand === actor) gpu = "on-demand";
      else throw new Error("Unknown GPU switch");

      // Switch with delay, making sure that refresh occurs after menu fadeout.
      this._delayExecute(50, this._switchGpu.bind(this), gpu);
    }

    /**
     * Prime switch gpu change event handler.
     *
     * @param  {Object} actor
     * @param  {String} gpu
     * @return {Void}
     */
    _handlePrimeGpuChange(actor, gpu) {
      this._refresh();
    }

    /**
     * Destructor.
     *
     * @return {Void}
     */
    destroy() {
      this._delayClear();

      if (this._switch) {
        this._switch.destroy();
        this._switch = null;
      }

      if (this._settings) {
        this._settings = null;
      }

      super.destroy();
    }
  },
);

/**
 * System Indicator for the panel.
 */
export const Indicator = GObject.registerClass(
  {
    GTypeName: "PrimeIndicatorMenuIndicator",
  },
  class Indicator extends QuickSettings.SystemIndicator {
    /**
     * Constructor.
     *
     * @param  {Extension} extension
     * @return {Void}
     */
    _init(extension) {
      super._init();

      this._extension = extension;

      // Create the toggle
      this._toggle = new Toggle(extension);

      // Add toggle to quick settings menu
      this.quickSettingsItems.push(this._toggle);
    }

    /**
     * Destructor.
     *
     * @return {Void}
     */
    destroy() {
      this._toggle?.destroy();
      this._toggle = null;
      super.destroy();
    }
  },
);
