/* -*- mode: js2; js2-basic-offset: 4; indent-tabs-mode: nil -*- */

// Strict mode.
"use strict";

import Gtk from "gi://Gtk";
import Gdk from "gi://Gdk";
import GLib from "gi://GLib";
import Gio from "gi://Gio";

/**
 * Default icon.
 *
 * @type {String}
 */
const DEFAULT_ICON = "prime-menu-default-symbolic";

/**
 * Preferences widget.
 */
export class Widget {
  /**
   * Constructor.
   *
   * @param  {Adw.PreferencesWindow} window
   * @param  {Extension}             extension
   * @return {Void}
   */
  constructor(window, extension) {
    this._window = window;
    this._extension = extension;
    this._settings = extension.getSettings();
    this._builder = Gtk.Builder.new_from_file(
      `${extension.path}/libs/prefs/widget.ui`,
    );

    this._nVidiaSettingsTimeoutId = null;

    this._loadStylesheet();
    this._includeIcons();
    this._buildUI();

    this.window.connect(
      "close-request",
      this._handleWindowCloseRequest.bind(this),
    );
  }

  /**
   * Destructor.
   *
   * @return {Void}
   */
  destroy() {
    if (this._nVidiaSettingsTimeoutId) {
      GLib.Source.remove(this._nVidiaSettingsTimeoutId);
      this._nVidiaSettingsTimeoutId = null;
    }

    this._builder = null;
    this._settings = null;
    this._extension = null;
    this._window = null;
  }

  /**
   * Window property getter.
   *
   * @return {Adw.PreferencesWindow}
   */
  get window() {
    return this._window;
  }

  /**
   * Settings property getter.
   *
   * @return {Gio.Settings}
   */
  get settings() {
    return this._settings;
  }

  /**
   * Builder property getter.
   *
   * @return {Gtk.Builder}
   */
  get builder() {
    return this._builder;
  }

  /**
   * Load stylesheet from CSS file.
   *
   * @return {Void}
   */
  _loadStylesheet() {
    try {
      const provider = new Gtk.CssProvider();
      provider.load_from_path(`${this._extension.path}/libs/prefs/widget.css`);

      Gtk.StyleContext.add_provider_for_display(
        Gdk.Display.get_default(),
        provider,
        Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION,
      );
    } catch (e) {
      console.error(`[prime-indicator] Failed to load stylesheet: ${e}`);
    }
  }

  /**
   * Include icons.
   *
   * @return {Void}
   */
  _includeIcons() {
    try {
      let theme = Gtk.IconTheme.get_for_display(Gdk.Display.get_default());
      let iconPath = `${this._extension.path}/icons`;
      if (!theme.get_search_path().includes(iconPath)) {
        theme.add_search_path(iconPath);
      }
    } catch (e) {
      console.error(`[prime-indicator] Failed to include icons: ${e}`);
    }
  }

  /**
   * Build UI.
   *
   * @return {Void}
   */
  _buildUI() {
    this._buildUIWindow();
    this._buildUIAddPages();
    this._buildUISettings();
    this._buildUIActions();
    this._buildUIAbout();
  }

  /**
   * Build UI:
   * set window size and enable search.
   *
   * @return {Void}
   */
  _buildUIWindow() {
    let width = this.settings.get_int("prefs-default-width");
    let height = this.settings.get_int("prefs-default-height");
    if (width && height) this.window.set_default_size(width, height);

    this.window.set_search_enabled(true);
  }

  /**
   * Build UI:
   * add all pages to window.
   *
   * @return {Void}
   */
  _buildUIAddPages() {
    let settingsPage = this._findChild("page-settings");
    let aboutPage = this._findChild("page-about");

    if (settingsPage) this.window.add(settingsPage);
    if (aboutPage) this.window.add(aboutPage);
  }

  /**
   * Build UI:
   * sync settings and its widgets.
   *
   * @return {Void}
   */
  _buildUISettings() {
    this._registerSettingWidget("auto-logout");
  }

  /**
   * Build UI:
   * bind action widgets.
   *
   * @return {Void}
   */
  _buildUIActions() {
    let nvidiaSettingsButton = this._findChild("action-nvidia-settings");
    if (nvidiaSettingsButton) {
      nvidiaSettingsButton.connect(
        "clicked",
        this._handleNVidiaSettingsButtonClick.bind(this),
      );
    }
  }

  /**
   * Build UI:
   * sync about page with metadata.
   *
   * @return {Void}
   */
  _buildUIAbout() {
    const _ = this._extension.gettext.bind(this._extension);
    let metadata = this._extension.metadata;
    let url = this._getMetadataProperty(metadata, "url");
    let webpage = `<a href="${url}">${url}</a>`;
    let gnomeVersion = this._getGnomeVersion();
    let sessionType = this._getSessionType();

    let aboutIcon = this._findChild("about-icon");
    let aboutTitle = this._findChild("about-title");
    let aboutDescription = this._findChild("about-description");
    let aboutVersion = this._findChild("about-version");
    let aboutGnome = this._findChild("about-gnome");
    let aboutSession = this._findChild("about-session");
    let aboutAuthor = this._findChild("about-author");
    let aboutWebpage = this._findChild("about-webpage");
    let aboutDonation = this._findChild("about-donation");
    let aboutLicense = this._findChild("about-license");

    if (aboutIcon) aboutIcon.set_from_icon_name(DEFAULT_ICON);
    if (aboutTitle)
      aboutTitle.set_label(
        this._getMetadataProperty(metadata, "name") || "Prime Indicator",
      );
    if (aboutDescription)
      aboutDescription.set_label(
        this._labelMarkupFix(
          this._getMetadataProperty(
            metadata,
            "description-html",
            "description",
          ),
        ),
      );
    if (aboutVersion)
      aboutVersion.set_label(
        String(this._getMetadataProperty(metadata, "version") || ""),
      );
    if (aboutGnome) aboutGnome.set_label(gnomeVersion);
    if (aboutSession) aboutSession.set_label(sessionType);
    if (aboutAuthor)
      aboutAuthor.set_label(
        this._labelMarkupFix(
          this._getMetadataProperty(
            metadata,
            "original-author-html",
            "original-author",
          ),
        ),
      );
    if (aboutWebpage) aboutWebpage.set_label(webpage);
    if (aboutDonation)
      aboutDonation.set_label(
        this._labelMarkupFix(
          this._getMetadataProperty(metadata, "donation-html", "donation"),
        ),
      );
    if (aboutLicense)
      aboutLicense.set_label(
        this._labelMarkupFix(
          this._getMetadataProperty(metadata, "license-html", "license"),
        ),
      );
  }

  /**
   * Register setting widget:
   * find widget, set its value (from settings) and bind change to store
   * value in our schema.
   *
   * @param  {String} property
   * @return {Void}
   */
  _registerSettingWidget(property) {
    const widget = this._findChild("setting-" + property);
    if (!widget)
      throw new Error(
        "Widget: can not register setting widget (unknown property).",
      );

    if (widget instanceof Gtk.Switch) {
      widget.set_active(this.settings.get_boolean(property));
      this.settings.bind(
        property,
        widget,
        "active",
        Gio.SettingsBindFlags.DEFAULT,
      );
    } else
      throw new Error(
        "Widget: can not register setting (unsupported widget type).",
      );
  }

  /**
   * Find child (builder object) by name.
   *
   * @param  {String}          name
   * @return {Gtk.Widget|Null}
   */
  _findChild(name) {
    return this.builder.get_object(name);
  }

  /**
   * Label markup fix:
   * since Gtk.Label markup doesn't support <br> tags, we're gonna replace
   * all line break tags with newlines.
   *
   * @param  {String} text
   * @return {String}
   */
  _labelMarkupFix(text) {
    return (text || "").replace(/<br\s*(?:\/?)\s*>/g, "\n");
  }

  /**
   * Get first truthy property from metadata.
   *
   * @param  {Object}    metadata
   * @param  {...String} props
   * @return {String}
   */
  _getMetadataProperty(metadata, ...props) {
    return props.reduce(
      (carry, item) => carry || (metadata[item] || ""),
      "",
    );
  }

  /**
   * Get GNOME version.
   *
   * @return {String}
   */
  _getGnomeVersion() {
    const _ = this._extension.gettext.bind(this._extension);
    try {
      const [, stdout] = GLib.spawn_command_line_sync("gnome-shell --version");
      if (stdout) {
        const decoder = new TextDecoder();
        const version = decoder.decode(stdout).trim();
        const match = version.match(/(\d+\.?\d*\.?\d*)/);
        if (match) {
          return match[1];
        }
      }
    } catch (e) {
      // Fallback
    }
    return _("unknown");
  }

  /**
   * Get session type (wayland, X11 or unknown).
   *
   * @return {String}
   */
  _getSessionType() {
    const _ = this._extension.gettext.bind(this._extension);
    return GLib.getenv("XDG_SESSION_TYPE") || _("unknown");
  }

  /**
   * Shell execute command.
   *
   * @param  {String}      command
   * @return {Number|Null}
   */
  _shellExecAsync(command) {
    try {
      let [ok, pid] = GLib.spawn_async(
        null,
        command.split(" "),
        null,
        GLib.SpawnFlags.SEARCH_PATH,
        null,
      );
      if (ok) return pid;
    } catch (e) {
      // pass
    }

    return null;
  }

  /**
   * Window close request event handler:
   * store window size to schema and destroy instance.
   *
   * @param  {Adw.PreferencesWindow} widget
   * @return {Boolean}
   */
  _handleWindowCloseRequest(widget) {
    if (widget.default_width !== this.settings.get_int("prefs-default-width"))
      this.settings.set_int("prefs-default-width", widget.default_width);
    if (widget.default_height !== this.settings.get_int("prefs-default-height"))
      this.settings.set_int("prefs-default-height", widget.default_height);

    this.destroy();
    return false;
  }

  /**
   * NVidia settings button click event handler.
   *
   * @param  {Gtk.Button} widget
   * @return {Void}
   */
  _handleNVidiaSettingsButtonClick(widget) {
    this._shellExecAsync("nvidia-settings");

    // This can last second or two, so let's disable widget for a moment
    // to prevent multiple button clicks.
    if (this._nVidiaSettingsTimeoutId) {
      GLib.Source.remove(this._nVidiaSettingsTimeoutId);
    }

    widget.set_sensitive(false);
    this._nVidiaSettingsTimeoutId = GLib.timeout_add(
      GLib.PRIORITY_DEFAULT,
      2500,
      () => {
        widget.set_sensitive(true);
        this._nVidiaSettingsTimeoutId = null;
        return GLib.SOURCE_REMOVE;
      },
    );
  }
}
