/* extension.js
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

import GObject from 'gi://GObject';
import St from 'gi://St';
import GLib from 'gi://GLib';
import Gio from 'gi://Gio';
import Clutter from 'gi://Clutter';

import {Extension, gettext as _} from 'resource:///org/gnome/shell/extensions/extension.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';

const Indicator = GObject.registerClass(
class Indicator extends PanelMenu.Button {
    _init() {
        super._init(0.0, _('Days Since Update'));

        this._label = new St.Label({ text: '', y_align: Clutter.ActorAlign.CENTER });
        this.add_child(this._label);
        this._updateLabel();

        // Update every minute
        this._timeout = GLib.timeout_add_seconds(GLib.PRIORITY_DEFAULT, 60, () => {
            this._updateLabel();
            return GLib.SOURCE_CONTINUE;
        });
    }

    _getLastUpdateDays() {
        try {
            const logFile = Gio.File.new_for_path('/var/log/pacman.log');
            const [success, contents] = logFile.load_contents(null);
            if (!success) return null;

            const decoder = new TextDecoder('utf-8');
            const text = decoder.decode(contents);
            const lines = text.split('\n');

            for (let i = lines.length - 1; i >= 0; i--) {
                const line = lines[i];
                if (line.includes('pacman -Syu') || line.includes('pacman -Su')) {
                    const match = line.match(/\[(\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2})/);
                    if (match) return (new Date() - new Date(match[1])) / 864e5 | 0;
                }
            }
            return null;
        } catch (e) {
            logError(e, 'Error getting days since update');
            return null;
        }
    }

    _updateLabel() {
        const days = this._getLastUpdateDays();
        this._label.set_text(days === null ? "?" : `${days}d`);
        this._label.style = days >= 10 ? 'color: #ffdd66;' : '';
    }

    destroy() {
        if (this._timeout) {
            GLib.source_remove(this._timeout);
            this._timeout = null;
        }
        super.destroy();
    }
});

export default class DaysExtension extends Extension {
    enable() {
        this._indicator = new Indicator();
        Main.panel.addToStatusArea(this.uuid, this._indicator);
    }

    disable() {
        this._indicator.destroy();
        this._indicator = null;
    }
}
