/*
 * ClipMaster - Database Manager
 * License: GPL-2.0-or-later
 */

import GLib from 'gi://GLib';
import Gio from 'gi://Gio';

import { gettext as _ } from 'resource:///org/gnome/shell/extensions/extension.js';

import { TimeoutManager, FileUtils, HashUtils } from '../Util/Utils.js';
import { ItemType, debugLog } from '../Util/Constants.js';
import { SimpleEncryption } from '../Util/Encryption.js';
import { ImageStorage } from '../Util/ImageStorage.js';

export class ClipboardDatabase {
    constructor(storagePath, settings, onNotification = null) {
        this._storagePath = storagePath || GLib.build_filenamev([
            GLib.get_user_data_dir(), 'clipmaster', 'clipboard.json'
        ]);
        this._imagesStoragePath = this._storagePath.endsWith('.json')
            ? this._storagePath.replace(/\.json$/, '.images.json')
            : `${this._storagePath}.images.json`;
        this._settings = settings;
        this._onNotification = onNotification;

        this._items = [];
        this._lists = [];
        this._nextId = 1;
        this._isDirty = false;

        this._isLoaded = false;
        this._pendingItems = [];

        this._lastWarningTime = 0;
        this._warningCooldownMs = 60000;
        this._isCleaning = false;

        this._timeoutManager = new TimeoutManager();
        this._saveDebounceMs = 500;

        this._encryption = null;

        // Fast duplicate lookup: contentHash -> itemId (newest)
        this._contentIndex = new Map();

        // Save state
        this._isSaving = false;
        this._saveAgainAfter = false;

        // Initialize image storage for cleanup operations
        this._imageStorage = new ImageStorage();

        // Don't call async init here, caller must call init()
    }

    async init() {
        FileUtils.ensureDirectory(this._storagePath);
        await this._setupEncryption();
        await this._load();
    }

    async _setupEncryption() {
        if (this._settings && this._settings.get_boolean('encrypt-database')) {
            // Key file path: same directory as clipboard.json, named clipmaster.key
            const keyFilePath = this._storagePath.replace(/\.json$/, '.key');

            let key = null;

            // Priority 1: Try to load key from .key file (survives system format)
            try {
                const keyFromFile = await FileUtils.loadTextFile(keyFilePath);
                if (keyFromFile && keyFromFile.trim().length >= 16) {
                    key = keyFromFile.trim();
                    debugLog('Encryption key loaded from key file');
                }
            } catch (e) {
                // Key file doesn't exist or is unreadable
            }

            // Priority 2: Fall back to GSettings (for backward compatibility)
            if (!key) {
                key = this._settings.get_string('encryption-key');
                if (key) {
                    debugLog('Encryption key loaded from GSettings (fallback)');
                }
            }

            // Priority 3: Generate new key if none exists
            if (!key) {
                this._encryption = new SimpleEncryption();
                key = this._encryption.getKey();
                debugLog('Generated new encryption key');
            } else {
                this._encryption = new SimpleEncryption(key);
            }

            // Always save key to both locations for redundancy
            try {
                await FileUtils.saveTextFile(keyFilePath, key);
                debugLog('Encryption key saved to key file');
            } catch (e) {
                console.error(`ClipMaster: Could not save key file: ${e.message}`);
            }

            // Also keep in GSettings for backward compatibility
            this._settings.set_string('encryption-key', key);
        }
    }

    async _load() {
        try {
            const jsonStr = await FileUtils.loadTextFile(this._storagePath);
            let loadedNonImages = [];
            let loadedImages = [];
            let loadedLists = [];
            let loadedNextId = this._nextId;

            if (jsonStr) {
                let decodedStr = jsonStr;
                if (this._encryption && decodedStr.startsWith('ENC:')) {
                    decodedStr = this._encryption.decrypt(decodedStr.substring(4));
                }
                const data = JSON.parse(decodedStr);
                const mainItems = data.items || [];
                const mainImages = mainItems.filter(i => i?.type === ItemType.IMAGE);
                const mainNonImages = mainItems.filter(i => i?.type !== ItemType.IMAGE);
                loadedNonImages = mainNonImages;
                loadedImages = [...loadedImages, ...mainImages];
                loadedLists = data.lists || [];
                loadedNextId = Math.max(data.nextId || 1, loadedNextId);
            }

            // Load image items from separate file (if exists)
            const imagesStr = await FileUtils.loadTextFile(this._imagesStoragePath);
            if (imagesStr) {
                let decodedImagesStr = imagesStr;
                if (this._encryption && decodedImagesStr.startsWith('ENC:')) {
                    decodedImagesStr = this._encryption.decrypt(decodedImagesStr.substring(4));
                }
                const imgData = JSON.parse(decodedImagesStr);
                loadedImages = imgData.items || [];
                loadedNextId = Math.max(imgData.nextId || 1, loadedNextId);
            }

            const loadedItems = [...loadedNonImages, ...loadedImages];

            // Merge pending items that might have been added during async load
            if (this._pendingItems.length > 0) {
                // Add pending items to the top if they are newer
                this._items = [...this._pendingItems, ...loadedItems];
                this._pendingItems = [];
                this._isDirty = true; // Need to save the merge
            } else {
                this._items = loadedItems;
            }

            this._lists = loadedLists;
            this._nextId = Math.max(loadedNextId, this._nextId);
            this._isLoaded = true;

            // Ensure contentHash exists for all loaded items and rebuild indexes
            this._rebuildContentIndex();
            // Ensure nextId is always above max existing id
            const maxId = this._items.reduce((m, i) => Math.max(m, i?.id ?? 0), 0);
            this._nextId = Math.max(this._nextId, maxId + 1);

            if (this._isDirty) {
                this._save();
            }
        } catch (e) {
            console.error(`ClipMaster: Error loading database: ${e.message}`);
            // Keep any pending items
            this._items = this._pendingItems;
            this._pendingItems = [];
            // _lists initialized to [] in constructor
            this._isLoaded = true;
            this._rebuildContentIndex();
        }
    }

    _rebuildContentIndex() {
        this._contentIndex.clear();
        if (!this._items) return;

        // Newest-first index (items are typically stored newest-first)
        for (const item of this._items) {
            if (!item) continue;
            if (!item.contentHash) {
                const combined = (item.title || '') + '||' + (item.content || '');
                item.contentHash = HashUtils.hashContent(combined);
            }
            if (item.contentHash)
                this._contentIndex.set(item.contentHash, item.id);
        }
    }

    _save() {
        if (!this._timeoutManager) return;

        this._isDirty = true;

        // Slightly longer debounce for large histories to reduce main-thread churn
        const itemCount = this._items?.length || 0;
        const debounceMs = itemCount >= 500 ? Math.max(this._saveDebounceMs, 1000) : this._saveDebounceMs;

        this._timeoutManager.add(
            GLib.PRIORITY_DEFAULT,
            debounceMs,
            () => {
                this._doSave();
                return GLib.SOURCE_REMOVE;
            },
            'database-save'
        );
    }

    _saveImmediate() {
        if (!this._timeoutManager) return;

        this._timeoutManager.remove('database-save');
        this._doSave();
    }

    async _doSave() {
        if (!this._isDirty) return;
        if (!this._isLoaded) {
            debugLog('Database not loaded yet, deferring save');
            return;
        }

        if (this._isSaving) {
            this._saveAgainAfter = true;
            return;
        }

        this._isSaving = true;
        try {
            const nonImageItems = this._items.filter(i => i?.type !== ItemType.IMAGE);
            const imageItems = this._items.filter(i => i?.type === ItemType.IMAGE);

            const data = {
                items: nonImageItems,
                lists: this._lists,
                nextId: this._nextId
            };
            const imageData = {
                items: imageItems,
                nextId: this._nextId
            };

            // Serialize during idle to reduce chances of UI jank.
            // Also avoid pretty-printing to reduce CPU + file size.
            let jsonStr = '';
            let imageJsonStr = '';
            await new Promise(resolve => {
                GLib.idle_add(GLib.PRIORITY_DEFAULT_IDLE, () => {
                    try {
                        jsonStr = JSON.stringify(data);
                        imageJsonStr = JSON.stringify(imageData);
                    } catch (e) {
                        console.error(`ClipMaster: JSON stringify error: ${e.message}`);
                        jsonStr = '';
                        imageJsonStr = '';
                    }
                    resolve();
                    return GLib.SOURCE_REMOVE;
                });
            });
            if (!jsonStr)
                return;

            if (this._encryption) {
                jsonStr = 'ENC:' + this._encryption.encrypt(jsonStr);
                imageJsonStr = imageJsonStr ? 'ENC:' + this._encryption.encrypt(imageJsonStr) : imageJsonStr;
            }

            const mainSaved = await FileUtils.saveTextFile(this._storagePath, jsonStr);
            const imagesSaved = await FileUtils.saveTextFile(this._imagesStoragePath, imageJsonStr);
            if (mainSaved && imagesSaved) {
                this._isDirty = false;
                await this._checkDatabaseSize();
            }
        } catch (e) {
            console.error(`ClipMaster: Error saving database: ${e.message}`);
        } finally {
            this._isSaving = false;
            if (this._saveAgainAfter) {
                this._saveAgainAfter = false;
                // If something dirtied the DB while saving, run one more save soon.
                this._save();
            }
        }
    }

    async getFileSize() {
        try {
            const file = Gio.File.new_for_path(this._storagePath);
            if (!file.query_exists(null)) {
                return 0;
            }

            return new Promise((resolve, reject) => {
                file.query_info_async(
                    'standard::size',
                    Gio.FileQueryInfoFlags.NONE,
                    GLib.PRIORITY_DEFAULT,
                    null,
                    (obj, res) => {
                        try {
                            const info = obj.query_info_finish(res);
                            resolve(info.get_size());
                        } catch (e) {
                            reject(e);
                        }
                    }
                );
            });
        } catch (e) {
            console.error(`ClipMaster: Error getting file size: ${e.message}`);
            return 0;
        }
    }

    async _checkDatabaseSize() {
        if (!this._settings || this._isCleaning) return;

        try {
            const maxSizeMB = this._settings.get_int('max-db-size-mb');
            if (maxSizeMB <= 0) return;

            const maxSizeBytes = maxSizeMB * 1024 * 1024;
            const currentSizeBytes = await this.getFileSize();

            if (currentSizeBytes <= 0) return;

            const usagePercent = (currentSizeBytes / maxSizeBytes) * 100;

            if (currentSizeBytes >= maxSizeBytes) {
                this._isCleaning = true;
                try {
                    const cleaned = this._enforceDatabaseSizeLimit(maxSizeBytes);
                    if (cleaned > 0 && this._onNotification) {
                        this._onNotification(
                            _('Database Size Limit Reached'),
                            _(`Database reached ${maxSizeMB}MB. Removed ${cleaned} oldest item(s) to free space.`)
                        );
                    }
                } finally {
                    this._isCleaning = false;
                }
            } else if (usagePercent >= 90) {
                const now = Date.now();
                if (now - this._lastWarningTime > this._warningCooldownMs) {
                    this._lastWarningTime = now;
                    if (this._onNotification) {
                        const remainingMB = ((maxSizeBytes - currentSizeBytes) / (1024 * 1024)).toFixed(1);
                        this._onNotification(
                            _('Database Size Warning'),
                            _(`Database is ${usagePercent.toFixed(0)}% full (${remainingMB}MB remaining). Old items will be automatically removed when limit is reached.`)
                        );
                    }
                }
            }
        } catch (e) {
            console.error(`ClipMaster: Error checking database size: ${e.message}`);
            this._isCleaning = false;
        }
    }

    _enforceDatabaseSizeLimit(maxSizeBytes) {
        let removedCount = 0;

        const favorites = this._items.filter(i => i.isFavorite);
        let nonFavorites = this._items.filter(i => !i.isFavorite);

        // Oldest first for removal
        nonFavorites.sort((a, b) => (a.created || 0) - (b.created || 0));

        // Estimate serialized size once, then subtract removed items.
        // This avoids O(n^2) full-database JSON.stringify in a loop.
        const encoder = new TextEncoder();
        const baseOverhead =
            encoder.encode('{"items":[],"lists":,"nextId":}'.replace(',"lists":', '')).length;
        const listsPart = encoder.encode(JSON.stringify(this._lists || [])).length;
        const nextIdPart = encoder.encode(String(this._nextId)).length;

        const itemSize = (it) => {
            try {
                return encoder.encode(JSON.stringify(it)).length;
            } catch (_) {
                return 0;
            }
        };

        const favSizes = favorites.map(itemSize);
        const nonFavSizes = nonFavorites.map(itemSize);

        let totalPlain =
            baseOverhead +
            listsPart +
            nextIdPart +
            favSizes.reduce((a, b) => a + b, 0) +
            nonFavSizes.reduce((a, b) => a + b, 0);

        // Apply rough encryption overhead: base64 expands by ~4/3, plus 'ENC:' prefix.
        const estimatedTotal = () => {
            if (!this._encryption) return totalPlain;
            return 4 + Math.ceil(totalPlain * 4 / 3);
        };

        const target = maxSizeBytes * 0.95;
        while (nonFavorites.length > 0 && estimatedTotal() >= target) {
            const removed = nonFavorites.shift();
            const removedBytes = nonFavSizes.shift() ?? itemSize(removed);
            totalPlain -= removedBytes;
            removedCount++;

            // Clean up image file if it's a file-based image
            if (removed?.type === ItemType.IMAGE &&
                removed.metadata?.storedAs === 'file' &&
                removed.content &&
                this._imageStorage) {
                try {
                    this._imageStorage.deleteImage(removed.content);
                } catch (e) {
                    console.error(`ClipMaster: Error deleting image file during size cleanup: ${e.message}`);
                }
            }

            if (removedCount > 2000) {
                console.log('ClipMaster: Safety limit reached while cleaning database');
                break;
            }
        }

        this._items = [...favorites, ...nonFavorites];
        this._rebuildContentIndex();

        if (removedCount > 0)
            this._saveImmediate();

        return removedCount;
    }

    destroy() {


        if (this._timeoutManager) {
            this._doSave();
            this._timeoutManager.removeAll();
            this._timeoutManager = null;
        }
    }

    addItem(item) {
        if (!this._items) return null;

        // Use Title + Content combination for uniqueness
        const combinedContent = (item.title || '') + '||' + (item.content || '');
        const contentHash = HashUtils.hashContent(combinedContent);

        // Fast duplicate lookup via index (loaded items).
        let existing = null;
        const existingId = this._contentIndex.get(contentHash) ?? null;
        if (existingId !== null && existingId !== undefined) {
            existing = this._items.find(i => i.id === existingId) || null;
            // Extra safety: ensure content actually matches (hash collision guard)
            if (existing) {
                const eCombined = (existing.title || '') + '||' + (existing.content || '');
                if (eCombined !== combinedContent)
                    existing = null;
            }
        }

        // Also check pending items (usually small)
        if (!existing && this._pendingItems?.length) {
            existing = this._pendingItems.find(i => {
                const h = i.contentHash || i.hash || '';
                if (h === contentHash) {
                    const pCombined = (i.title || '') + '||' + (i.content || '');
                    return pCombined === combinedContent;
                }
                return false;
            }) || null;
        }

        const skipDuplicates = this._settings?.get_boolean('skip-duplicates') ?? true;
        debugLog(`skip-duplicates setting = ${skipDuplicates}`);

        debugLog(`existing=${!!existing}, skipDuplicates=${skipDuplicates}`);

        if (existing && skipDuplicates) {
            debugLog(`Skipping duplicate, updating existing item`);
            existing.lastUsed = Date.now();
            existing.useCount = (existing.useCount ?? 0) + 1;

            // Only move to top if it's already in _items (main list)
            if (this._items.includes(existing)) {
                this._moveToTop(existing.id);
            }

            this._save();
            return existing.id;
        }

        debugLog(`Adding new item (existing=${!!existing}, skipDuplicates=${skipDuplicates})`);

        const uniqueHash = skipDuplicates ? contentHash : `${contentHash}_${Date.now()}`;

        const newItem = {
            id: this._nextId++,
            type: item.type || ItemType.TEXT,
            content: item.content,
            thumbnail: item.thumbnail || null,  // Base64 thumbnail for IMAGE type
            plainText: item.plainText || item.content,
            preview: item.preview || (item.content || '').substring(0, 200),
            title: item.title || null,
            hash: uniqueHash,
            contentHash: contentHash,
            isFavorite: false,
            listId: null,
            sourceApp: item.sourceApp || null,
            created: Date.now(),
            lastUsed: Date.now(),
            useCount: 1,
            imageFormat: item.imageFormat || null,
            metadata: item.metadata || null
        };

        if (this._isLoaded) {
            this._items.unshift(newItem);
            this._contentIndex.set(contentHash, newItem.id);
        } else {
            this._pendingItems.push(newItem);
        }

        this._save();
        return newItem.id;
    }

    _moveToTop(itemId) {
        const index = this._items.findIndex(i => i.id === itemId);
        if (index > 0) {
            const item = this._items.splice(index, 1)[0];
            this._items.unshift(item);
        }
    }

    /**
     * Move an item to the top of the list (public method)
     * @param {number} itemId - The ID of the item to move
     */
    moveToTop(itemId) {
        this._moveToTop(itemId);
        this._save();
    }

    getItems(options = {}) {
        // Return mostly from _items. Pending items are not shown until loaded.
        let items = [...this._items];

        if (options.type) {
            // CODE category includes both CODE and HTML types
            if (options.type === ItemType.CODE) {
                items = items.filter(i => i.type === ItemType.CODE || i.type === ItemType.HTML);
            } else if (options.type === ItemType.TEXT && options.textIncludeUrlCode) {
                // TEXT filter includes URL and CODE when setting is enabled
                items = items.filter(i => 
                    i.type === ItemType.TEXT || 
                    i.type === ItemType.URL || 
                    i.type === ItemType.CODE ||
                    i.type === ItemType.HTML
                );
            } else {
                items = items.filter(i => i.type === options.type);
            }
            
            // Exclude favorites from type filters (they show in Favorites tab)
            if (options.excludeFavorites) {
                items = items.filter(i => !i.isFavorite);
            }
        }

        if (options.listId !== undefined) {
            if (options.listId === -1) {
                items = items.filter(i => i.isFavorite);
            } else if (options.listId !== null) {
                items = items.filter(i => i.listId === options.listId);
            }
        }

        if (options.search) {
            const query = options.search.toLowerCase();
            items = items.filter(i =>
                (i.content && i.content.toLowerCase().includes(query)) ||
                (i.plainText && i.plainText.toLowerCase().includes(query)) ||
                (i.title && i.title.toLowerCase().includes(query))
            );
        }

        if (options.limit) {
            items = items.slice(0, options.limit);
        }

        return items;
    }

    getItem(itemId) {
        return this._items.find(i => i.id === itemId) || this._pendingItems.find(i => i.id === itemId);
    }

    updateItem(itemId, updates) {
        const item = this._items.find(i => i.id === itemId) || this._pendingItems.find(i => i.id === itemId);
        if (item) {
            Object.assign(item, updates);
            this._save();
            return true;
        }
        return false;
    }

    deleteItem(itemId) {
        let index = this._items.findIndex(i => i.id === itemId);
        if (index >= 0) {
            const item = this._items[index];

            // Clean up image file if it's a file-based image
            if (item.type === ItemType.IMAGE &&
                item.metadata?.storedAs === 'file' &&
                item.content &&
                this._imageStorage) {
                try {
                    this._imageStorage.deleteImage(item.content);
                    debugLog(`Deleted image file: ${item.content}`);
                } catch (e) {
                    console.error(`ClipMaster: Error deleting image file: ${e.message}`);
                }
            }

            this._items.splice(index, 1);
            if (item?.contentHash) {
                const mapped = this._contentIndex.get(item.contentHash);
                if (mapped === itemId) {
                    this._contentIndex.delete(item.contentHash);
                    // Find another item with same contentHash (rare)
                    const fallback = this._items.find(i => i.contentHash === item.contentHash);
                    if (fallback)
                        this._contentIndex.set(item.contentHash, fallback.id);
                }
            }
            this._save();
            return true;
        }

        index = this._pendingItems.findIndex(i => i.id === itemId);
        if (index >= 0) {
            this._pendingItems.splice(index, 1);
            return true;
        }

        return false;
    }

    /**
     * Enforce history size limit by removing oldest non-favorite items
     * @param {number} maxItems - Maximum number of items to keep
     */
    enforceLimit(maxItems) {
        if (!maxItems || maxItems <= 0) return;

        const favorites = this._items.filter(i => i.isFavorite);
        const nonFavorites = this._items.filter(i => !i.isFavorite);

        if (nonFavorites.length > maxItems) {
            // Sort by created date (oldest first)
            nonFavorites.sort((a, b) => (a.created || 0) - (b.created || 0));

            // Remove oldest items beyond the limit
            const itemsToRemove = nonFavorites.slice(0, nonFavorites.length - maxItems);

            for (const item of itemsToRemove) {
                // Clean up image files for file-based images
                if (item.type === ItemType.IMAGE &&
                    item.metadata?.storedAs === 'file' &&
                    item.content &&
                    this._imageStorage) {
                    try {
                        this._imageStorage.deleteImage(item.content);
                        debugLog(`enforceLimit: Deleted image file: ${item.content}`);
                    } catch (e) {
                        console.error(`ClipMaster: Error deleting image file during cleanup: ${e.message}`);
                    }
                }
            }

            // Keep favorites + remaining non-favorites within limit
            const keptNonFavorites = nonFavorites.slice(nonFavorites.length - maxItems);
            this._items = [...favorites, ...keptNonFavorites];

            // Re-sort by lastUsed/created (newest first)
            this._items.sort((a, b) => (b.lastUsed || b.created || 0) - (a.lastUsed || a.created || 0));
            this._rebuildContentIndex();

            if (itemsToRemove.length > 0) {
                debugLog(`enforceLimit: Removed ${itemsToRemove.length} old items`);
                this._save();
            }
        }
    }

    /**
     * Remove duplicate items, keeping the most recent one
     * Uses Title + Content combination for uniqueness
     * @returns {number} Number of duplicates removed
     */
    cleanupDuplicates() {
        const seen = new Set();
        const toRemove = [];

        // Sort by created date (newest first) so we keep the newest version
        const sorted = [...this._items].sort((a, b) => (b.created || 0) - (a.created || 0));

        for (const item of sorted) {
            if (!item) continue;

            // Use cached contentHash when available; otherwise compute and cache
            if (!item.contentHash) {
                const combinedContent = (item.title || '') + '||' + (item.content || '');
                item.contentHash = HashUtils.hashContent(combinedContent);
            }

            const hash = item.contentHash || '';
            if (hash && seen.has(hash)) {
                // This is a duplicate (older), mark for removal
                toRemove.push(item.id);
            } else {
                if (hash) seen.add(hash);
            }
        }

        if (toRemove.length > 0) {
            // Clean up image files for removed items
            for (const id of toRemove) {
                const item = this._items.find(i => i.id === id);
                if (!item) continue;
                if (item.type === ItemType.IMAGE &&
                    item.metadata?.storedAs === 'file' &&
                    item.content &&
                    this._imageStorage) {
                    try {
                        this._imageStorage.deleteImage(item.content);
                    } catch (e) {
                        console.error(`ClipMaster: Error deleting image file during duplicate cleanup: ${e.message}`);
                    }
                }
            }

            const removeSet = new Set(toRemove);
            this._items = this._items.filter(i => !removeSet.has(i.id));
            this._rebuildContentIndex();
            this._save();
            debugLog(`cleanupDuplicates: Removed ${toRemove.length} duplicate items`);
        }

        return toRemove.length;
    }

    toggleFavorite(itemId) {
        const item = this._items.find(i => i.id === itemId);
        if (item) {
            item.isFavorite = !item.isFavorite;
            this._save();
            return item.isFavorite;
        }
        return false;
    }

    clearHistory(keepFavorites = true) {
        if (keepFavorites) {
            this._items = this._items.filter(i => i.isFavorite);
        } else {
            this._items = [];
        }
        this._pendingItems = []; // Clear pending too
        this._rebuildContentIndex();
        this._save();
    }

    requestsSave() {
        this._save();
    }

    // ... lists and import/export methods remain mostly same but logging fixed ...

    getLists() {
        return this._lists;
    }

    createList(name, color = null) {
        const list = {
            id: this._lists.length > 0 ? Math.max(...this._lists.map(l => l.id)) + 1 : 1,
            name: name,
            color: color,
            isPinned: false,
            created: Date.now()
        };
        this._lists.push(list);
        this._save();
        return list.id;
    }

    deleteList(listId) {
        this._items.forEach(item => {
            if (item.listId === listId) {
                item.listId = null;
            }
        });

        this._lists = this._lists.filter(l => l.id !== listId);
        this._save();
    }

    updateList(listId, updates) {
        const list = this._lists.find(l => l.id === listId);
        if (list) {
            if (updates.name !== undefined) list.name = updates.name;
            if (updates.color !== undefined) list.color = updates.color;
            this._save();
            return true;
        }
        return false;
    }

    getListById(listId) {
        return this._lists.find(l => l.id === listId) || null;
    }

    addItemToList(itemId, listId) {
        const item = this._items.find(i => i.id === itemId);
        if (item) {
            item.listId = listId;
            item.isFavorite = true;
            this._save();
        }
    }

    exportData() {
        return JSON.stringify({
            version: 1,
            items: this._items,
            lists: this._lists,
            exported: new Date().toISOString()
        }, null, 2);
    }

    importData(jsonString) {
        try {
            const data = JSON.parse(jsonString);
            if (data.items) {
                data.items.forEach(item => {
                    const combined = (item.title || '') + '||' + (item.content || '');
                    const contentHash = HashUtils.hashContent(combined);
                    const existingId = this._contentIndex.get(contentHash);
                    const existing = existingId ? this._items.find(i => i.id === existingId) : null;
                    const isSame = existing
                        ? ((existing.title || '') + '||' + (existing.content || '')) === combined
                        : false;

                    if (!existing || !isSame) {
                        item.id = this._nextId++;
                        item.contentHash = contentHash;
                        this._items.push(item);
                    }
                });
            }
            if (data.lists) {
                data.lists.forEach(list => {
                    if (!this._lists.find(l => l.name === list.name)) {
                        list.id = this._lists.length > 0 ? Math.max(...this._lists.map(l => l.id)) + 1 : 1;
                        this._lists.push(list);
                    }
                });
            }
            this._rebuildContentIndex();
            this._save();
            return true;
        } catch (e) {
            console.error(`ClipMaster: Import error: ${e.message}`);
            return false;
        }
    }

    getStats() {
        const stats = {
            total: this._items.length,
            favorites: this._items.filter(i => i.isFavorite).length,
            byType: {}
        };

        this._items.forEach(item => {
            stats.byType[item.type] = (stats.byType[item.type] || 0) + 1;
        });

        return stats;
    }
}
