import { Extension } from 'resource:///org/gnome/shell/extensions/extension.js';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import St from 'gi://St';
import Clutter from 'gi://Clutter';
import GObject from 'gi://GObject';
import Gio from 'gi://Gio';
import GLib from 'gi://GLib';

// --- CLASSE UNITY BUTTONS ---
const UnityButtons = GObject.registerClass(
class UnityButtons extends PanelMenu.Button {
    _init() {
        super._init(0.0, 'UnityButtons');
        
        // On remplace le style inline par une classe CSS
        this.add_style_class_name('unity-panel-button');
        
        this._container = new St.BoxLayout({ style_class: 'unity-container', reactive: true });
        this._btnBox = new St.BoxLayout({ style: 'spacing: 6px;' });

        const createBtn = (color, hoverColor, action) => {
            let btn = new St.Button({
                style: `background-color: ${color};`, // On garde juste la couleur unique ici
                style_class: 'unity-button',         // Tout le reste (taille, bords) vient du CSS
                y_align: Clutter.ActorAlign.CENTER,
                reactive: true,
                track_hover: true
            });

            btn.connect('notify::hover', () => {
                btn.set_style(`background-color: ${btn.hover ? hoverColor : color};`);
            });
            btn.connect('clicked', action);
            return btn;
        };

        // Boutons Fermer et Réduire
        this._btnBox.add_child(createBtn('#FFB347', '#FF8C00', () => {
            let win = global.display.get_focus_window();
            if (win) win.delete(global.get_current_time());
        }));

        this._btnBox.add_child(createBtn('#C0C0C0', '#808080', () => {
            let win = global.display.get_focus_window();
            if (win && !win.minimized) win.minimize();
            else this._restoreLast();
        }));

        this._titleLabel = new St.Label({
            style_class: 'unity-title',
            y_align: Clutter.ActorAlign.CENTER
        });

        this._container.add_child(this._btnBox);
        this._container.add_child(this._titleLabel);
        this.add_child(this._container);

        this._displaySignals = [];
        this._workspaceSignals = [];
        this._titleSignalId = 0;
        this._currentWin = null;

        this._displaySignals.push(global.display.connect('notify::focus-window', () => this._updateAll()));
        this._workspaceSignals.push(global.workspace_manager.connect('workspace-switched', () => this._updateAll()));
        this._displaySignals.push(global.display.connect('window-created', () => this._updateAll()));
        this._updateAll();
    }

    _updateAll() {
        let win = global.display.get_focus_window();
        let workspace = global.workspace_manager.get_active_workspace();
        let windows = workspace.list_windows().filter(w => w.get_window_type() === 0);
        this.visible = windows.length > 0;
        if (this._currentWin && this._titleSignalId) this._currentWin.disconnect(this._titleSignalId);
        if (win && win.get_window_type() === 0) {
            this._currentWin = win;
            this._titleLabel.set_text(win.get_title());
            this._titleLabel.show();
            this._titleSignalId = win.connect('notify::title', () => {
                this._titleLabel.set_text(win.get_title());
            });
        } else {
            this._titleLabel.hide();
            this._currentWin = null;
        }
    }

    _restoreLast() {
        let workspace = global.workspace_manager.get_active_workspace();
        let windows = workspace.list_windows().filter(w => w.get_window_type() === 0);
        if (windows.length > 0) {
            windows.sort((a, b) => b.get_user_time() - a.get_user_time());
            windows[0].activate(global.get_current_time());
        }
    }

    destroy() {
        if (this._currentWin && this._titleSignalId) this._currentWin.disconnect(this._titleSignalId);
        this._displaySignals.forEach(id => global.display.disconnect(id));
        this._workspaceSignals.forEach(id => global.workspace_manager.disconnect(id));
        super.destroy();
    }
});

// --- CLASSE EXTENSION ---
export default class UnityButtonsExtension extends Extension {
    enable() {
        this._settings = new Gio.Settings({ schema_id: 'org.gnome.desktop.wm.preferences' });
        this._configFile = Gio.File.new_for_path(this.path + '/last_layout.txt');
        this._isInternalChange = false;

        let current = this._settings.get_string('button-layout');
        if (current && current.length > 1 && current !== ':' && current.includes('close')) {
            this._saveToDisk(current);
        } else if (!this._configFile.query_exists(null)) {
            this._saveToDisk('close,minimize,maximize:');
        }

        this._settingsSignal = this._settings.connect('changed::button-layout', () => {
            if (this._isInternalChange || !this._settings) return;
            let newVal = this._settings.get_string('button-layout');
            if (newVal && newVal.length > 1 && newVal !== ':' && newVal.includes('close')) {
                this._saveToDisk(newVal);
                GLib.timeout_add(GLib.PRIORITY_DEFAULT, 500, () => {
                    if (this._settings) {
                        this._isInternalChange = true;
                        this._settings.set_string('button-layout', ':');
                        this._isInternalChange = false;
                    }
                    return GLib.SOURCE_REMOVE;
                });
            }
        });

        this._isInternalChange = true;
        this._settings.set_string('button-layout', ':');
        this._isInternalChange = false;

        this._indicator = new UnityButtons();
        Main.panel.addToStatusArea('unity-buttons-v2', this._indicator, 0, 'left');
    }

    _saveToDisk(layout) {
        if (!layout || layout.length < 2 || layout === ':' || !layout.includes('close')) return;
        try {
            let isLeft = layout.endsWith(':');
            let sidePart = isLeft ? layout.split(':')[0] : layout.split(':')[1];
            let btns = sidePart.split(',').filter(b => b !== '');
            if (!btns.includes('close')) btns.push('close');
            let finalLayout = isLeft 
                ? ['close', ...btns.filter(b => b !== 'close')].join(',') + ':'
                : ':' + [...btns.filter(b => b !== 'close'), 'close'].join(',');
            this._configFile.replace_contents(finalLayout, null, false, Gio.FileCreateFlags.NONE, null);
        } catch (e) {
            console.error("Erreur sauvegarde:", e);
        }
    }

    disable() {
        if (this._settingsSignal) this._settings.disconnect(this._settingsSignal);
        let layoutToRestore = 'close,minimize,maximize:';
        try {
            if (this._configFile.query_exists(null)) {
                let [success, contents] = this._configFile.load_contents(null);
                if (success) {
                    let saved = new TextDecoder().decode(contents).trim();
                    if (saved && saved.includes('close')) layoutToRestore = saved;
                }
            }
        } catch (e) {}
        if (this._settings) {
            this._isInternalChange = true;
            this._settings.set_string('button-layout', layoutToRestore);
            Gio.Settings.sync();
        }
        if (this._indicator) this._indicator.destroy();
        this._settings = null;
    }
}
