import { Extension } from 'resource:///org/gnome/shell/extensions/extension.js';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import St from 'gi://St';
import Clutter from 'gi://Clutter';
import GObject from 'gi://GObject';
import Gio from 'gi://Gio';
import GLib from 'gi://GLib';
import Meta from 'gi://Meta';

// --- CLASSE UNITY BUTTONS ---
const UnityButtons = GObject.registerClass(
class UnityButtons extends PanelMenu.Button {
    _init() {
        super._init(0.0, 'UnityButtons');
        this.add_style_class_name('unity-panel-button');
        
        this._container = new St.BoxLayout({ style_class: 'unity-container', reactive: true });
        this._btnBox = new St.BoxLayout({ style: 'spacing: 6px;' });

        const createBtn = (color, hoverColor, action) => {
            let btn = new St.Button({
                style: `background-color: ${color};`,
                style_class: 'unity-button',
                y_align: Clutter.ActorAlign.CENTER,
                reactive: true,
                track_hover: true
            });
            btn.connect('notify::hover', () => {
                btn.set_style(`background-color: ${btn.hover ? hoverColor : color};`);
            });
            btn.connect('clicked', action);
            return btn;
        };

        this._btnBox.add_child(createBtn('#FFB347', '#FF8C00', () => {
            let win = global.display.get_focus_window();
            if (win) win.delete(global.get_current_time());
        }));

        this._btnBox.add_child(createBtn('#C0C0C0', '#808080', () => {
            let win = global.display.get_focus_window();
            if (win) win.unmaximize(Meta.MaximizeFlags.BOTH);
        }));

        this._titleLabel = new St.Label({
            style_class: 'unity-title',
            y_align: Clutter.ActorAlign.CENTER
        });

        this._container.add_child(this._btnBox);
        this._container.add_child(this._titleLabel);
        this.add_child(this._container);

        this._signals = [];
        this._currentWin = null;
        this._titleSignalId = 0;

        // Écoute les changements de focus
        this._signals.push(global.display.connect('notify::focus-window', () => this._updateAll()));
        
        // Écoute les changements de taille/état via le Window Manager
        // C'est ici que l'on capture la maximisation/restauration
        this._signals.push(global.window_manager.connect('size-change', () => this._updateAll()));
        
        this._updateAll();
    }

    _updateAll() {
    let win = global.display.get_focus_window();
    
    // 1. On récupère le titre
    let title = win ? win.get_title() : "";

    // 2. On définit si la fenêtre est valide (pas le bureau, pas un dock)
    // On ignore les fenêtres qui s'appellent "Desktop Icons" ou qui ont le type DESKTOP
    let isDesktop = title && title.includes("Desktop Icons");
    let isMax = win && win.get_maximized() === Meta.MaximizeFlags.BOTH;
    let isNormal = win && win.get_window_type() === Meta.WindowType.NORMAL;

    // L'indicateur est visible seulement si c'est une fenêtre normale, agrandie et PAS le bureau
    this.visible = isNormal && isMax && !isDesktop;

    if (this._currentWin && this._titleSignalId) {
        this._currentWin.disconnect(this._titleSignalId);
        this._titleSignalId = 0;
    }

    if (this.visible) {
        this._currentWin = win;
        this._titleLabel.set_text(title);
        this._titleLabel.show();
        this._titleSignalId = win.connect('notify::title', () => {
            this._titleLabel.set_text(win.get_title());
        });
    } else {
        this._titleLabel.hide();
        this._currentWin = null;
    }
    
    if (this._extParent) {
        this._extParent.updateSystemButtons(this.visible);
    }
}

    destroy() {
        if (this._currentWin && this._titleSignalId) {
            this._currentWin.disconnect(this._titleSignalId);
        }
        this._signals.forEach(id => {
            // Nettoyage intelligent selon l'origine du signal
            global.display.disconnect(id);
            global.window_manager.disconnect(id);
        });
        super.destroy();
    }
});

// --- CLASSE EXTENSION ---
export default class UnityButtonsExtension extends Extension {
    enable() {
        this._settings = new Gio.Settings({ schema_id: 'org.gnome.desktop.wm.preferences' });
        this._configFile = Gio.File.new_for_path(this.path + '/last_layout.txt');
        this._isInternalChange = false;

        let current = this._settings.get_string('button-layout');
        if (current && current !== ':' && !this._configFile.query_exists(null)) {
            this._saveToDisk(current);
        }

        this._indicator = new UnityButtons();
        this._indicator._extParent = this;
        Main.panel.addToStatusArea('unity-buttons-v2', this._indicator, 0, 'left');
    }

    updateSystemButtons(hideOnWindow) {
        if (!this._settings || this._isInternalChange) return;

        let targetLayout = hideOnWindow ? ':' : this._getSavedLayout();
        if (this._settings.get_string('button-layout') === targetLayout) return;

        this._isInternalChange = true;
        this._settings.set_string('button-layout', targetLayout);
        this._isInternalChange = false;
    }

    _getSavedLayout() {
        try {
            if (this._configFile.query_exists(null)) {
                let [success, contents] = this._configFile.load_contents(null);
                if (success) return new TextDecoder().decode(contents).trim();
            }
        } catch (e) {}
        return 'close,minimize,maximize:';
    }

    _saveToDisk(layout) {
        try {
            this._configFile.replace_contents(layout, null, false, Gio.FileCreateFlags.NONE, null);
        } catch (e) {}
    }

    disable() {
        if (this._settings) {
            this._isInternalChange = true;
            this._settings.set_string('button-layout', this._getSavedLayout());
            this._isInternalChange = false;
        }
        if (this._indicator) {
            this._indicator.destroy();
            this._indicator = null;
        }
        this._settings = null;
    }
}
