import St from "gi://St";
import GObject from "gi://GObject";
import Clutter from "gi://Clutter";
import GLib from "gi://GLib";
import { ControlButtons } from "./ControlButtons.js";
import { AlbumArt } from "./AlbumArt.js";
import { ProgressSlider } from "./ProgressSlider.js";
import { PlayerTabs } from "./PlayerTabs.js";

export const MediaControls = GObject.registerClass(
  {
    Signals: {
      "play-pause": {},
      next: {},
      previous: {},
      shuffle: {},
      repeat: {},
      seek: { param_types: [GObject.TYPE_DOUBLE] },
      "player-changed": { param_types: [GObject.TYPE_STRING] },
    },
  },
  class MediaControls extends St.BoxLayout {
    _init(settings) {
      super._init({
        vertical: true,
        style_class: "media-controls-modern",
        style: "min-width: 340px; max-width: 340px;",
      });

      this._settings = settings;
      this._currentPlayerName = null;
      this._playerSliderPositions = new Map();
      this._playerArtCache = new Map();
      this._artistScrollTimeout = null;
      this._artistScrollPosition = 0;
      this._titleScrollTimeout = null;
      this._titleScrollPosition = 0;

      this._buildUI();
    }

    _buildUI() {
      const headerBox = new St.BoxLayout({
        style: "margin-bottom: 20px; spacing: 8px;",
        x_align: Clutter.ActorAlign.CENTER,
      });

      this._playerTabs = new PlayerTabs();
      this._playerTabs.connect("player-changed", (_, name) =>
        this.emit("player-changed", name),
      );
      headerBox.add_child(this._playerTabs);
      this.add_child(headerBox);

      this._albumArt = new AlbumArt(this._settings);
      this.add_child(this._albumArt);

      const infoBox = new St.BoxLayout({
        vertical: true,
        style: "spacing: 6px; margin-bottom: 24px;",
        x_align: Clutter.ActorAlign.CENTER,
      });

      this._titleLabel = new St.Label({
        text: globalThis._?.("No media playing") || "No media playing",
        style:
          "font-weight: 700; font-size: 16px; text-align: center; max-width: 300px;",
        x_align: Clutter.ActorAlign.CENTER,
      });
      this._titleLabel.clutter_text.ellipsize = 3;
      this._titleLabel.clutter_text.line_alignment = 2;
      infoBox.add_child(this._titleLabel);

      this._artistLabel = new St.Label({
        text: "",
        style:
          "font-size: 13px; font-weight: 500; text-align: center; max-width: 280px;",
        x_align: Clutter.ActorAlign.CENTER,
      });
      this._artistLabel.clutter_text.ellipsize = 0;
      this._artistLabel.clutter_text.line_alignment = 2;
      infoBox.add_child(this._artistLabel);
      this.add_child(infoBox);

      this._progressSlider = new ProgressSlider();
      this._progressSlider.connect("seek", (_, position) =>
        this.emit("seek", position),
      );
      this._progressSlider.connect("drag-begin", () =>
        this.stopPositionUpdate(),
      );
      this._progressSlider.connect("drag-end", () => {
        if (this._currentPlayerName) {
          this._playerSliderPositions.set(this._currentPlayerName, {
            position: this._progressSlider.currentPosition,
            length: this._progressSlider.trackLength,
            value: this._progressSlider.sliderValue,
          });
        }
      });
      this.add_child(this._progressSlider);

      this._controlButtons = new ControlButtons();
      this._controlButtons.connect("play-pause", () => this.emit("play-pause"));
      this._controlButtons.connect("next", () => this.emit("next"));
      this._controlButtons.connect("previous", () => this.emit("previous"));
      this._controlButtons.connect("shuffle", () => this.emit("shuffle"));
      this._controlButtons.connect("repeat", () => this.emit("repeat"));
      this.add_child(this._controlButtons);
    }

    _startArtistScrolling(fullText) {
      this._stopArtistScrolling();

      const MAX_CHARS = 35;

      // Check if scrolling is enabled in settings
      const scrollEnabled = this._settings.get_boolean("enable-artist-scroll");

      if (fullText.length <= MAX_CHARS || !scrollEnabled) {
        // If text is short enough or scrolling disabled, just display it
        this._artistLabel.text =
          fullText.length > MAX_CHARS
            ? fullText.substring(0, MAX_CHARS - 3) + "..."
            : fullText;
        return;
      }

      const paddedText = fullText + "   •   ";

      // Get scroll speed from settings (1-10)
      const scrollSpeed = this._settings.get_int("artist-scroll-speed");
      // Convert speed to interval (lower interval = faster scroll)
      // Speed 1 = 250ms, Speed 10 = 25ms
      const interval = Math.max(25, 275 - scrollSpeed * 25);

      this._artistScrollTimeout = GLib.timeout_add(
        GLib.PRIORITY_LOW,
        interval,
        () => {
          this._artistScrollPosition++;

          if (this._artistScrollPosition >= paddedText.length) {
            this._artistScrollPosition = 0;
          }

          const displayText =
            paddedText.substring(this._artistScrollPosition) +
            paddedText.substring(0, this._artistScrollPosition);

          this._artistLabel.text = displayText.substring(0, MAX_CHARS);

          return GLib.SOURCE_CONTINUE;
        },
      );
    }

    _stopArtistScrolling() {
      if (this._artistScrollTimeout) {
        GLib.source_remove(this._artistScrollTimeout);
        this._artistScrollTimeout = null;
      }
      this._artistScrollPosition = 0;
    }

    _startTitleScrolling(fullText) {
      this._stopTitleScrolling();

      const MAX_CHARS = 35;

      // Check if scrolling is enabled in settings
      const scrollEnabled = this._settings.get_boolean("enable-title-scroll");

      if (fullText.length <= MAX_CHARS || !scrollEnabled) {
        // If text is short enough or scrolling disabled, just display it
        this._titleLabel.text =
          fullText.length > MAX_CHARS
            ? fullText.substring(0, MAX_CHARS - 3) + "..."
            : fullText;
        return;
      }

      const paddedText = fullText + "   •   ";

      // Get scroll speed from settings (1-10)
      const scrollSpeed = this._settings.get_int("title-scroll-speed");
      // Convert speed to interval (lower interval = faster scroll)
      const interval = Math.max(25, 275 - scrollSpeed * 25);

      this._titleScrollTimeout = GLib.timeout_add(
        GLib.PRIORITY_LOW,
        interval,
        () => {
          this._titleScrollPosition++;

          if (this._titleScrollPosition >= paddedText.length) {
            this._titleScrollPosition = 0;
          }

          const displayText =
            paddedText.substring(this._titleScrollPosition) +
            paddedText.substring(0, this._titleScrollPosition);

          this._titleLabel.text = displayText.substring(0, MAX_CHARS);

          return GLib.SOURCE_CONTINUE;
        },
      );
    }

    _stopTitleScrolling() {
      if (this._titleScrollTimeout) {
        GLib.source_remove(this._titleScrollTimeout);
        this._titleScrollTimeout = null;
      }
      this._titleScrollPosition = 0;
    }

    update(info, playerName, manager) {
      if (!info) return;

      const playerChanged = this._currentPlayerName !== playerName;
      this._currentPlayerName = playerName;

      this._progressSlider.setPlayerName(playerName);

      const metadata = this._getMetadata(playerName, manager);

      if (playerChanged) {
        const savedArt = this._playerArtCache.get(playerName);
        if (savedArt && savedArt === info.artUrl) {
          this._albumArt.loadCover(info.artUrl, true);
        } else if (info.artUrl) {
          this._albumArt.loadCover(info.artUrl, true);
        } else {
          this._albumArt.setDefaultCover();
        }
      } else {
        if (
          info.artUrl &&
          this._playerArtCache.get(playerName) !== info.artUrl
        ) {
          this._albumArt.loadCover(info.artUrl);
        }
      }

      this._playerArtCache.set(playerName, info.artUrl);

      // Use title scrolling instead of just setting text
      const titleText = info.title || globalThis._?.("Unknown") || "Unknown";
      this._startTitleScrolling(titleText);

      if (info.artists && info.artists.length > 0) {
        const artistText = info.artists.join(", ");
        this._startArtistScrolling(artistText);
        this._artistLabel.show();
      } else {
        this._stopArtistScrolling();
        this._artistLabel.hide();
      }

      this._controlButtons.updateButtons(info);

      this._progressSlider.updatePlaybackState(
        info.status === "Playing",
        metadata,
        info.status,
      );

      // Control album art rotation based on playback status
      if (info.status === "Playing") {
        this._albumArt.startRotation(true);
      } else if (info.status === "Paused") {
        this._albumArt.pauseRotation();
      } else {
        this._albumArt.stopRotation();
      }
    }

    _getMetadata(playerName, manager) {
      if (!playerName || !manager) return null;

      const proxy = manager._proxies.get(playerName);
      if (!proxy) return null;

      const metaV = proxy.get_cached_property("Metadata");
      if (!metaV) return null;

      const meta = {};
      const len = metaV.n_children();

      for (let i = 0; i < len; i++) {
        const item = metaV.get_child_value(i);
        const key = item.get_child_value(0).get_string()[0];
        const valueVariant = item.get_child_value(1).get_variant();

        if (key) {
          meta[key] = valueVariant ? valueVariant.recursiveUnpack() : null;
        }
      }

      return meta;
    }

    updateTabs(players, currentPlayer, manager) {
      this._playerTabs.updateTabs(players, currentPlayer, manager);
    }

    startPositionUpdate() {
      this._progressSlider.startPositionUpdate();
    }

    stopPositionUpdate() {
      this._progressSlider.stopPositionUpdate();
      this._stopArtistScrolling();
      this._stopTitleScrolling();
      
      // Stop album art rotation when menu closes
      if (this._albumArt) {
        this._albumArt.pauseRotation();
      }
    }

    onSeeked(position) {
      this._progressSlider.onSeeked(position);

      if (this._currentPlayerName) {
        this._playerSliderPositions.set(this._currentPlayerName, {
          position: this._progressSlider.currentPosition,
          length: this._progressSlider.trackLength,
          value:
            this._progressSlider.currentPosition /
            this._progressSlider.trackLength,
        });
      }
    }

    destroy() {
      this.stopPositionUpdate();
      this._stopArtistScrolling();
      this._stopTitleScrolling();

      this._playerSliderPositions.clear();
      this._playerArtCache.clear();

      if (this._albumArt) {
        this._albumArt.destroy();
      }

      super.destroy();
    }
  },
);