import Clutter from 'gi://Clutter';
import GLib from 'gi://GLib';
import GObject from 'gi://GObject';
import St from 'gi://St';
import Gio from 'gi://Gio';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import * as PopupMenu from 'resource:///org/gnome/shell/ui/popupMenu.js';
import * as Slider from 'resource:///org/gnome/shell/ui/slider.js';
import { Extension, gettext as _ } from 'resource:///org/gnome/shell/extensions/extension.js';

import { QuranPlayer } from './player.js';
import { STATIONS, CATEGORIES } from './stations.js';

const LANGUAGES = [
  { code: 'ar', name: 'العربية' },
  { code: 'en', name: 'English' }
];

const TRANSLATIONS = {
  ar: {
    title: 'قائمة الإذاعات',
    favorites: 'المفضلة',
    search: 'بحث...',
    noResults: 'لا توجد نتائج',
    playing: '-- يعمل الآن --',
    paused: '-- متوقف --',
    all: 'الكل',
    quran: 'قرآن',
    tafsir: 'تفسير',
    fatawa: 'فتاوى',
    miscellaneous: 'منوعات',
    translations: 'ترجمات',
    athkar: 'أذكار',
    ruqyah: 'رقية شرعية',
    hadith: 'الحديث',
    seerah: 'السيرة النبوية',
    companions: 'سيرة الصحابة',
    warsh: 'رواية ورش عن نافع',
    qalun: 'رواية قالون عن نافع',
    alduri: 'رواية الدوري عن أبي عمرو',
    alsusi: 'رواية السوسي عن أبي عمرو',
    otherRiwayat: 'روايات أخرى',
    tooltip: 'Quran Radio'
  },
  en: {
    title: 'Radio Stations',
    favorites: 'Favorites',
    search: 'Search...',
    noResults: 'No results found',
    playing: '-- Playing --',
    paused: '-- Paused --',
    all: 'All',
    quran: 'Quran',
    tafsir: 'Tafsir',
    fatawa: 'Fatawa',
    miscellaneous: 'Miscellaneous',
    translations: 'Translations',
    athkar: 'Athkar',
    ruqyah: 'Ruqyah',
    hadith: 'Hadith',
    seerah: 'Prophet Biography',
    companions: 'Companions Biography',
    warsh: 'Warsh Narration',
    qalun: 'Qalun Narration',
    alduri: 'Al-Duri Narration',
    alsusi: 'Al-Susi Narration',
    otherRiwayat: 'Other Narrations',
    tooltip: 'Quran Radio'
  }
};

// Category mapping for translation
const CATEGORY_TRANSLATIONS = {
  ar: {
    'المفضلة': 'المفضلة',
    'الكل': 'الكل',
    'قرآن': 'قرآن',
    'تفسير': 'تفسير',
    'فتاوى': 'فتاوى',
    'منوعات': 'منوعات',
    'ترجمات': 'ترجمات',
    'أذكار': 'أذكار',
    'رقية شرعية': 'رقية شرعية',
    'الحديث': 'الحديث',
    'السيرة النبوية': 'السيرة النبوية',
    'سيرة الصحابة': 'سيرة الصحابة',
    'رواية ورش عن نافع': 'رواية ورش عن نافع',
    'رواية قالون عن نافع': 'رواية قالون عن نافع',
    'رواية الدوري عن أبي عمرو': 'رواية الدوري عن أبي عمرو',
    'رواية السوسي عن أبي عمرو': 'رواية السوسي عن أبي عمرو',
    'روايات أخرى': 'روايات أخرى'
  },
  en: {
    'المفضلة': 'Favorites',
    'الكل': 'All',
    'قرآن': 'Quran',
    'تفسير': 'Tafsir',
    'فتاوى': 'Fatawa',
    'منوعات': 'Miscellaneous',
    'ترجمات': 'Translations',
    'أذكار': 'Athkar',
    'رقية شرعية': 'Ruqyah',
    'الحديث': 'Hadith',
    'السيرة النبوية': 'Seerah',
    'سيرة الصحابة': 'Companions',
    'رواية ورش عن نافع': 'Warsh',
    'رواية قالون عن نافع': 'Qalun',
    'رواية الدوري عن أبي عمرو': 'Al-Duri',
    'رواية السوسي عن أبي عمرو': 'Al-Susi',
    'روايات أخرى': 'Other Narrations'
  }
};

const QuranIndicator = GObject.registerClass({
  GTypeName: 'QuranIndicator',
}, class QuranIndicator extends PanelMenu.Button {
  _init(extension) {
    super._init(0.0, 'Quran Radio');

    this._extension = extension;
    this._settings = extension.getSettings('org.gnome.shell.extensions.quran-radio');
    this._player = new QuranPlayer(this._settings);

    this._favorites = this._settings.get_strv('favorites');
    this._selectedCategory = this._settings.get_string('selected-category') || 'الكل';
    this._searchQuery = '';
    this._stationItems = new Map();
    this._categoryButtons = [];
    this._currentLang = 'ar';
    this._langMenuOpen = false;
    this._currentLangMenu = null;

    this._buildUI();
    this._connectSignals();
    this._restoreLastStation();
  }

  _getText(key) {
    return TRANSLATIONS[this._currentLang][key] || TRANSLATIONS['ar'][key];
  }

  _getCategoryText(category) {
    return CATEGORY_TRANSLATIONS[this._currentLang][category] || category;
  }

  _buildUI() {
    const box = new St.BoxLayout({
      style_class: 'panel-status-menu-box',
      vertical: false,
    });

    // Load custom icon from extension directory
    const extensionPath = this._extension.path;
    const iconPath = GLib.build_filenamev([extensionPath, 'icon.ico']);
    const iconFile = Gio.File.new_for_path(iconPath);
    const icon = new Gio.FileIcon({ file: iconFile });

    this._icon = new St.Icon({
      gicon: icon,
      style_class: 'system-status-icon',
      icon_size: 16,
    });

    box.add_child(this._icon);
    this.add_child(box);

    // Add tooltip on hover
    this.tooltip = new St.Label({
      text: this._getText('tooltip'),
      style_class: 'quran-tooltip',
    });
    this.tooltip.hide();
    Main.layoutManager.addChrome(this.tooltip);

    this.connect('notify::hover', () => {
      if (this.hover) {
        const [x, y] = this.get_transformed_position();
        const width = this.get_width();
        this.tooltip.set_position(
          Math.floor(x + width / 2 - this.tooltip.get_width() / 2),
          Math.floor(y + this.get_height() + 5)
        );
        this.tooltip.show();
      } else {
        this.tooltip.hide();
      }
    });

    this._buildMenu();
  }

  _buildMenu() {
    this.menu.removeAll();

    this._buildControlSection();
    this._buildHeader();
    this._buildCategoryFilter();
    this._buildSearch();
    this._buildCategoryTags();
    this._buildStationsList();
  }

  _buildControlSection() {
    const controlItem = new PopupMenu.PopupBaseMenuItem({
      reactive: false,
      can_focus: false,
    });

    const controlBox = new St.BoxLayout({
      style_class: 'quran-control-box',
      vertical: true,
    });

    // Top row: Play button on left, status on right
    const topRow = new St.BoxLayout({
      style_class: 'quran-control-top',
      vertical: false,
    });

    // Play button (square style)
    this._playButtonIcon = new St.Icon({
      icon_name: 'media-playback-start-symbolic',
      icon_size: 24,
    });
    
    this._playButton = new St.Button({
      style_class: 'quran-play-button-square',
      child: this._playButtonIcon,
    });

    // Status section (right side)
    const statusBox = new St.BoxLayout({
      style_class: 'quran-status-box',
      vertical: true,
    });

    this._statusLabel = new St.Label({
      text: this._getText('paused'),
      style_class: 'quran-status-text',
    });

    this._stationLabel = new St.Label({
      text: '',
      style_class: 'quran-station-name-display',
    });

    statusBox.add_child(this._statusLabel);
    statusBox.add_child(this._stationLabel);

    topRow.add_child(this._playButton);
    topRow.add_child(statusBox);
    controlBox.add_child(topRow);

    // Volume row with icons on both sides
    const volumeRow = new St.BoxLayout({
      style_class: 'quran-volume-row',
      vertical: false,
    });

    const lowVolumeIcon = new St.Icon({
      icon_name: 'audio-volume-low-symbolic',
      icon_size: 14,
    });

    this._volumeSlider = new Slider.Slider(this._player.getVolume());
    this._volumeSlider.style_class = 'quran-volume-slider';

    const highVolumeIcon = new St.Icon({
      icon_name: 'audio-volume-high-symbolic',
      icon_size: 14,
    });

    volumeRow.add_child(lowVolumeIcon);
    volumeRow.add_child(this._volumeSlider);
    volumeRow.add_child(highVolumeIcon);
    controlBox.add_child(volumeRow);

    controlItem.add_child(controlBox);
    this.menu.addMenuItem(controlItem);
  }

  _buildHeader() {
    const headerItem = new PopupMenu.PopupBaseMenuItem({
      reactive: false,
      can_focus: false,
    });

    const headerBox = new St.BoxLayout({
      style_class: 'quran-header-box',
      vertical: false,
    });

    // Favorites button on the left
    this._favoritesButton = new St.Button({
      style_class: 'quran-favorites-filter-button',
      label: this._getText('favorites'),
    });

    // Title in center
    this._titleLabel = new St.Label({
      text: this._getText('title'),
      style_class: 'quran-title',
    });

    // Language dropdown on right
    this._langButton = new St.Button({
      style_class: 'quran-lang-dropdown',
    });

    const langContent = new St.BoxLayout({
      style_class: 'quran-lang-content',
      vertical: false,
    });

    this._langLabel = new St.Label({
      text: LANGUAGES.find(l => l.code === this._currentLang)?.name || 'العربية',
      style_class: 'quran-lang-text',
    });

    const langIcon = new St.Icon({
      icon_name: 'pan-down-symbolic',
      icon_size: 12,
    });

    langContent.add_child(this._langLabel);
    langContent.add_child(langIcon);
    this._langButton.set_child(langContent);

    // Language dropdown menu
    this._langButton.connect('clicked', () => {
      this._showLanguageMenu();
    });

    headerBox.add_child(this._favoritesButton);
    // Title moved below search bar
    headerBox.add_child(this._langButton);

    headerItem.add_child(headerBox);
    this.menu.addMenuItem(headerItem);

    // Connect favorites button
    this._favoritesButton.connect('clicked', () => {
      this._selectCategory('المفضلة');
    });
  }

  _showLanguageMenu() {
    log('Quran Radio: Language button clicked');
    
    // Toggle menu - close if already open
    if (this._langMenuOpen && this._currentLangMenu) {
      this._closeLanguageMenu();
      return;
    }
    
    this._langMenuOpen = true;
    
    const langMenu = new PopupMenu.PopupMenuSection();
    this._currentLangMenu = langMenu;
    
    LANGUAGES.forEach(lang => {
      const item = new PopupMenu.PopupMenuItem(lang.name);
      item.connect('activate', () => {
        log(`Quran Radio: Language selected - ${lang.code}`);
        this._changeLanguage(lang.code);
        this._closeLanguageMenu();
      });
      if (lang.code === this._currentLang) {
        item.setOrnament(PopupMenu.Ornament.CHECK);
      }
      langMenu.addMenuItem(item);
    });

    // Add to menu at position 2 (after control section and header)
    this.menu.addMenuItem(langMenu, 2);
    log('Quran Radio: Language menu added');
  }

  _closeLanguageMenu() {
    if (this._currentLangMenu) {
      this._currentLangMenu.destroy();
      this._currentLangMenu = null;
    }
    this._langMenuOpen = false;
  }

  _changeLanguage(langCode) {
    this._currentLang = langCode;
    
    // Update UI texts
    this._titleLabel.text = this._getText('title');
    this._favoritesButton.label = this._getText('favorites');
    this._searchEntry.hint_text = this._getText('search');
    this._langLabel.text = LANGUAGES.find(l => l.code === langCode)?.name || 'العربية';
    
    // Update tooltip
    if (this.tooltip) {
      this.tooltip.text = this._getText('tooltip');
    }
    
    // Update status if needed
    if (this._player.isPlaying()) {
      this._statusLabel.text = this._getText('playing');
    } else {
      this._statusLabel.text = this._getText('paused');
    }
    
    // Refresh category tags with new language
    this._updateCategoryTags();
    
    // Refresh stations list to update "no results" text
    this._updateStationsList();
  }

  _updateCategoryTags() {
    // Update existing category button labels
    this._categoryButtons.forEach(({ button, category }) => {
      button.label = this._getCategoryText(category);
    });
  }

  _buildCategoryFilter() {
    // Category filter removed - only tag buttons are used now
  }

  _buildSearch() {
    const searchItem = new PopupMenu.PopupBaseMenuItem({
      reactive: false,
      can_focus: false,
    });

    const searchBox = new St.BoxLayout({
      style_class: 'quran-search-container',
      vertical: false,
    });

    this._searchEntry = new St.Entry({
      style_class: 'quran-search-entry',
      hint_text: this._getText('search'),
      x_expand: true,
    });

    const searchIcon = new St.Icon({
      icon_name: 'edit-find-symbolic',
      icon_size: 16,
      style_class: 'quran-search-icon',
    });

    searchBox.add_child(this._searchEntry);
    searchBox.add_child(searchIcon);
    searchItem.add_child(searchBox);
    this.menu.addMenuItem(searchItem);

    this._searchEntry.clutter_text.connect('text-changed', () => {
      this._searchQuery = this._searchEntry.text.toLowerCase();
      this._updateStationsList();
    });

    // Add title label below search bar
    this._buildTitleSection();
  }

  _buildTitleSection() {
    const titleItem = new PopupMenu.PopupBaseMenuItem({
      reactive: false,
      can_focus: false,
    });

    const titleBox = new St.BoxLayout({
      style_class: 'quran-title-section',
      vertical: false,
      x_align: Clutter.ActorAlign.END,
    });

    this._titleLabel = new St.Label({
      text: this._getText('title'),
      style_class: 'quran-title-below-search',
    });

    titleBox.add_child(this._titleLabel);
    titleItem.add_child(titleBox);
    this.menu.addMenuItem(titleItem);
  }

  _buildCategoryTags() {
    const tagsItem = new PopupMenu.PopupBaseMenuItem({
      reactive: false,
      can_focus: false,
    });

    const tagsScroll = new St.ScrollView({
      style_class: 'quran-tags-scroll',
      hscrollbar_policy: St.PolicyType.NEVER,
      vscrollbar_policy: St.PolicyType.AUTOMATIC,
    });

    const tagsBox = new St.BoxLayout({
      style_class: 'quran-tags-box',
      vertical: true,
    });

    // Filter out 'المفضلة' and 'الكل' from tags
    const tagCategories = CATEGORIES.filter(cat => cat !== 'المفضلة' && cat !== 'الكل');

    // Create rows of tags (3 tags per row)
    const tagsPerRow = 3;
    let currentRow = null;

    this._categoryButtons = [];
    tagCategories.forEach((category, index) => {
      // Create new row every 3 tags
      if (index % tagsPerRow === 0) {
        currentRow = new St.BoxLayout({
          style_class: 'quran-tags-row',
          vertical: false,
        });
        tagsBox.add_child(currentRow);
      }

      const tagButton = new St.Button({
        label: this._getCategoryText(category),
        style_class: 'quran-tag-button',
        x_expand: true,
      });

      if (category === this._selectedCategory) {
        tagButton.add_style_class_name('active');
      }

      tagButton.connect('clicked', () => {
        log(`Quran Radio: Tag clicked - ${category}`);
        this._selectCategory(category);
      });

      this._categoryButtons.push({ button: tagButton, category });
      currentRow.add_child(tagButton);
    });

    tagsScroll.add_child(tagsBox);
    tagsItem.add_child(tagsScroll);
    this.menu.addMenuItem(tagsItem);
  }

  _buildStationsList() {
    const listItem = new PopupMenu.PopupBaseMenuItem({
      reactive: false,
      can_focus: false,
    });

    const scrollView = new St.ScrollView({
      style_class: 'quran-stations-scroll',
      hscrollbar_policy: St.PolicyType.NEVER,
      vscrollbar_policy: St.PolicyType.AUTOMATIC,
    });

    this._stationsBox = new St.BoxLayout({
      style_class: 'quran-stations-container',
      vertical: true,
    });

    scrollView.add_child(this._stationsBox);
    listItem.add_child(scrollView);
    this.menu.addMenuItem(listItem);

    this._updateStationsList();
  }

  _createStationRow(station) {
    const row = new St.BoxLayout({
      style_class: 'quran-station-row',
      vertical: false,
    });

    const isFavorite = this._favorites.includes(station.id);
    const isPlaying = this._player.getCurrentStation()?.id === station.id && this._player.isPlaying();

    // Heart column (fixed width)
    const heartCol = new St.BoxLayout({
      style_class: 'quran-station-col heart-col',
      vertical: false,
      x_align: Clutter.ActorAlign.CENTER,
    });

    const heartButton = new St.Button({
      style_class: 'quran-heart-button',
      child: new St.Icon({
        icon_name: 'emblem-favorite-symbolic',
        icon_size: 18,
      }),
    });

    if (isFavorite) {
      heartButton.add_style_class_name('active');
    }

    heartCol.add_child(heartButton);

    // Name column (expands to fill space)
    const nameCol = new St.BoxLayout({
      style_class: 'quran-station-col name-col',
      vertical: false,
      x_expand: true,
      x_align: Clutter.ActorAlign.END,
    });

    const nameLabel = new St.Label({
      text: station.name,
      style_class: 'quran-station-name',
    });

    if (isPlaying) {
      nameLabel.add_style_class_name('playing');
    }

    nameCol.add_child(nameLabel);

    // Play column (fixed width)
    const playCol = new St.BoxLayout({
      style_class: 'quran-station-col play-col',
      vertical: false,
      x_align: Clutter.ActorAlign.CENTER,
    });

    const playButton = new St.Button({
      style_class: 'quran-station-play-btn',
      child: new St.Icon({
        icon_name: 'media-playback-start-symbolic',
        icon_size: 18,
      }),
    });

    playCol.add_child(playButton);

    // Connect signals
    heartButton.connect('clicked', () => {
      this._toggleFavorite(station);
    });

    playButton.connect('clicked', () => {
      if (isPlaying) {
        this._player.stop();
      } else {
        this._player.play(station);
      }
    });

    row.add_child(heartCol);
    row.add_child(nameCol);
    row.add_child(playCol);

    return row;
  }

  _updateStationsList() {
    log(`Quran Radio: Updating stations list. Category: ${this._selectedCategory}, Search: ${this._searchQuery}`);
    
    const stations = this._getFilteredStations();
    log(`Quran Radio: Found ${stations.length} stations`);

    for (const child of this._stationsBox.get_children()) {
      child.destroy();
    }

    if (stations.length === 0) {
      const noResults = new St.Label({
        text: this._getText('noResults'),
        style_class: 'quran-no-results',
      });
      this._stationsBox.add_child(noResults);
      return;
    }

    stations.forEach(station => {
      const row = this._createStationRow(station);
      this._stationsBox.add_child(row);
    });
  }

  _getFilteredStations() {
    let stations = STATIONS;

    log(`Quran Radio: Filtering - Category: ${this._selectedCategory}, Query: ${this._searchQuery}`);

    if (this._selectedCategory === 'المفضلة') {
      stations = stations.filter(s => this._favorites.includes(s.id));
      log(`Quran Radio: Favorites filter - ${stations.length} stations`);
    } else if (this._selectedCategory !== 'الكل') {
      stations = stations.filter(s => s.category.includes(this._selectedCategory));
      log(`Quran Radio: Category filter - ${stations.length} stations`);
    }

    if (this._searchQuery) {
      stations = stations.filter(s => 
        s.name.toLowerCase().includes(this._searchQuery)
      );
      log(`Quran Radio: Search filter - ${stations.length} stations`);
    }

    return stations;
  }

  _selectCategory(category) {
    log(`Quran Radio: Selecting category - ${category}`);
    
    this._selectedCategory = category;
    this._settings.set_string('selected-category', category);

    // Update tag buttons
    this._categoryButtons.forEach(({ button, category: btnCategory }) => {
      if (btnCategory === category) {
        button.add_style_class_name('active');
      } else {
        button.remove_style_class_name('active');
      }
    });

    // Update favorites button
    if (category === 'المفضلة') {
      this._favoritesButton.add_style_class_name('active');
    } else {
      this._favoritesButton.remove_style_class_name('active');
    }

    this._updateStationsList();
  }

  _showCategoryMenu() {
    // Create a simple submenu for category selection
    const menu = new PopupMenu.PopupMenuSection();
    
    CATEGORIES.forEach(category => {
      const item = new PopupMenu.PopupMenuItem(category);
      item.connect('activate', () => {
        this._selectCategory(category);
        menu.destroy();
      });
      if (category === this._selectedCategory) {
        item.setOrnament(PopupMenu.Ornament.CHECK);
      }
      menu.addMenuItem(item);
    });

    // Add the menu temporarily
    this.menu.addMenuItem(menu);
    
    // Remove it when menu closes
    const closeHandler = () => {
      menu.destroy();
      this.menu.disconnect(closeHandler);
    };
    this.menu.connect('open-state-changed', (menu2, open) => {
      if (!open) closeHandler();
    });
  }

  _toggleFavorite(station) {
    const index = this._favorites.indexOf(station.id);
    if (index > -1) {
      this._favorites.splice(index, 1);
    } else {
      this._favorites.push(station.id);
    }
    this._settings.set_strv('favorites', this._favorites);
    this._updateStationsList();
  }

  _updateStatus(isPlaying, station) {
    log(`Quran Radio: _updateStatus called - isPlaying=${isPlaying}`);
    
    if (isPlaying) {
      this._playButtonIcon.icon_name = 'media-playback-pause-symbolic';
      this._playButton.add_style_class_name('playing');
      this._statusLabel.text = this._getText('playing');
      this._stationLabel.text = station ? station.name : '';
    } else {
      this._playButtonIcon.icon_name = 'media-playback-start-symbolic';
      this._playButton.remove_style_class_name('playing');
      this._statusLabel.text = this._getText('paused');
      this._stationLabel.text = station ? station.name : '';
    }
    this._updateStationsList();
  }

  _connectSignals() {
    this._playButton.connect('clicked', () => {
      log('Quran Radio: Play button clicked');
      log(`Quran Radio: Current icon = ${this._playButtonIcon.icon_name}`);
      
      // Toggle based on current icon (immediate visual feedback)
      const isCurrentlyPlaying = this._playButtonIcon.icon_name === 'media-playback-pause-symbolic';
      
      if (isCurrentlyPlaying) {
        log('Quran Radio: Button shows pause icon, pausing playback');
        this._player.pause();
      } else {
        const station = this._player.getCurrentStation();
        log(`Quran Radio: Button shows play icon, station = ${station?.name || 'none'}`);
        if (station) {
          log(`Quran Radio: Playing station ${station.name}`);
          this._player.play(station);
        } else {
          log('Quran Radio: No station selected, cannot play');
        }
      }
    });

    this._volumeSlider.connect('drag-end', () => {
      log(`Quran Radio: Volume changed to ${this._volumeSlider.value}`);
      this._player.setVolume(this._volumeSlider.value);
    });
    
    // Also connect to value changes for immediate feedback
    this._volumeSlider.connect('notify::value', () => {
      this._player.setVolume(this._volumeSlider.value);
    });

    this._player.onStateChanged((isPlaying, station) => {
      this._updateStatus(isPlaying, station);
    });

    this.menu.connect('open-state-changed', (menu, open) => {
      if (!open) {
        this._searchEntry.text = '';
        this._searchQuery = '';
        // Close language menu when main menu closes
        this._closeLanguageMenu();
      }
    });
  }

  _restoreLastStation() {
    const lastStationId = this._player.getLastStationId();
    if (lastStationId) {
      const station = STATIONS.find(s => s.id === lastStationId);
      if (station) {
        this._stationLabel.text = station.name;
      }
    }
  }

  destroy() {
    // Clean up tooltip
    if (this.tooltip) {
      Main.layoutManager.removeChrome(this.tooltip);
      this.tooltip.destroy();
      this.tooltip = null;
    }
    this._player.destroy();
    super.destroy();
  }
});

export default class QuranRadioExtension extends Extension {
  enable() {
    this._indicator = new QuranIndicator(this);
    Main.panel.addToStatusArea('quran-radio', this._indicator);
  }

  disable() {
    this._indicator?.destroy();
    this._indicator = null;
  }
}