import Gio from 'gi://Gio';
import GLib from 'gi://GLib';
import Gst from 'gi://Gst';

Gst.init(null);

export class QuranPlayer {
  constructor(settings) {
    this._settings = settings;
    this._playbin = null;
    this._currentStation = null;
    this._isPlaying = false;
    this._volume = this._settings.get_double('volume') || 0.7;
    this._reconnectTimer = null;
    this._stateChangedCallbacks = [];
    this._errorCallbacks = [];
    this._bufferingPercent = 0;
    this._sourceSetupId = null;
    
    log(`Quran Radio: Initializing player with volume ${this._volume}`);
  }

  _createPlaybin() {
    if (this._playbin) {
      this._stop();
      this._playbin = null;
    }

    // Try playbin3 first (modern), fallback to playbin
    this._playbin = Gst.ElementFactory.make('playbin3', 'quran-playbin');
    if (!this._playbin) {
      log('Quran Radio: playbin3 not available, using playbin');
      this._playbin = Gst.ElementFactory.make('playbin', 'quran-playbin');
    }

    if (!this._playbin) {
      const error = 'Failed to create GStreamer playbin. Make sure GStreamer is properly installed.';
      logError(new Error(error));
      this._notifyError(error);
      return false;
    }

    // Configure playbin for network streaming
    this._playbin.set_property('buffer-size', 1024 * 1024); // 1MB buffer
    this._playbin.set_property('buffer-duration', 5 * Gst.SECOND); // 5 seconds buffer
    
    // Connect to source-setup signal to configure HTTP source with headers
    this._sourceSetupId = this._playbin.connect('source-setup', (playbin, source) => {
      log(`Quran Radio: Source element setup: ${source.get_name()}`);
      
      // Configure souphttpsrc or similar HTTP source
      if (source.set_property) {
        try {
          // Set User-Agent to mimic browser
          if (source.get_property('user-agent') !== undefined) {
            source.set_property('user-agent', 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36');
            log('Quran Radio: Set User-Agent header');
          }
          
          // Set additional headers
          if (source.get_property('extra-headers') !== undefined) {
            const headers = Gst.Structure.new_empty('headers');
            // Add any custom headers here if needed
            source.set_property('extra-headers', headers);
          }
          
          // Enable automatic redirect following
          if (source.get_property('automatic-redirect') !== undefined) {
            source.set_property('automatic-redirect', true);
          }
          
          // Set connection speed
          if (source.get_property('connection-speed') !== undefined) {
            source.set_property('connection-speed', 0); // 0 = unlimited
          }
        } catch (e) {
          log(`Quran Radio: Could not set source property: ${e.message}`);
        }
      }
    });
    
    const bus = this._playbin.get_bus();
    bus.add_signal_watch();
    bus.connect('message', this._onBusMessage.bind(this));
    
    log('Quran Radio: Playbin created successfully');
    return true;
  }

  _onBusMessage(bus, message) {
    switch (message.type) {
      case Gst.MessageType.ERROR:
        const [error, debug] = message.parse_error();
        const errorMsg = error ? error.message : 'Unknown error';
        logError(error || new Error('Unknown error'), `Quran Radio Error: ${debug || errorMsg}`);
        this._notifyError(`Playback error: ${errorMsg}`);
        this._isPlaying = false;
        this._notifyStateChanged();
        this._scheduleReconnect();
        break;

      case Gst.MessageType.EOS:
        log('Quran Radio: End of stream reached');
        this._scheduleReconnect();
        break;

      case Gst.MessageType.STATE_CHANGED:
        if (message.src === this._playbin) {
          const [oldState, newState, pendingState] = message.parse_state_changed();
          log(`Quran Radio: State changed from ${this._stateToString(oldState)} to ${this._stateToString(newState)}`);
          
          if (newState === Gst.State.PLAYING) {
            this._isPlaying = true;
            log('Quran Radio: Now playing');
          } else if (newState === Gst.State.PAUSED) {
            log('Quran Radio: Paused');
          } else if (newState === Gst.State.NULL) {
            this._isPlaying = false;
            log('Quran Radio: Stopped');
          }
          this._notifyStateChanged();
        }
        break;

      case Gst.MessageType.BUFFERING:
        const percent = message.parse_buffering();
        this._bufferingPercent = percent;
        
        if (percent < 100) {
          if (this._isPlaying && this._playbin) {
            log(`Quran Radio: Buffering ${percent}% - pausing`);
            this._playbin.set_state(Gst.State.PAUSED);
          }
        } else {
          if (this._isPlaying && this._playbin) {
            log('Quran Radio: Buffering complete - resuming');
            this._playbin.set_state(Gst.State.PLAYING);
          }
        }
        break;

      case Gst.MessageType.STREAM_STATUS:
        const [statusType, _owner] = message.parse_stream_status();
        log(`Quran Radio: Stream status - ${statusType}`);
        break;

      case Gst.MessageType.ASYNC_DONE:
        log('Quran Radio: Async operation complete');
        break;

      case Gst.MessageType.TAG:
        // Handle metadata tags if needed
        break;

      default:
        // Ignore other message types
        break;
    }
  }

  _stateToString(state) {
    switch (state) {
      case Gst.State.VOID_PENDING: return 'VOID_PENDING';
      case Gst.State.NULL: return 'NULL';
      case Gst.State.READY: return 'READY';
      case Gst.State.PAUSED: return 'PAUSED';
      case Gst.State.PLAYING: return 'PLAYING';
      default: return 'UNKNOWN';
    }
  }

  play(station) {
    log(`Quran Radio: Attempting to play ${station.name}`);
    log(`Quran Radio: URL: ${station.url}`);
    
    if (!this._createPlaybin()) {
      return false;
    }

    if (this._currentStation && this._currentStation.id !== station.id) {
      this._saveLastStation(station);
    }

    this._currentStation = station;
    this._playbin.set_property('uri', station.url);
    this._setVolume(this._volume);

    // Check if this is an HTTP stream
    if (station.url.startsWith('http')) {
      log('Quran Radio: HTTP stream detected - will configure headers');
    }

    // Set playing state immediately for responsive UI
    this._isPlaying = true;
    this._notifyStateChanged();

    const ret = this._playbin.set_state(Gst.State.PLAYING);
    if (ret === Gst.StateChangeReturn.FAILURE) {
      const error = 'Failed to start playback - GStreamer state change failed';
      logError(new Error(error), 'Quran Radio');
      this._isPlaying = false;
      this._notifyStateChanged();
      this._notifyError(error);
      return false;
    } else if (ret === Gst.StateChangeReturn.ASYNC) {
      log('Quran Radio: State change is async, waiting...');
    } else {
      log('Quran Radio: State change successful');
    }

    this._clearReconnectTimer();
    return true;
  }

  pause() {
    log('Quran Radio: Pausing playback');
    // Set paused state immediately for responsive UI
    this._isPlaying = false;
    this._notifyStateChanged();
    
    if (this._playbin) {
      this._playbin.set_state(Gst.State.PAUSED);
    }
    this._clearReconnectTimer();
  }

  stop() {
    log('Quran Radio: Stopping playback');
    // Set stopped state immediately for responsive UI
    this._isPlaying = false;
    this._notifyStateChanged();
    this._stop();
    this._clearReconnectTimer();
  }

  _stop() {
    if (this._playbin) {
      if (this._sourceSetupId) {
        try {
          this._playbin.disconnect(this._sourceSetupId);
        } catch (e) {
          // Ignore disconnect errors
        }
        this._sourceSetupId = null;
      }
      
      this._playbin.set_state(Gst.State.NULL);
      const bus = this._playbin.get_bus();
      bus.remove_signal_watch();
    }
    this._isPlaying = false;
  }

  _scheduleReconnect() {
    if (this._reconnectTimer) {
      return;
    }

    if (this._currentStation && this._isPlaying) {
      log('Quran Radio: Scheduling reconnect in 5 seconds...');
      this._reconnectTimer = GLib.timeout_add_seconds(GLib.PRIORITY_DEFAULT, 5, () => {
        this._reconnectTimer = null;
        if (this._currentStation) {
          log('Quran Radio: Attempting to reconnect...');
          this.play(this._currentStation);
        }
        return GLib.SOURCE_REMOVE;
      });
    }
  }

  _clearReconnectTimer() {
    if (this._reconnectTimer) {
      GLib.source_remove(this._reconnectTimer);
      this._reconnectTimer = null;
    }
  }

  setVolume(volume) {
    this._volume = Math.max(0, Math.min(1, volume));
    log(`Quran Radio: Setting volume to ${this._volume}`);
    this._setVolume(this._volume);
    this._settings.set_double('volume', this._volume);
  }

  _setVolume(volume) {
    if (this._playbin) {
      this._playbin.set_property('volume', volume);
    }
  }

  getVolume() {
    return this._volume;
  }

  getCurrentStation() {
    return this._currentStation;
  }

  isPlaying() {
    return this._isPlaying;
  }

  getBufferingPercent() {
    return this._bufferingPercent;
  }

  onStateChanged(callback) {
    this._stateChangedCallbacks.push(callback);
    return this._stateChangedCallbacks.length - 1;
  }

  _notifyStateChanged() {
    for (const callback of this._stateChangedCallbacks) {
      callback(this._isPlaying, this._currentStation);
    }
  }

  onError(callback) {
    this._errorCallbacks.push(callback);
    return this._errorCallbacks.length - 1;
  }

  _notifyError(errorMessage) {
    for (const callback of this._errorCallbacks) {
      callback(errorMessage);
    }
  }

  _saveLastStation(station) {
    this._settings.set_string('last-station', station.id);
  }

  getLastStationId() {
    return this._settings.get_string('last-station');
  }

  destroy() {
    this._stop();
    this._clearReconnectTimer();
    this._stateChangedCallbacks = [];
    this._errorCallbacks = [];
    this._playbin = null;
  }
}
