import Gio from "gi://Gio";

Gio._promisify(
  Gio.Subprocess.prototype,
  "communicate_utf8_async",
  "communicate_utf8_finish",
);

export class StorageSensor {
  _currentUsage = 0;
  _total = "0";
  _used = "0";
  _available = "0";

  _formatBytes(bytes) {
    const unit = 1000;
    if (bytes < unit) return `${bytes} B`;
    const gb = bytes / Math.pow(unit, 3);

    if (gb >= 1) {
      // GNOME System Monitor usually shows 1 decimal place (e.g., 253.3 GB)
      return `${gb.toFixed(1)} GB`;
    }

    const mb = bytes / Math.pow(unit, 2);
    return `${mb.toFixed(1)} MB`;
  }

  async getValue() {
    try {
      const path = "/";
      // Use --si to get decimal units or -B1 for raw bytes to calculate manually
      // We use -B1 to get the most accurate raw integer for our own formatting
      const proc = Gio.Subprocess.new(
        ["df", "-B1", "--output=size,used,avail,pcent", path],
        Gio.SubprocessFlags.STDOUT_PIPE,
      );
      const [stdout] = await proc.communicate_utf8_async(null, null);
      if (!stdout) return 0;
      const lines = stdout.trim().split("\n");
      if (lines.length < 2) return 0;
      // Extract values from the second line, splitting by any whitespace
      const columns = lines[1].trim().split(/\s+/);

      if (columns.length >= 4) {
        const totalBytes = parseInt(columns[0]);
        const usedBytes = parseInt(columns[1]);
        const availableBytes = parseInt(columns[2]);
        this._total = this._formatBytes(totalBytes);
        this._used = this._formatBytes(usedBytes);
        this._available = this._formatBytes(availableBytes);
        // The percentage is usually the 4th column in this specific output
        const usageStr = columns[3].replace("%", "");
        this._currentUsage = usageStr ? parseInt(usageStr) : 0;
      }
      return this._currentUsage;
    } catch (e) {
      console.error(`StorageSensor Error: ${e}`);
      return 0;
    }
  }

  async getDetails() {
    const details = [
      { label: "Total Capacity", value: this._total },
      { label: "Used Space", value: this._used },
      { label: "Available", value: this._available },
      { label: "Usage (%)", value: `${Math.round(this._currentUsage)}%` },
    ];
    return details;
  }
}
