const { GObject, Clutter, St, GLib } = imports.gi;
const { panelMenu, popupMenu, main } = imports.ui;
const { getCurrentExtension } = imports.misc.extensionUtils;
const AppManager = getCurrentExtension().imports.AppManager;

var TrayIndicator = GObject.registerClass(
  class TrayIndicator extends panelMenu.Button {
    _init(settings) {
      this._icons = [];

      super._init(0.0, null, false);
      this._settings = settings;
      this._appManager = new AppManager.AppManager(this._settings);
      this._overflow = false;
      
      // Initialize size/margin/padding from settings immediately
      this._size = this._settings.get_int('icon-size');
      this._margin = { 
        vertical: this._settings.get_int('icon-margin-vertical'), 
        horizontal: this._settings.get_int('icon-margin-horizontal') 
      };
      this._padding = { 
        vertical: this._settings.get_int('icon-padding-vertical'), 
        horizontal: this._settings.get_int('icon-padding-horizontal') 
      };

      // Use margin from settings for spacing, minimum 2px
      const spacing = Math.max(this._margin.horizontal, 2);
      this._indicators = new St.BoxLayout({ 
        style_class: 'tray-icons-box',
        x_expand: false,
        y_expand: false,
        x_align: Clutter.ActorAlign.START,
        style: `spacing: ${spacing}px;`,
      });
      this.add_child(this._indicators);

      this._icon = new St.Icon({
        icon_name: "view-more-horizontal",
        style_class: "system-status-icon",
        reactive: true,
        track_hover: true,
      });
      this._indicators.add_child(this._icon);

      this._menuItem = new popupMenu.PopupBaseMenuItem({
        reactive: false,
        can_focus: true,
      });
      this.menu.addMenuItem(this._menuItem);
      this.menu.actor.add_style_class_name("TrayIndicatorPopup");
      this.hide();
      
      // Watch for scale factor changes (display scaling)
      this._themeContext = St.ThemeContext.get_for_stage(global.stage);
      this._scaleFactorChangedId = this._themeContext.connect('notify::scale-factor', () => {
        this._refreshAllIconSizes();
      });
      
      // Watch for allocation changes on TrayIndicator itself (triggered by panel layout changes)
      // Use debounce to prevent multiple rapid calls
      this._selfAllocationId = this.connect('notify::allocation', () => {
        if (this._refreshPending) return;
        this._refreshPending = true;
        GLib.timeout_add(GLib.PRIORITY_DEFAULT, 100, () => {
          this._refreshPending = false;
          this._refreshAllIconSizes();
          return GLib.SOURCE_REMOVE;
        });
      });
      
      // Watch for changes in the panel right box (where VPN and system icons appear)
      this._panelRightBoxId = main.panel._rightBox.connect('notify::allocation', () => {
        if (this._refreshPending) return;
        this._refreshPending = true;
        GLib.timeout_add(GLib.PRIORITY_DEFAULT, 100, () => {
          this._refreshPending = false;
          this._refreshAllIconSizes();
          return GLib.SOURCE_REMOVE;
        });
      });
      
      this.connect('destroy', this._onDestroy.bind(this));
    }

    get size() {
      const context = St.ThemeContext.get_for_stage(global.stage);
      return this._size * context.scale_factor;
    }

    setSize(size, margin, padding) {
      this._size = size;
      this._margin = margin;
      this._padding = padding;

      this._refreshAllIconSizes();
    }
    
    _refreshAllIconSizes() {
      const expectedSize = this.size;
      this._icons.forEach((icon, index) => {
        const button = icon.get_parent();
        // Skip invalid icons (destroyed or not in stage)
        if (!button || !button.get_stage()) {
          return;
        }
        const allocation = button.get_allocation_box();
        // Skip buttons without valid allocation
        if (!isFinite(allocation.x1)) {
          return;
        }
        // Check if button is actually in overflow menu (not in _indicators)
        const isInOverflow = button.get_parent() !== this._indicators;
        
        // Log current state before changes
        const currentWidth = allocation.x2 - allocation.x1;
        const [iconW, iconH] = icon.get_size();
        
        let totalWidth, totalHeight;
        if (isInOverflow) {
          const overflowPadding = this._padding.horizontal > 0 ? this._padding.horizontal : 8;
          totalWidth = expectedSize + (overflowPadding * 2);
          totalHeight = expectedSize + (overflowPadding * 2);
        } else {
          totalWidth = expectedSize + (this._padding.horizontal * 2);
          totalHeight = expectedSize + (this._padding.vertical * 2);
        }
        
        
        // Apply style and size
        button.style = this._getButtonStyle(isInOverflow);
        button.set_size(totalWidth, totalHeight);
        icon.set_size(expectedSize, expectedSize);
      });
      // Force complete relayout by hiding and showing
      this._indicators.hide();
      this._indicators.show();
      this._indicators.queue_relayout();
    }

    addIcon(icon) {
      const isHidden = this._appManager.getAppSetting(icon, "hidden");
      if (isHidden) return;

      const isAlwaysOnTop = this._appManager.getAppSetting(icon, "alwaysOnTop");
      const button = new St.Button({
        child: icon,
        button_mask:
          St.ButtonMask.ONE | St.ButtonMask.TWO | St.ButtonMask.THREE,
        style: this._getButtonStyle(this._overflow),
        style_class: "tray-icon-button",
        x_expand: false,
        y_expand: false,
      });
      icon.opacity = 0;
      icon.set_x_align(Clutter.ActorAlign.CENTER);
      icon.set_y_align(Clutter.ActorAlign.CENTER);
      icon.set_x_expand(false);
      icon.set_y_expand(false);
      icon.inOverflow = this._overflow;
      icon.timeout = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 80, () => {
        icon.set_size(this.size, this.size);
        // Check if button is actually in overflow menu
        const isInOverflow = button.get_parent() !== this._indicators;
        button.style = this._getButtonStyle(isInOverflow);
        // Explicitly set button size to prevent layout changes
        let totalWidth, totalHeight;
        if (isInOverflow) {
          // For overflow, use larger padding
          const overflowPadding = this._padding.horizontal > 0 ? this._padding.horizontal : 8;
          totalWidth = this.size + (overflowPadding * 2);
          totalHeight = this.size + (overflowPadding * 2);
        } else {
          totalWidth = this.size + (this._padding.horizontal * 2);
          totalHeight = this.size + (this._padding.vertical * 2);
        }
        button.set_size(totalWidth, totalHeight);
        icon.ease({
          opacity: 255,
          duration: 400,
          mode: Clutter.AnimationMode.EASE_OUT_QUAD,
        });
        this._addEffectIcon(icon);
        if (!isInOverflow) {
          this._indicators.queue_relayout();
        }
        return GLib.SOURCE_REMOVE;
      });
      
      // Watch for icon size changes and force reset (with loop protection)
      icon._sizeChangedId = icon.connect('notify::size', () => {
        const expectedSize = this.size;
        const [currentWidth, currentHeight] = icon.get_size();
        // Only reset if size actually differs (prevents infinite loop)
        if (Math.abs(currentWidth - expectedSize) > 1 || Math.abs(currentHeight - expectedSize) > 1) {
          GLib.idle_add(GLib.PRIORITY_DEFAULT, () => {
            icon.set_size(expectedSize, expectedSize);
            const isInOverflow = button.get_parent() !== this._indicators;
            button.style = this._getButtonStyle(isInOverflow);
            let totalWidth, totalHeight;
            if (isInOverflow) {
              const overflowPadding = this._padding.horizontal > 0 ? this._padding.horizontal : 8;
              totalWidth = expectedSize + (overflowPadding * 2);
              totalHeight = expectedSize + (overflowPadding * 2);
            } else {
              totalWidth = expectedSize + (this._padding.horizontal * 2);
              totalHeight = expectedSize + (this._padding.vertical * 2);
            }
            button.set_size(totalWidth, totalHeight);
            if (!isInOverflow) {
              this._indicators.queue_relayout();
            }
            return GLib.SOURCE_REMOVE;
          });
        }
      });
      
      // Watch for button allocation changes
      button._allocationChangedId = button.connect('notify::allocation', () => {
        const alloc = button.get_allocation_box();
        const width = alloc.x2 - alloc.x1;
      });
      
			icon.connect("destroy", () => {
		    try {
          if (icon._sizeChangedId) {
            icon.disconnect(icon._sizeChangedId);
            icon._sizeChangedId = null;
          }
          if (button._allocationChangedId) {
            button.disconnect(button._allocationChangedId);
            button._allocationChangedId = null;
          }
	        button.destroy();
		    } catch (e) {
	        log('button.destroy() error: ' + e.message);
		    }
			});

      button.connect("button-release-event", (actor, event) => {
        this.menu.close();
        switch (event.get_button()) {
          case 1:
            this._appManager.leftClick(icon, event);
            break;
          case 2:
            this._appManager.middleClick(icon, event);
            break;
          case 3:
            icon.click(event);
            break;
        }
      });

      this._icons.push(icon);

      function isEmpty(value){
        return (value == null || value == '');
      }
      if (this._overflow && ( isAlwaysOnTop == false || isEmpty(isAlwaysOnTop) )) {
        this._menuItem.actor.add_child(button); //this._menuItem.actor.add(button);
      } else {
        this._indicators.insert_child_at_index(button, 0);
        //this._indicators.insert_child_below(button, this._indicators.get_next_sibling());
      }
      this.checkOverflow();
    }

    removeIcon(icon, ignoreCheckOverflow) {
      const index = this._icons.indexOf(icon);
      this._icons.splice(index, 1);

      if (icon.timeout) {
        GLib.Source.remove(icon.timeout);
        icon.timeout = null;
      }

      const actor = icon.get_parent();
      actor.remove_actor(icon);
      actor.destroy();

      if (!ignoreCheckOverflow) {
        this.checkOverflow();
      }
    }

    checkOverflow() {
      if (this._icons.length >= this._settings.get_int("icons-limit")) {
        this._overflow = true;
        this._icon.visible = true;
        this.reactive = true;
        this.style_class = "panel-button TrayIndicator";
        this._menuItem.show();
      } else {
        this._overflow = false;
        this._icon.visible = false;
        this.reactive = false;
        this.style_class = "TrayIndicator";
        this._menuItem.hide();
      }

      if (this._icons.length) {
        this.show();
      } else {
        this.hide();
      }

      this._refreshIcons(this._overflow);
    }

    setEffect(contrast, saturation, brightness) {
      this._icons.forEach((icon) => {
        let brightnessContrast = icon.get_effect("brightnessContrast");
        brightnessContrast.set_contrast(contrast / 100);
        brightnessContrast.set_brightness(brightness / 100);

        let desaturate = icon.get_effect("desaturate");
        desaturate.set_factor(saturation / 100);
      });
    }

    _queue_relayout() {
      if (this._indicators.get_stage()) {
          this._indicators.queue_relayout();
      } else {
      }
    }

    _refreshIcons(overflow) {
      this._icons.forEach((icon) => {
        if (icon.inOverflow != overflow) {
          this.removeIcon(icon, true);
          this.addIcon(icon);
        }
      });
      this._queue_relayout();
    }

    _getButtonStyle(isOverflow = false) {
      // Default values if not yet set
      const padding = this._padding || { horizontal: 0, vertical: 0 };
      const size = this.size || 20;
      
      if (!isOverflow) {
        // Include padding in button dimensions to prevent icon overlap
        // No margin here - spacing is handled by container
        const totalWidth = size + (padding.horizontal * 2);
        const totalHeight = size + (padding.vertical * 2);
        const style = `width: ${totalWidth}px; height: ${totalHeight}px; min-width: ${totalWidth}px; min-height: ${totalHeight}px; padding: ${padding.vertical}px ${padding.horizontal}px;`;
        return style;
      }
      // For overflow menu, use size with padding from settings
      const overflowPadding = padding.horizontal > 0 ? padding.horizontal : 8;
      const overflowSize = size + (overflowPadding * 2);
      return `width: ${overflowSize}px; height: ${overflowSize}px; min-width: ${overflowSize}px; min-height: ${overflowSize}px; padding: ${overflowPadding}px;`;
    }

    _addEffectIcon(icon) {
      let brightnessContrast = new Clutter.BrightnessContrastEffect({});
      brightnessContrast.set_contrast(
        this._settings.get_int("icon-contrast") / 100
      );
      brightnessContrast.set_brightness(
        this._settings.get_int("icon-brightness") / 100
      );
      icon.add_effect_with_name("brightnessContrast", brightnessContrast);
      icon.add_effect_with_name(
        "desaturate",
        new Clutter.DesaturateEffect({
          factor: this._settings.get_int("icon-saturation") / 100,
        })
      );
    }

    _onDestroy() {
      if (this._scaleFactorChangedId && this._themeContext) {
        this._themeContext.disconnect(this._scaleFactorChangedId);
        this._scaleFactorChangedId = null;
      }
      if (this._selfAllocationId) {
        this.disconnect(this._selfAllocationId);
        this._selfAllocationId = null;
      }
      if (this._panelRightBoxId) {
        main.panel._rightBox.disconnect(this._panelRightBoxId);
        this._panelRightBoxId = null;
      }
      this._icons.forEach((icon) => {
        this.removeIcon(icon, true);
      });
    }
  }
);
