/* exported getUniqueBusName, getBusNames, getProcessId, introspectBusObject,
   dbusNodeImplementsInterfaces */

const {GLib, Gio} = imports.gi;

let localImports;
try {
    ({imports: localImports} = imports.misc.extensionUtils.getCurrentExtension());
} catch (e) {
    localImports = imports;
}

const {logger: Logger} = localImports;
const {promiseUtils: PromiseUtils} = localImports;

var BUS_ADDRESS_REGEX = /([a-zA-Z0-9._-]+\.[a-zA-Z0-9.-]+)|(:[0-9]+\.[0-9]+)$/;

Gio._promisify(Gio.DBusConnection.prototype, 'call', 'call_finish');
Gio._promisify(Gio._LocalFilePrototype, 'read', 'read_finish');
Gio._promisify(Gio.InputStream.prototype, 'read_bytes_async', 'read_bytes_finish');

const UNSIGNED_VARIANT_TYPE = new GLib.VariantType('(u)');
const STRING_VARIANT_TYPE = new GLib.VariantType('(s)');
const STRING_ARRAY_VARIANT_TYPE = new GLib.VariantType('(as)');

function indicatorId(service, busName, objectPath) {
    if (service && service !== busName && service.match(BUS_ADDRESS_REGEX))
        return service;

    return `${busName}@${objectPath}`;
}

async function getUniqueBusName(bus, name, cancellable) {
    if (name[0] === ':')
        return name;

    if (!bus)
        bus = Gio.DBus.session;

    const variantName = new GLib.Variant('(s)', [name]);
    const [unique] = (await bus.call('org.freedesktop.DBus', '/', 'org.freedesktop.DBus',
        'GetNameOwner', variantName, STRING_VARIANT_TYPE,
        Gio.DBusCallFlags.NONE, -1, cancellable)).deep_unpack();

    return unique;
}

async function getBusNames(bus, cancellable) {
    if (!bus)
        bus = Gio.DBus.session;

    const [names] = (await bus.call('org.freedesktop.DBus', '/', 'org.freedesktop.DBus',
        'ListNames', null, STRING_ARRAY_VARIANT_TYPE, Gio.DBusCallFlags.NONE,
        -1, cancellable)).deep_unpack();

    const uniqueNames = new Map();
    const requests = names.map(name => getUniqueBusName(bus, name, cancellable));
    const results = await Promise.allSettled(requests);

    for (let i = 0; i < results.length; i++) {
        const result = results[i];
        if (result.status === 'fulfilled') {
            let namesForBus = uniqueNames.get(result.value);
            if (!namesForBus) {
                namesForBus = new Set();
                uniqueNames.set(result.value, namesForBus);
            }
            namesForBus.add(result.value !== names[i] ? names[i] : null);
        } else if (!result.reason.matches(Gio.IOErrorEnum, Gio.IOErrorEnum.CANCELLED)) {
            Logger.debug(`Impossible to get the unique name of ${names[i]}: ${result.reason}`);
        }
    }

    return uniqueNames;
}

async function getProcessId(connectionName, cancellable = null, bus = Gio.DBus.session) {
    const res = await bus.call('org.freedesktop.DBus', '/',
        'org.freedesktop.DBus', 'GetConnectionUnixProcessID',
        new GLib.Variant('(s)', [connectionName]),
        UNSIGNED_VARIANT_TYPE,
        Gio.DBusCallFlags.NONE,
        -1,
        cancellable);
    const [pid] = res.deepUnpack();
    return pid;
}

async function introspectT(bus, name, cancellable, interfaces, path) {
    const [introspection] = (await bus.call(name, path,
        'org.freedesktop.DBus.Introspectable', 'Introspect', null,
        STRING_VARIANT_TYPE, Gio.DBusCallFlags.NONE,
        5000, cancellable)).deep_unpack();

    const nodeInfo = Gio.DBusNodeInfo.new_for_xml(introspection);

    return {nodeInfo, path};
}

async function* introspectBusObject(bus, name, cancellable, interfaces = undefined, path = undefined) {
    if (!path)
        path = '/';

    const {nodeInfo} = await introspectT(bus, name, cancellable, interfaces, path);

    if (interfaces && dbusNodeImplementsInterfaces(nodeInfo, interfaces))
        yield {nodeInfo, path};

    // const [introspection] = (await bus.call(name, path,
    //     'org.freedesktop.DBus.Introspectable', 'Introspect', null,
    //     STRING_VARIANT_TYPE, Gio.DBusCallFlags.NONE,
    //     5000, cancellable)).deep_unpack();

    // const nodeInfo = Gio.DBusNodeInfo.new_for_xml(introspection);
    // // const nodes = [];

    // if (interfaces && dbusNodeImplementsInterfaces(nodeInfo, interfaces))
    //     yield { nodeInfo, path };

    if (path === '/')
        path = '';

    // async function* iterateSubNodes() {
    //     for (const subNodeInfo of nodeInfo.nodes) {
    //         const subPath = `${path}/${subNodeInfo.path}`;
    //         yield introspectT(bus, name, cancellable, interfaces, subPath);
    //     }
    // }

    // for await (const subNodes of iterateSubNodes()) {
    //     // print('subnode', subNodes)
    //     yield subNodes
    // }

    // return nodes;

    for (const subNodeInfo of nodeInfo.nodes) {
        const subPath = `${path}/${subNodeInfo.path}`;
        yield* introspectBusObject(bus, name, cancellable, interfaces, subPath);
    }

    // for (const subNodeInfo of nodeInfo.nodes) {
    //     const subPath = `${path}/${subNodeInfo.path}`;
    //     print('subnode', subPath)
    //     yield introspectT(bus, name, cancellable, interfaces, subPath);
    // }
}

function dbusNodeImplementsInterfaces(nodeInfo, interfaces) {
    if (!(nodeInfo instanceof Gio.DBusNodeInfo) || !Array.isArray(interfaces))
        return false;

    return interfaces.some(iface => nodeInfo.lookup_interface(iface));
}
