// SPDX-License-Identifier: GPL-2.0-or-later

import Gtk from 'gi://Gtk';
import Adw from 'gi://Adw';
import Gio from 'gi://Gio';
import Gdk from 'gi://Gdk';
import {ExtensionPreferences} from 'resource:///org/gnome/Shell/Extensions/js/extensions/prefs.js';

import {isValidAccelerator} from './lib/shortcutKeys.js';

const BAR_POSITIONS = ['left', 'center', 'right'];
const BAR_POSITION_LABELS = ['Left', 'Center', 'Right'];
const MAX_WORKSPACES = 10;

export default class WorkspaceShortcutsBarPrefs extends ExtensionPreferences {
    fillPreferencesWindow(window) {
        const settings = this.getSettings();

        const mutterSettings = new Gio.Settings({schema_id: 'org.gnome.mutter'});
        const isDynamic = mutterSettings.get_boolean('dynamic-workspaces');

        this._addAppearancePage(window, settings, isDynamic);
        this._addSwitchShortcutsPage(window, settings, isDynamic);
        this._addMoveShortcutsPage(window, settings, isDynamic);
        this._addBehaviorPage(window, settings);
    }

    _addAppearancePage(window, settings, isDynamic) {
        const page = new Adw.PreferencesPage({
            title: 'Appearance',
            icon_name: 'preferences-desktop-appearance-symbolic',
        });
        window.add(page);

        const group = new Adw.PreferencesGroup({
            title: 'Appearance',
        });
        page.add(group);

        // Bar Position
        const positionRow = new Adw.ComboRow({
            title: 'Bar Position',
            subtitle: 'Where the workspace bar appears in the top panel',
            model: Gtk.StringList.new(BAR_POSITION_LABELS),
        });

        const currentPosition = settings.get_string('bar-position');
        const positionIndex = BAR_POSITIONS.indexOf(currentPosition);
        if (positionIndex >= 0)
            positionRow.selected = positionIndex;

        positionRow.connect('notify::selected', () => {
            const selected = BAR_POSITIONS[positionRow.selected];
            if (selected)
                settings.set_string('bar-position', selected);
        });
        group.add(positionRow);

        // Max Shortcuts (dynamic mode only)
        if (isDynamic) {
            const maxAdjustment = new Gtk.Adjustment({
                lower: 1,
                upper: 10,
                step_increment: 1,
                value: settings.get_int('max-shortcuts'),
            });

            const maxRow = new Adw.SpinRow({
                title: 'Maximum Shortcuts',
                subtitle: 'Number of workspace shortcuts to register',
                adjustment: maxAdjustment,
            });

            settings.bind(
                'max-shortcuts',
                maxAdjustment,
                'value',
                Gio.SettingsBindFlags.DEFAULT
            );

            group.add(maxRow);
        }
    }

    _addSwitchShortcutsPage(window, settings, isDynamic) {
        const page = new Adw.PreferencesPage({
            title: 'Switch Shortcuts',
            icon_name: 'input-keyboard-symbolic',
        });
        window.add(page);

        const group = new Adw.PreferencesGroup({
            title: 'Switch to Workspace',
        });
        page.add(group);

        const rowCount = this._getShortcutRowCount(settings, isDynamic);
        for (let i = 1; i <= rowCount; i++) {
            this._addShortcutRow(group, settings, i, 'wsb-switch-to-workspace');
        }
    }

    _addMoveShortcutsPage(window, settings, isDynamic) {
        const page = new Adw.PreferencesPage({
            title: 'Move Shortcuts',
            icon_name: 'view-sort-ascending-symbolic',
        });
        window.add(page);

        const group = new Adw.PreferencesGroup({
            title: 'Move Window to Workspace',
        });
        page.add(group);

        const rowCount = this._getShortcutRowCount(settings, isDynamic);
        for (let i = 1; i <= rowCount; i++) {
            this._addShortcutRow(group, settings, i, 'wsb-move-to-workspace');
        }
    }

    _addBehaviorPage(window, settings) {
        const page = new Adw.PreferencesPage({
            title: 'Behavior',
            icon_name: 'preferences-other-symbolic',
        });
        window.add(page);

        const group = new Adw.PreferencesGroup({
            title: 'Window Movement',
        });
        page.add(group);

        const followRow = new Adw.SwitchRow({
            title: 'Follow Window',
            subtitle: 'Switch to the target workspace after moving a window',
        });

        settings.bind(
            'move-follows-window',
            followRow,
            'active',
            Gio.SettingsBindFlags.DEFAULT
        );

        group.add(followRow);
    }

    _getShortcutRowCount(settings, isDynamic) {
        if (isDynamic) {
            return settings.get_int('max-shortcuts');
        }

        const wmPrefs = new Gio.Settings({schema_id: 'org.gnome.desktop.wm.preferences'});
        const numWorkspaces = wmPrefs.get_int('num-workspaces');
        return Math.min(numWorkspaces, MAX_WORKSPACES);
    }

    _addShortcutRow(group, settings, workspaceNum, keyPrefix) {
        const keyName = `${keyPrefix}-${workspaceNum}`;
        const bindings = settings.get_strv(keyName);
        const current = bindings.length > 0 ? bindings[0] : '';

        const row = new Adw.ActionRow({
            title: `Workspace ${workspaceNum}`,
        });

        const shortcutLabel = new Gtk.ShortcutLabel({
            accelerator: current,
            disabled_text: 'Disabled',
            valign: Gtk.Align.CENTER,
        });

        const changeButton = new Gtk.Button({
            label: 'Change\u2026',
            valign: Gtk.Align.CENTER,
        });

        changeButton.connect('clicked', () => {
            this._openKeyCaptureDialog(
                row.get_root(),
                settings,
                keyName,
                workspaceNum,
                shortcutLabel
            );
        });

        row.add_suffix(shortcutLabel);
        row.add_suffix(changeButton);
        group.add(row);
    }

    _openKeyCaptureDialog(parentWindow, settings, keyName, workspaceNum, shortcutLabel) {
        const dialog = new Adw.Window({
            modal: true,
            transient_for: parentWindow,
            title: 'Set Shortcut',
            default_width: 300,
            default_height: 150,
            resizable: false,
        });

        const mainBox = new Gtk.Box({
            orientation: Gtk.Orientation.VERTICAL,
            spacing: 12,
            margin_top: 24,
            margin_bottom: 24,
            margin_start: 24,
            margin_end: 24,
        });

        const promptLabel = new Gtk.Label({
            label: 'Press a key combination\u2026',
            halign: Gtk.Align.CENTER,
            valign: Gtk.Align.CENTER,
            vexpand: true,
        });
        mainBox.append(promptLabel);

        const capturedLabel = new Gtk.ShortcutLabel({
            accelerator: '',
            disabled_text: '',
            halign: Gtk.Align.CENTER,
            visible: false,
        });
        mainBox.append(capturedLabel);

        const buttonBox = new Gtk.Box({
            orientation: Gtk.Orientation.HORIZONTAL,
            spacing: 12,
            halign: Gtk.Align.END,
            visible: false,
        });

        const cancelButton = new Gtk.Button({label: 'Cancel'});
        const setButton = new Gtk.Button({
            label: 'Set',
            css_classes: ['suggested-action'],
        });

        buttonBox.append(cancelButton);
        buttonBox.append(setButton);
        mainBox.append(buttonBox);

        const toolbarView = new Adw.ToolbarView();
        toolbarView.add_top_bar(new Adw.HeaderBar());
        toolbarView.set_content(mainBox);
        dialog.set_content(toolbarView);

        let capturedAccelerator = '';

        const keyController = new Gtk.EventControllerKey();
        keyController.connect('key-pressed', (_controller, keyval, _keycode, state) => {
            const mask = state & Gtk.accelerator_get_default_mod_mask();

            if (_isModifierOnly(keyval))
                return true;

            const accelerator = Gtk.accelerator_name(keyval, mask);
            if (!accelerator)
                return true;

            if (!isValidAccelerator(accelerator))
                return true;

            capturedAccelerator = accelerator;
            promptLabel.visible = false;
            capturedLabel.accelerator = accelerator;
            capturedLabel.visible = true;
            buttonBox.visible = true;

            return true;
        });
        dialog.add_controller(keyController);

        setButton.connect('clicked', () => {
            settings.set_strv(keyName, [capturedAccelerator]);
            shortcutLabel.accelerator = capturedAccelerator;
            dialog.close();
        });

        cancelButton.connect('clicked', () => {
            dialog.close();
        });

        const escController = new Gtk.EventControllerKey();
        escController.connect('key-pressed', (_controller, keyval) => {
            if (keyval === Gdk.KEY_Escape && !buttonBox.visible) {
                dialog.close();
                return true;
            }
            return false;
        });
        dialog.add_controller(escController);

        dialog.present();
    }
}

/**
 * Check whether a keyval corresponds to a lone modifier key.
 * @param {number} keyval
 * @returns {boolean}
 */
function _isModifierOnly(keyval) {
    return keyval === Gdk.KEY_Shift_L ||
           keyval === Gdk.KEY_Shift_R ||
           keyval === Gdk.KEY_Control_L ||
           keyval === Gdk.KEY_Control_R ||
           keyval === Gdk.KEY_Alt_L ||
           keyval === Gdk.KEY_Alt_R ||
           keyval === Gdk.KEY_Super_L ||
           keyval === Gdk.KEY_Super_R ||
           keyval === Gdk.KEY_Meta_L ||
           keyval === Gdk.KEY_Meta_R ||
           keyval === Gdk.KEY_Hyper_L ||
           keyval === Gdk.KEY_Hyper_R ||
           keyval === Gdk.KEY_ISO_Level3_Shift ||
           keyval === Gdk.KEY_Caps_Lock ||
           keyval === Gdk.KEY_Num_Lock;
}
