import { clamp } from '../utils.js';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';

class ControllerCoordinator {
    _focusLock = null;
    _allControllers = new Set();
    add(controller) {
        this._allControllers.add(controller);
    }
    remove(controller) {
        this._allControllers.delete(controller);
    }
    acquireFocus(controller) {
        this._focusLock = controller;
    }
    maybeAcquireFocus(controller) {
        if (this._focusLock === controller) {
            return true;
        }
        if (!this._focusLock || !this._focusLock.isGestureRunning) {
            this.acquireFocus(controller);
            return true;
        }
    }
    isTheOnlyRunningGesture(controller) {
        for (const c of this._allControllers) {
            if (c.isGestureRunning && c !== controller) {
                return false;
            }
        }
        return true;
    }
    get hasControllers() {
        return this._allControllers.size > 0;
    }
}
class GestureController {
    static _coordinators = new Map();
    _klass;
    _isGestureRunning = false;
    constructor(klass) {
        this._klass = klass;
        if (!GestureController._coordinators.has(klass)) {
            GestureController._coordinators.set(klass, new ControllerCoordinator());
        }
        this._coordinator.add(this);
    }
    gestureBegin() {
        this.gestureProgress(0);
    }
    gestureProgress(progress) {
        if (this._coordinator.maybeAcquireFocus(this)) {
            if (!this._isGestureRunning) {
                this._doBegin();
            }
            else {
                this._doProgress(progress);
            }
        }
        this._isGestureRunning = true;
    }
    gestureEnd(direction) {
        if (this._coordinator.isTheOnlyRunningGesture(this) && this._isGestureRunning) {
            this._doEnd(direction ?? null);
        }
        this._isGestureRunning = false;
    }
    gestureCancel() {
        if (this._coordinator.isTheOnlyRunningGesture(this) && this._isGestureRunning) {
            this._doCancel();
        }
        this._isGestureRunning = false;
    }
    get isGestureRunning() {
        return this._isGestureRunning;
    }
    get _coordinator() {
        return GestureController._coordinators.get(this._klass);
    }
    destroy() {
        this._coordinator.remove(this);
        if (!this._coordinator.hasControllers) {
            GestureController._coordinators.delete(this._klass);
        }
    }
}
class OverviewGestureController extends GestureController {
    _baseDist = global.screenHeight;
    _initialProgress = 0;
    _currentProgress = 0;
    _cancelProgress = 0;
    constructor() {
        super(OverviewGestureController);
    }
    _doBegin() {
        Main.overview._gestureBegin({
            confirmSwipe: (baseDistance, points, progress, cancelProgress) => {
                this._baseDist = baseDistance;
                // The following tenary expression is needed to fix a bug (presumably in Gnome Shell's
                // OverviewControls) that causes a `progress` of 1 to be passed to this callback on the first
                // gesture begin, even if the overview is not visible:
                this._initialProgress = Main.overview.visible ? progress : 0;
                this._currentProgress = this._initialProgress;
                this._cancelProgress = cancelProgress;
            }
        });
    }
    _doProgress(progress) {
        this._currentProgress = this._initialProgress + progress;
        Main.overview._gestureUpdate({}, this._currentProgress);
    }
    _doEnd(direction) {
        try {
            if (this.isGestureRunning) {
                if (direction === 'up') {
                    Main.overview._gestureEnd(null, 300, clamp(Math.round(this._currentProgress + 0.5), 1, 2));
                }
                else if (direction === 'down') {
                    Main.overview._gestureEnd(null, 300, clamp(Math.round(this._currentProgress - 0.5), 0, 1));
                }
                else {
                    Main.overview._gestureEnd(null, 300, clamp(Math.round(this._currentProgress), 0, 2));
                }
            }
        }
        catch (e) {
        }
    }
    _doCancel() {
        Main.overview._gestureEnd(null, 300, this._cancelProgress);
    }
    get baseDist() {
        return this._baseDist;
    }
    get initialProgress() {
        return this._initialProgress;
    }
    get currentProgress() {
        return this._currentProgress;
    }
}
class WorkspaceGestureController extends GestureController {
    //@ts-ignore
    _wsController = Main.wm._workspaceAnimation;
    _monitorIndex;
    _baseDist = 900;
    _initialProgress = 0;
    _currentProgress = 0;
    _cancelProgress = 0;
    constructor(props) {
        // @ts-ignore
        super(WorkspaceGestureController);
        this._monitorIndex = props.monitorIndex;
    }
    _doBegin() {
        this._wsController._switchWorkspaceBegin({
            confirmSwipe: (baseDistance, points, progress, cancelProgress) => {
                this._baseDist = baseDistance;
                this._initialProgress = progress;
                this._currentProgress = this._initialProgress;
                this._cancelProgress = cancelProgress;
            }
        }, this._monitorIndex ?? Main.layoutManager.primaryIndex);
    }
    _doProgress(progress) {
        this._currentProgress = this._initialProgress + progress;
        this._wsController._switchWorkspaceUpdate({}, this._currentProgress);
    }
    _doEnd(direction) {
        if (this.isGestureRunning) {
            // TODO: debug occasional cases of non-ending gesture
            if (direction === 'left' || direction === 'right') {
                this._wsController._switchWorkspaceEnd({}, 500, Math.round(this._currentProgress + (direction == 'left' ? 0.5 : -0.5)));
            }
            else {
                this._wsController._switchWorkspaceEnd({}, 500, Math.round(this._currentProgress));
            }
        }
    }
    _doCancel() {
        this._wsController._switchWorkspaceEnd({}, 500, this._cancelProgress);
    }
    get baseDist() {
        return this._baseDist;
    }
    get initialProgress() {
        return this._initialProgress;
    }
    get currentProgress() {
        return this._currentProgress;
    }
    get monitorIndex() {
        return this._monitorIndex;
    }
    set monitorIndex(value) {
        this._monitorIndex = value;
    }
}

export { OverviewGestureController, WorkspaceGestureController };
