import Gio from 'gi://Gio';
import Clutter from 'gi://Clutter';
import GLib from 'gi://GLib';
import St from 'gi://St';
import Gst from 'gi://Gst';
import * as PanelMenu from 'resource:///org/gnome/shell/ui/panelMenu.js';
import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import { Extension } from 'resource:///org/gnome/shell/extensions/extension.js';

export default class RelojLCDExtension extends Extension {
    enable() {
        if (!Gst.is_initialized()) Gst.init(null);

        this._settings = this.getSettings();
        this._isAlarming = false;
        this._player = null;
        this._dotState = true;
        this._alarmBlinkState = true;
        this._clockTimeoutId = null;
        this._alarmTimeoutId = null;
        this._blinkTimeoutId = null;
        this._busSignalId = null;

        this._buildIndicator();

        this._settings.connectObject(
            'changed::font-size', () => this._updateStyle(),
                                     'changed::clock-color', () => this._updateStyle(),
                                     'changed::glow-intensity', () => this._updateStyle(),
                                     'changed::show-seconds', () => {
                                         this._updateClock();
                                         this._updateStyle();
                                     },
                                     'changed::blink-dots', () => this._updateClock(),
                                     'changed::clock-format-24h', () => this._updateClock(),
                                     'changed::panel-position', () => {
                                         if (this._indicator) {
                                             this._indicator.destroy();
                                             this._indicator = null;
                                         }
                                         this._buildIndicator();
                                     },
                                     this
        );

        this._updateClock();
    }

    disable() {
        this._stopAlarm();
        this._removeClockTimeout();

        if (this._indicator) {
            this._indicator.destroy();
            this._indicator = null;
        }

        this._settings?.disconnectObject(this);
    }

    _removeClockTimeout() {
        if (this._clockTimeoutId) {
            GLib.Source.remove(this._clockTimeoutId);
            this._clockTimeoutId = null;
        }
    }

    _buildIndicator() {
        const pos = this._settings.get_string('panel-position');
        const showSeconds = this._settings.get_boolean('show-seconds');
        this._indicator = new PanelMenu.Button(0.5, 'RelojLCD', false);
        this._clockLabel = new St.Label({
            text: showSeconds ? '88:88:88' : '88:88',
            y_align: Clutter.ActorAlign.CENTER,
            style_class: 'reloj-lcd-label'
        });
        this._indicator.add_child(this._clockLabel);

        this._indicator.connect('button-press-event', () => {
            if (this._isAlarming) {
                this._stopAlarm();
                return Clutter.EVENT_STOP;
            } else {
                this.openPreferences();
                return Clutter.EVENT_PROPAGATE;
            }
        });

        Main.panel.addToStatusArea('relojlcd', this._indicator, 1, pos);
        this._updateStyle();

        GLib.timeout_add(GLib.PRIORITY_DEFAULT, 100, () => {
            this._updateClock();
            return false;
        });
    }

    _updateClock() {
        this._removeClockTimeout();

        const update = () => {
            const now = GLib.DateTime.new_now_local();
            const is24h = this._settings.get_boolean('clock-format-24h');
            const showSeconds = this._settings.get_boolean('show-seconds');
            const blink = this._settings.get_boolean('blink-dots');

            this._dotState = blink ? !this._dotState : true;
            let sep = this._dotState ? ':' : ' ';

            let timeStr;
            if (showSeconds) {
                let format = is24h ? `%H${sep}%M${sep}%S` : `%I${sep}%M${sep}%S %p`;
                timeStr = now.format(format);
            } else {
                let format = is24h ? `%H${sep}%M` : `%I${sep}%M %p`;
                timeStr = now.format(format);
            }

            this._clockLabel.set_text(timeStr);
            this._checkAlarm(now);
            return true;
        };

        update();
        const blinkEnabled = this._settings.get_boolean('blink-dots');
        const showSeconds = this._settings.get_boolean('show-seconds');
        const interval = blinkEnabled ? 500 : (showSeconds ? 1000 : 10000);
        this._clockTimeoutId = GLib.timeout_add(GLib.PRIORITY_DEFAULT, interval, update);
    }

    _checkAlarm(now) {
        if (!this._settings.get_boolean('alarm-enabled') || this._isAlarming) return;

        if (now.get_hour() === this._settings.get_int('alarm-hour') &&
            now.get_minute() === this._settings.get_int('alarm-minute') &&
            now.get_second() === 0) {
            this._triggerAlarm();
            }
    }

    _triggerAlarm() {
        this._isAlarming = true;

        // Notificación de GNOME
        const message = this._settings.get_string('alarm-message') || '¡Alarma!';
        Main.notify('Reloj LCD', message);

        this._player = Gst.ElementFactory.make('playbin', 'player');
        if (this._player) {
            this._player.set_property('uri', 'file://' + this.path + '/assets/alarm.ogg');
            const bus = this._player.get_bus();
            bus.add_signal_watch();
            this._busSignalId = bus.connect('message::eos', () => {
                this._player.seek_simple(Gst.Format.TIME, Gst.SeekFlags.FLUSH, 0);
            });
            this._player.set_state(Gst.State.PLAYING);
        }

        this._blinkTimeoutId = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 500, () => {
            this._alarmBlinkState = !this._alarmBlinkState;
            this._clockLabel.set_opacity(this._alarmBlinkState ? 255 : 40);
            return true;
        });

        this._alarmTimeoutId = GLib.timeout_add(GLib.PRIORITY_DEFAULT, 60000, () => {
            this._stopAlarm();
            return false;
        });
    }

    _stopAlarm() {
        if (this._player) {
            this._player.set_state(Gst.State.NULL);
            const bus = this._player.get_bus();
            if (this._busSignalId) {
                bus.disconnect(this._busSignalId);
                this._busSignalId = null;
            }
            bus.remove_signal_watch();
            this._player = null;
        }

        if (this._alarmTimeoutId) {
            GLib.Source.remove(this._alarmTimeoutId);
            this._alarmTimeoutId = null;
        }

        if (this._blinkTimeoutId) {
            GLib.Source.remove(this._blinkTimeoutId);
            this._blinkTimeoutId = null;
        }

        this._isAlarming = false;
        if (this._clockLabel) {
            this._clockLabel.set_opacity(255);
            this._updateStyle();
        }
    }

    _updateStyle() {
        if (!this._clockLabel) return;
        const fontSize = this._settings.get_double('font-size');
        const colorType = this._settings.get_string('clock-color');
        const glow = this._settings.get_double('glow-intensity');
        const showSeconds = this._settings.get_boolean('show-seconds');
        const horizontalPadding = showSeconds ? 16 : 24;

        let style = `font-family: 'Digital-7', monospace; font-size: ${fontSize.toFixed(1)}em; font-weight: 200; padding: 2px ${horizontalPadding}px; border-radius: 8px;`;

        if (colorType === 'gray') {
            style += `color: #1a1a1a !important; background-color: rgba(158, 167, 146, 0.95); border: 2px solid #555; text-shadow: 2px 2px 0px rgba(0, 0, 0, ${glow/20});`;
        } else {
            const theme = colorType === 'amber'
            ? { main: '#ffb000', bg: 'rgba(255, 176, 0, 0.12)' }
            : { main: '#00ff00', bg: 'rgba(0, 255, 0, 0.12)' };
            style += `color: ${theme.main} !important; background-color: ${theme.bg}; border: 1px solid ${theme.main}; text-shadow: 0 0 ${glow.toFixed(0)}px ${theme.main};`;
        }
        this._clockLabel.set_style(style);
    }
}
