import Gio from 'gi://Gio';
import GLib from 'gi://GLib';


export function searchApps(text, limit = 50) {
    if (!text || text.trim() === '') return [];
    let query = text.trim().toLowerCase();

    let allApps = Gio.AppInfo.get_all();
    let matches = [];

    allApps.forEach(app => {
        let name = app.get_name();
        let id = app.get_id();

        if (!name) return;
        name = name.toLowerCase();
        id = id ? id.toLowerCase() : '';

        let isSetting = id.includes('gnome-control-center') ||
                        id.includes('panel') ||
                        id.includes('org.gnome.settings');

        if (!app.should_show() && !isSetting) return;

        if (name.includes(query) || id.includes(query)) {
            matches.push({
                type: 'app',
                name: app.get_name(),
                description: app.get_description(),
                id: app.get_id(),
                icon: app.get_icon(),
                appInfo: app,
                isSetting: isSetting
            });
        }
    });

    return matches.sort((a, b) => {
        let nameA = a.name.toLowerCase();
        let nameB = b.name.toLowerCase();
        let startA = nameA.startsWith(query);
        let startB = nameB.startsWith(query);

        if (startA && !startB) return -1;
        if (startB && !startA) return 1;
        return nameA.localeCompare(nameB);
    }).slice(0, limit);
}

let _searchCancellable = null;

export function searchFiles(text, callback) {
    if (!text || !text.startsWith('.')) {
        callback([]);
        return;
    }

    let query = text.substring(1).trim().toLowerCase();
    if (query.length < 2) {
        callback([]);
        return;
    }

    if (_searchCancellable) {
        _searchCancellable.cancel();
        _searchCancellable = null;
    }

    const cancellable = new Gio.Cancellable();
    _searchCancellable = cancellable;

    const homePath = GLib.get_home_dir();
    const homeDir  = Gio.File.new_for_path(homePath);
    const results  = [];
    let   pending  = 0;
    let   finished = false;

    function done() {
        if (finished) return;
        if (_searchCancellable === cancellable) _searchCancellable = null;
        finished = true;
        callback(results.slice(0, 50));
    }

    function scanDir(dir, depth) {
        if (depth > 3 || cancellable.is_cancelled()) return;

        pending++;
        dir.enumerate_children_async(
            'standard::name,standard::icon,standard::type',
            Gio.FileQueryInfoFlags.NONE,
            GLib.PRIORITY_DEFAULT_IDLE,
            cancellable,
            (d, res) => {
                let enumerator;
                try {
                    enumerator = d.enumerate_children_finish(res);
                } catch (_) {
                    if (--pending === 0) done();
                    return;
                }
                readBatch(enumerator, dir, depth);
            }
        );
    }

    function readBatch(enumerator, parentDir, depth) {
        if (cancellable.is_cancelled()) {
            if (--pending === 0) done();
            return;
        }

        enumerator.next_files_async(
            20,
            GLib.PRIORITY_DEFAULT_IDLE,
            cancellable,
            (e, res) => {
                let infos;
                try {
                    infos = e.next_files_finish(res);
                } catch (_) {
                    if (--pending === 0) done();
                    return;
                }

                if (infos.length === 0) {
                    enumerator.close_async(GLib.PRIORITY_DEFAULT, null, null);
                    if (--pending === 0) done();
                    return;
                }

                for (let info of infos) {
                    let name = info.get_name();
                    if (name.startsWith('.')) continue;

                    let child    = parentDir.get_child(name);
                    let fileType = info.get_file_type();

                    if (name.toLowerCase().includes(query)) {
                        results.push({
                            type: 'file',
                            name,
                            description: child.get_path().replace(homePath, '~'),
                            icon: info.get_icon() ?? new Gio.ThemedIcon({ name: 'text-x-generic' }),
                            file: child,
                        });
                    }

                    if (fileType === Gio.FileType.DIRECTORY && depth < 3)
                        scanDir(child, depth + 1);
                }

                readBatch(enumerator, parentDir, depth);
            }
        );
    }

    scanDir(homeDir, 0);
}