/**
 * @typedef {import('../../../shared/core/context/jobs.js').Jobs.Job} Job
 */

import St from 'gi://St';
import Clutter from 'gi://Clutter';
import Context from '../../core/context.js';
import { Component } from './component.js';
import { Animation, AnimationDuration } from './animation.js';
import { Event, Delay } from '../../../shared/enums/general.js';

const FADE_EFFECT_NAME = 'fade';
const SCROLL_SPEED_MAX = AnimationDuration.Slower;
const SCROLL_SPEED_MIN = AnimationDuration.Faster;

/** @type {{[prop: string]: *}} */
const DefaultProps = {
    clip_to_allocation: true,
    reactive: false,
    hscrollbar_policy: St.PolicyType.EXTERNAL,
    vscrollbar_policy: St.PolicyType.NEVER
};

/**
 * Note: This component uses `St.ScrollView` behind the scenes, providing a scrollable layout.
 *
 * @augments Component<St.BoxLayout>
 */
export class ScrollView extends Component {

    /** @type {St.BoxLayout?} */
    #actor = null;

    /** @type {St.ScrollView?} */
    #container = null;

    /** @type {St.Adjustment?} */
    #scroll = null;

    /** @type {number} */
    #scrollPosition = 0;

    /** @type {number} */
    #scrollLimit = -1;

    /** @type {Job?} */
    #handleScrollJob = null;

    /**
     * @override
     * @type {St.BoxLayout}
     */
    get actor() {
        if (!this.#actor) throw new Error(`${this.constructor.name} is invalid.`);
        return this.#actor;
    }

    /** @type {St.ScrollView} */
    get container() {
        if (!this.#container) throw new Error(`${this.constructor.name} is invalid.`);
        return this.#container;
    }

    /** @type {number} 0...`scrollSize` - `pageSize` */
    get scrollPosition() {
        const scrollValue = this.#scroll?.value ?? 0;
        return Math.max(scrollValue, this.#scrollPosition);
    }

    /** @type {number} */
    get pageSize() {
        return this.#scroll?.pageSize ?? 0;
    }

    /** @type {number} */
    get scrollSize() {
        return this.#scroll?.upper ?? 0;
    }

    /** @param {number} value -1...0...`scrollSize` - `pageSize` */
    set scrollLimit(value) {
        if (typeof value !== 'number') return;
        this.#scrollLimit = value;
    }

    /**
     * @param {string?} [name]
     */
    constructor(name = null) {
        const container = new St.ScrollView(DefaultProps);
        // @ts-ignore
        super(container);
        this.#actor = new St.BoxLayout({ name });
        this.#container = container;
        this.#scroll = container.hadjustment;
        this.#handleScrollJob = Context.jobs.new(this.#scroll, Delay.Debounce);
        this.#scroll.connect(Event.Changed, () =>
            this.#handleScrollJob?.reset().enqueue(() =>
            this.#handleScrollSize()));
        container.add_child(this.#actor);
        this.connect(Event.Destroy, () => this.#destroy());
        if (typeof name !== 'string') return;
        container.set_name(`${name}-Container`);
    }

    /**
     * @param {Clutter.Actor|Component<St.Widget>} actor
     * @param {boolean} [deceleration]
     * @returns {Promise<boolean>?}
     */
    scrollToActor(actor, deceleration = false) {
        if (!this.#container || !this.#scroll || !this.hasAllocation) return null;
        if (actor instanceof Component && actor.isValid) {
            actor = actor.actor;
        }
        if (actor instanceof Clutter.Actor === false) return null;
        let { value, pageSize, upper } = this.#scroll;
        if (pageSize >= ~~upper) return null;
        const fadeEffect = this.#container.get_effect(FADE_EFFECT_NAME);
        const allocation = actor.get_allocation_box();
        const { x1, x2 } = allocation;
        const offset = fadeEffect instanceof St.ScrollViewFade ?
                       fadeEffect.fade_margins.left : x2 - x1;
        if (x1 < value + offset) {
            value = Math.max(0, x1 - offset);
        } else if (x2 > value + pageSize - offset) {
            value = Math.min(upper - pageSize, x2 + offset - pageSize);
        }
        return this.scrollToPosition(~~value, deceleration);
    }

    /**
     * @param {number} value 0...`scrollSize` - `pageSize`
     * @param {boolean} [deceleration]
     * @returns {Promise<boolean>?}
     */
    scrollToPosition(value = 0, deceleration = false) {
        if (!this.#scroll || !this.hasAllocation ||
            this.#scrollPosition === value ||
            (this.#scroll.value === value && this.#scrollPosition <= value) ||
            (this.#scrollLimit !== -1 && value > this.#scrollLimit)) return null;
        const { pageSize, upper } = this.#scroll;
        const maxValue = upper - pageSize;
        const valueOffset = Math.abs(this.#scroll.value - value);
        const targetSpeed = deceleration ? SCROLL_SPEED_MAX : SCROLL_SPEED_MIN;
        const currentSpeed = deceleration ? SCROLL_SPEED_MIN + valueOffset : SCROLL_SPEED_MAX - valueOffset;
        const speed = deceleration ? Math.min(targetSpeed, currentSpeed) :
                      value >= maxValue || !value ? targetSpeed : Math.max(targetSpeed, currentSpeed);
        const mode = Clutter.AnimationMode.EASE_OUT_QUAD;
        this.#scrollPosition = value;
        return Animation(this.#scroll, speed, { value, mode });
    }

    #destroy() {
        this.#handleScrollJob?.destroy();
        this.#handleScrollJob = null;
        this.#container = null;
        this.#actor = null;
        this.#scroll = null;
    }

    /**
     * Note: Calling `update_fade_effect` to remove fade effect from the scroll view.
     *       It's not implemented in native code for some reason.
     */
    #handleScrollSize() {
        if (!this.#container || !this.#scroll || !this.isValid) return;
        const className = `h${FADE_EFFECT_NAME}`;
        const { pageSize, upper } = this.#scroll;
        if (pageSize < ~~upper) return this.#container.add_style_class_name(className);
        if (!this.#container.has_style_class_name(className)) return;
        this.#container.remove_style_class_name(className);
        this.#container.update_fade_effect(new Clutter.Margin());
    }

}
